package app.pivo.android.prosdk;

import android.app.ActivityManager;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattService;
import android.content.Context;
import android.os.Handler;
import android.os.ParcelUuid;
import android.text.TextUtils;
import android.util.Log;
import android.widget.Toast;

import com.polidea.rxandroidble.NotificationSetupMode;
import com.polidea.rxandroidble.RxBleClient;
import com.polidea.rxandroidble.RxBleConnection;
import com.polidea.rxandroidble.RxBleDevice;
import com.polidea.rxandroidble.RxBleDeviceServices;
import com.polidea.rxandroidble.internal.RxBleLog;
import com.polidea.rxandroidble.scan.ScanFilter;
import com.polidea.rxandroidble.scan.ScanSettings;
import com.polidea.rxandroidble.utils.ConnectionSharingAdapter;

import java.util.Arrays;
import java.util.List;
import java.util.UUID;

import app.pivo.android.prosdk.events.*;
import app.pivo.android.prosdk.exceptions.*;
import app.pivo.android.prosdk.util.Degree;
import app.pivo.android.prosdk.util.Direction;
import app.pivo.android.prosdk.util.PivoDevice;
import app.pivo.android.prosdk.util.Sound;
import app.pivo.android.prosdk.util.Version;
import rx.Observable;
import rx.Subscription;
import rx.android.schedulers.AndroidSchedulers;
import rx.internal.util.SubscriptionList;
import rx.subjects.PublishSubject;

import static app.pivo.android.prosdk.events.PivoEventBus.NAME_CHANGED;


class PivoController extends IPivoController {

    private final String TAG = "PivoBasicSDK";
    private Context context;

    private RxBleDevice bleDevice;
    private RxBleClient rxBleClient;
    private final PublishSubject<Void> disconnectTriggerSubject = PublishSubject.create();
    private Observable<RxBleConnection> connectionObservable;

    private Rotator instance;

    private String macAddress;

    private boolean waitingForData = false;
    private boolean isTimerRunning = false;
    private boolean keepAlive = true;

    private IVerify verifier;

    static class Builder{
        PivoController pivoController = new PivoController();

        Builder setContext(Context context){
            pivoController.context = context;
            pivoController.verifier = new Verifier(context);

            RxBleClient.setLogLevel(RxBleLog.DEBUG);
            pivoController.setBleClient(pivoController.getRxBleClient());
            return this;
        }

        PivoController build(){
            return pivoController;
        }
    }

    private void setBleClient(RxBleClient client){
        rxBleClient = client;
    }

    private void setBleDevice(RxBleDevice bleDevice) {
        this.bleDevice = bleDevice;
    }

    private void setMacAddress(String macAddress) {
        this.macAddress = macAddress;
        setBleDevice(getRxBleClient().getBleDevice(macAddress));
    }

    private PublishSubject<Void> getDisconnectTriggerSubject() {
        return disconnectTriggerSubject;
    }

    private RxBleClient getRxBleClient() {
        if (rxBleClient == null) {
            rxBleClient = RxBleClient.create(context);
        }

        return rxBleClient;
    }

    private RxBleDevice getBleDevice() {
        if (TextUtils.isEmpty(macAddress)) return null;

        setBleDevice(getRxBleClient().getBleDevice(macAddress));
        return bleDevice;
    }

    private void connect() {
        if (getBleDevice() != null) {
            connectionObservable = prepareConnectionObservable();
            subscriptionList.add(
                    connectionObservable.subscribe(this::onConnectionReceived, this::onConnectionFailure));
        } else {
            Log.d(TAG,  "NULL");
        }
    }

    private Observable<RxBleConnection> prepareConnectionObservable() {
        return bleDevice
                .establishConnection(false)
                .takeUntil(disconnectTriggerSubject)
                .doOnUnsubscribe(this::updateUI)
                .compose(new ConnectionSharingAdapter());
    }

    private void onConnectionFailure(Throwable throwable) {
        Log.d(TAG, "onConnectionFailure");
        postConnectionFailure();
    }

    private void onConnectionReceived(RxBleConnection connection) {
        Subscription serviceSubscription = connection.discoverServices()
                .observeOn(AndroidSchedulers.mainThread())
                .doOnUnsubscribe(this::updateUI)
                .subscribe(this::swapScanResult, this::onConnectionFailure);
        subscriptionList.add(serviceSubscription);
    }

    private Boolean bothChannelsSetup = false;
    private void onNotificationChannelSetupSuccess(Observable<byte[]> notificationObservable) {
        Log.d(TAG, "Notification channel created successfully");

        Subscription notificationSub = notificationObservable
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(this::onNotificationReceived, this::onNotificationSetupFailure);

        subscriptionList.add(notificationSub);

        if (bothChannelsSetup)
            postConnectionReceived();

        bothChannelsSetup = true;
    }


    private void onNotificationReceived(byte[] bytes) {
        if (bytes.length < 6){// TODO: 2020/11/26 check byte lenght for new Pivo devices
            printData(bytes, " ERROR: Bytes less than 6:");
            return;
        }

        // Authentication Notification
        if (bytes[2] == 0x6F) {
            if (bytes[3] == 0x01) {// Pivo HW to App with answer
                int inquiry = (bytes[6]&0xff)<<24 | (bytes[7]&0xff)<<16 | (bytes[8]&0xff)<<8 | (bytes[9]&0xff);
                int answer = (bytes[10]&0xff)<<24 | (bytes[11]&0xff)<<16 | (bytes[12]&0xff)<<8 | (bytes[13]&0xff);

                byte[] sendBytes = instance.authenticatePivo((byte)0x02, answer, inquiry);

                if (sendBytes == null) { // Error
                    return;
                }
                //printData(sendBytes, "3rd Handshake Stream : ");
                write(sendBytes);

                if (sendBytes[5] == 0) {
                    // Status 0: Pivo HW Authentication Fail!
                    postNotifyPivoAuthResult(0);

                    // Disconnect the connection
                    if (isPivoConnected()){
                        disconnect();
                    }
                }
            } else if (bytes[3] == 0x03) {	// Verification notificatoin from Pivo HW
                if (bytes[5] == 1) {// Check authentication result from Pivo
                    // Status 2: Mutual Authentication Sucess
                    postNotifyPivoAuthResult(2);
                } else {
                    // Status 1: Pivo HW rejected this app
                    postNotifyPivoAuthResult(1);
                }
            }
        }else if (!(instance.getBatteryLevel(bytes) == -1)) {
            postBatteryLevelChanged(instance.getBatteryLevel(bytes));
        }else if (instance.verifyVersion(bytes)!=null){//check Pivo version handler, after receive version, trigger "Connection_Completed" event
            postVersionUpdate(instance.getVersion());
        }else if (bytes[2] == 5){// Remote Controller events from a device to App
            postRemoteControllerEvent(instance.getRemoteControllerButtonEvent(bytes));
        }else if (bytes[2] == (byte)0x1B){
            postMacAddress(bytes);
        }else if (bytes[2] == (byte)0x10 || bytes[2] == (byte)0x06){
            // TODO: 2020/11/26 New Speed is set
        }else if (bytes[2] == (byte)0xFA && bytes[5] == (byte)0x21){
            // if Notification receiving is enabled, it sends rotated to 1 to right direction
        }else if(bytes[2] == (byte)0xFA && bytes[5] == (byte)0x23){
            // if Notification receiving is enabled, it sends rotated to 1 to left direction
        }else if (bytes[2] == (byte)0X7F && bytes[4] == (byte)0x01){// if the name is changed
            PivoEventBus.publish(NAME_CHANGED, new PivoEvent.NameChanged(getName(bytes)));
        } else if (bytes[2] == (byte)0x51 && bytes[4] == (byte)0x01){
            // if pairing started
        }else if (bytes[2] == (byte)0x52 && bytes[4] == (byte)0x01){
            //pair cancelled
        }else if (bytes[2] == (byte)0x55 && bytes[4] == (byte)0x01 && bytes[5] == (byte)0x01){
            //pair succeed
        }else if (bytes[2] == (byte)0x53 && bytes[4] == (byte)0x01 && bytes[5] == (byte)0x00){
            //pair failed
        }
//        else if (bytes[2] == (byte)0x00 && bytes[4] == (byte)0x01 && bytes[5] == (byte)0x00){
//            return 8;
//        }
        else if (bytes[2] == (byte)0x54 && bytes[4] == (byte)0x01 && bytes[5] == (byte)0x01){
            // pressing power button 3 times quickly
        } else if (bytes[2] == (byte)0x54 && bytes[4] == (byte)0x01 && bytes[5] == (byte)0x00){
            // Pressing power button once within 3 s in pairing mode
        }else if (bytes[2] == (byte)0x55 && bytes[4] == (byte)0x01 && bytes[5] == (byte)0x00){
            // going from pairing
        }else{
            if (!waitingForData) return;
            waitingForData = false;
            postNotificationReceived();
        }
    }

    private String getName(byte[] bytes){
        StringBuilder name = new StringBuilder();
        //0X05	0XAF	0X7F	LEN	0X00	CHAR0	CHAR1	CHAR2	CHAR3	CHAR4	CHAR5	CHAR6
        //Command To App: (AA 55 81 00) 05 AF 7F 03 01 67 67 67
        if (bytes[0] == (byte)0x05 && bytes[2] == (byte)0x7F && bytes.length>=5){
            for (int i = 5; i<bytes.length;i++){
                name.append((char) bytes[i]);
            }
        }else if (bytes[0] == (byte)0xAA && bytes[1] == (byte)0x55 && bytes.length>=10){
            for (int i = 9; i<bytes.length;i++){
                name.append((char) bytes[i]);
            }
        }
        return name.toString();
    }

    private void onNotificationSetupFailure(Throwable throwable) {
        Log.d(TAG,  "Notifications error: " + throwable);
    }

    private void onWriteSuccess(byte[] bytes) {
        Log.d(TAG, "Write success");

        waitingForData = true;
    }

    private void onWriteFailure(Throwable throwable) {
        Log.e(TAG, "Write failed");

        postConnectionFailure();
    }

    private void swapScanResult(RxBleDeviceServices services) {
        BluetoothGattService mBTService = null;

        for (BluetoothGattService service : services.getBluetoothGattServices()) {
            instance = Rotator.getInstance(service.getUuid(), bleDevice.getName(), bleDevice.getMacAddress());
            if (instance != null) {
                mBTService = service;
                break;
            }
        }

        if (instance == null) {
            postConnectionFailure();
            return;
        }

        if (mBTService == null || mBTService.getUuid() == null) {
            postConnectionFailure();
            return;
        }

        for (BluetoothGattCharacteristic characteristic : mBTService.getCharacteristics()) {
//            Log.e(TAG,"------------ Characteristic: " + characteristic.getUuid());
//            Log.e(TAG, "Read: " + isCharacteristicReadable(characteristic));
//            Log.e(TAG, "Notify: " + isCharacteristicNotifiable(characteristic));
//            Log.e(TAG, "Write: " + isCharacteristicWriteable(characteristic));
//            Log.e(TAG, "getDescriptors: " + characteristic.getDescriptors().size());
            writeNotificationCharacteristic(characteristic);
        }

        //postConnectionReceived();
    }

    private final Handler handler = new Handler();
    private final Runnable runnable = this::afficher;

    private void afficher() {
        Log.d(TAG, "Timer »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»» afficher  »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»");
        if (isTimerRunning && isAppOnForeground()) {
            requestBatteryLevel();
        }
        handler.postDelayed(runnable, 20000);
    }

    private boolean isAppOnForeground() {
        ActivityManager activityManager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
        List<ActivityManager.RunningAppProcessInfo> appProcesses = activityManager.getRunningAppProcesses();
        if (appProcesses == null) {
            return false;
        }
        final String packageName = context.getPackageName();
        for (ActivityManager.RunningAppProcessInfo appProcess : appProcesses) {
            if (appProcess.importance == ActivityManager.RunningAppProcessInfo.IMPORTANCE_FOREGROUND && appProcess.processName.equals(packageName)) {
                return true;
            }
        }
        return false;
    }

    private void startTimer() {
        if (isTimerRunning) return;
        Log.d(TAG, "Timer »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»» startTimer  »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»");
        isTimerRunning = true;
        runnable.run();
    }

    private void stopTimer() {
        Log.d(TAG, "Timer »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»» stopTimer  »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»");
        handler.removeCallbacks(runnable);
        isTimerRunning = false;
    }

    private String printData(byte[] data, String text) {
        final StringBuilder stringBuilder = new StringBuilder(data.length);
        for (byte byteChar : data)
            stringBuilder.append(String.format("%02X ", byteChar));
        Log.d(TAG, text + stringBuilder);
        return stringBuilder.toString();
    }

    private void write(byte[] data) {
        if (isPivoConnected() && connectionObservable != null) {
            Subscription subscription = connectionObservable
                    .flatMap(rxBleConnection -> rxBleConnection.writeCharacteristic(instance.getRxUUID(), data))
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(this::onWriteSuccess, this::onWriteFailure);
            subscriptionList.add(subscription);
        } else {
            postConnectionFailure();
        }
    }

    private void writeNotificationCharacteristic(BluetoothGattCharacteristic chr) {
        if (isPivoConnected()) {

            Subscription subscription = connectionObservable
                    .flatMap(rxBleConnection -> rxBleConnection.setupNotification(chr.getUuid(), NotificationSetupMode.COMPAT))
//                    .flatMap(notificationObservable -> notificationObservable)
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(this::onNotificationChannelSetupSuccess, this::onNotificationSetupFailure);
            subscriptionList.add(subscription);
        }
    }

    void authenticatePivo(){
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        byte[] arrAuthInit = instance.authenticatePivo((byte)0, 0, 0);
        write(arrAuthInit);
    }

    private void checkConnection(){
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        Log.d(TAG,"checkConnection");
        write(instance.checkConnection());
    }

    private void getMacAddressFromPivo(){
        if (instance == null){
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        Log.d(TAG, "getMacAddress");
        write(instance.getMacAddress());
    }

    private void setTimeoutToTurnOff(int sec){
        if (instance == null){
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        Log.d(TAG, "setTimeOut");
        write(instance.getTimeoutByteData(sec));
    }

    @Override
    void getLicenseContent(String content) throws LicenseDateException, PackageNameException, InvalidLicenseException, SdkPlanTypeException {
        if (verifier!=null){
            verifier.readLicense(content);
        }
    }

    @Override
    boolean isVerified(){
        if (verifier!=null){
            return verifier.isValid();
        }else {
            return false;
        }
    }

    @Override
    void turnLeft(int angle) {
        rotate(angle, Direction.LEFT);
    }

    @Override
    void turnLeft(int angle, int speed) {
        rotate(angle, Direction.LEFT, speed);
    }

    @Override
    void turnRight(int angle) {
        rotate(angle, Direction.RIGHT);
    }

    @Override
    void turnRight(int angle, int speed) {
        rotate(angle, Direction.RIGHT, speed);
    }

    @Override
    void turnRightContinuously() {
        rotate(-1, Direction.RIGHT);
    }

    @Override
    void turnRightContinuously(int speed) {
        rotate(-1, Direction.RIGHT, speed);
    }

    @Override
    void turnLeftContinuously() {
        rotate(-1, Direction.LEFT);
    }

    @Override
    void turnLeftContinuously(int speed) {
        rotate(-1, Direction.LEFT, speed);
    }

    @Override
    void stop() {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }

        write(instance.getStopBytes());
        waitingForData = false;
    }

    @Override
    void changeName(String name) {
        if (instance !=null){
            if (isPivoConnected() && connectionObservable!=null){
                write(instance.changeRotatorName(name));
            }
        }
    }

    @Override
    void changeSpeed(int speed) {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        if (instance.isSpeedSupported(speed)){
            write(instance.getSpeedData(speed));
        }else {
            Log.e(TAG, "The given speed is not supported, please call to get supported speeds from getSupportedSpeedsInSecondsPerRound()");
        }
    }

    @Override
    boolean isPivoConnected() {
        return bleDevice != null && bleDevice.getConnectionState() == RxBleConnection.RxBleConnectionState.CONNECTED;
    }

    @Override
    Version getVersion() {
        if (instance == null) {
            Log.e( TAG,"There is no connected rotator");
            postConnectionFailure();
            return null;
        }

        return instance.getVersion();
    }

    @Override
    String getPivoVersion() {
        if (instance == null) {
            Log.e( TAG,"There is no connected rotator");
            postConnectionFailure();
            return "UNKNOWN";
        }

        return "VERSION_"+instance.getVersion().getVersion();
    }

    @Override
    void turnOnNotification() {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        write(instance.turnOnNotifier());
    }

    @Override
    void turnOffNotification() {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        write(instance.turnOffNotifier());
    }

    @Override
    void changeMode() {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        // TODO: 2020/04/01 add mode
//        rotator.setMode(mode);
//        write(rotator.getModeBytes());
    }

    @Override
    void startPairingMode() {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        write(instance.startPairing());
    }

    @Override
    void cancelPairingMode() {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        write(instance.cancelPairing());
    }

    @Override
    void makeBeepSound(Sound sound) {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }

        instance.setSound(sound);
        write(instance.getSoundBytes());
    }

    @Override
    void getMacAddress() {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        if (getVersion().getVersion()<5){
            Log.e(TAG, "There is no connected rotator");
        }
        write(instance.getMacAddress());
    }

    @Override
    void keepAlive(boolean alive) {
        this.keepAlive = alive;
        if (keepAlive){
            stopTimer();
            startTimer();
        }else {
            stopTimer();
        }
    }

    @Override
    void changeMinRotationDegree(Degree degree) {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }

        instance.setDegree(degree);
    }

    @Override
    void scan() {
        if (isScanning())return;
        if (!isVerified()) {
            Log.e(TAG, "******************* Exception: Invalid license is provided. *********************");
            return;
        }
        //disconnect first, if the phone's connected to the device
        while (getDisconnectTriggerSubject().hasObservers()) {
            getDisconnectTriggerSubject().onNext(null);
        }

        //clear results if they're scanned earlier
        unsubscribe();

        scanSubscription = rxBleClient.scanBleDevices(
                new ScanSettings.Builder()
                        .setScanMode(ScanSettings.SCAN_MODE_LOW_LATENCY)
                        .setCallbackType(ScanSettings.CALLBACK_TYPE_ALL_MATCHES)
                        .build(),
                new ScanFilter.Builder()
                        .setServiceUuid(new ParcelUuid(PivoUUID.serviceUUID))
                        .build()
        )
                .observeOn(AndroidSchedulers.mainThread())
                .doOnUnsubscribe(this::unsubscribe)
                .subscribe(this::addDevice, throwable -> {
                    // TODO: 2020/04/02 implement not found case
                    Log.e(TAG, "NotFound");
                });
    }

    @Override
    void connectTo(PivoDevice pivoDevice) {
        if (!isVerified()) {
            Log.e(TAG, "******************* Exception: Invalid license is provided. *********************");
            return;
        }
        setMacAddress(pivoDevice.getMacAddress());

        bothChannelsSetup = false;
        disconnect();
        connect();
    }

    @Override
    void stopScan() {
        disconnect();
    }

    @Override
    void disconnect() {
        while (getDisconnectTriggerSubject().hasObservers()) {
            getDisconnectTriggerSubject().onNext(null);
        }
        unsubscribe();
    }

    @Override
    int[] getSupportedSpeedsInSecondsPerRound() {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return new int[0];
        }
        return instance.getSupportedSpeedList();
    }

    @Override
    int[] getSupportedSpeedsByRemoteInSecondsPerRound() {
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return new int[0];
        }
        return instance.getRemoteSupportedSpeedList();
    }

    @Override
    void requestBatteryLevel() {
        if (instance != null) {
            Log.d(TAG, "Battery level request is sent.");
            if (isPivoConnected() && connectionObservable != null) {
                write(instance.battery_Level());
            } else {
                stopTimer();
            }
        }
    }

    private Subscription scanSubscription;
    private SubscriptionList subscriptionList = new SubscriptionList();

    private boolean isScanning(){
        return scanSubscription!=null;
    }

    private void addDevice(com.polidea.rxandroidble.scan.ScanResult result){
        if (result.getBleDevice()!=null){
            PivoEventBus.publish(PivoEventBus.SCAN_DEVICE, new PivoEvent.Scanning(new PivoDevice(result.getBleDevice().getName(), result.getBleDevice().getMacAddress())));
        }
    }

    //clear the results and unsubscribe
    private void unsubscribe() {
        unSubscribeScanning();
        /* Remove all the subscription that we have */
        subscriptionList.unsubscribe();
        /* Refresh the list, because it is one time use only */
        subscriptionList = new SubscriptionList();
    }

    private void unSubscribeScanning(){
        if (scanSubscription!=null){
            scanSubscription.unsubscribe();
            scanSubscription = null;
        }
    }

    private void rotate(int angle, Direction direction, int speed){
        if (instance == null) {
            Log.e(TAG, "There is no connected rotator");
            postConnectionFailure();
            return;
        }
        if (!instance.isSupportedAngle(angle)){
            Log.e(TAG, "The given rotation angle is unsupported, please give positive numbers");
            return;
        }

        if (!instance.isSpeedSupported(speed)){
            Log.e(TAG, "The given speed is not supported, please call to get supported speeds from getSupportedSpeedsInSecondsPerRound()");
            return;
        }
        write(instance.getRotatingData(angle, direction, speed));
    }

    private void rotate(int angle, Direction direction) {
        if (instance == null) {
            Log.e( TAG,"There is no connected rotator");
            postConnectionFailure();
            return;
        }
        if (!instance.isSupportedAngle(angle)){
            Log.e(TAG, "The given rotation angle is unsupported, please give positive numbers");
            return;
        }
        write(instance.getRotationBytes(angle, direction));
    }

    private void updateUI() {
        //Log.d(TAG, "updateUI");
    }

    private boolean isCharacteristicNotifiable(BluetoothGattCharacteristic characteristic) {
        return (characteristic.getProperties() & BluetoothGattCharacteristic.PROPERTY_NOTIFY) != 0;
    }

    private boolean isCharacteristicReadable(BluetoothGattCharacteristic characteristic) {
        return ((characteristic.getProperties() & BluetoothGattCharacteristic.PROPERTY_READ) != 0);
    }

    private boolean isCharacteristicWriteable(BluetoothGattCharacteristic characteristic) {
        return (characteristic.getProperties() & (BluetoothGattCharacteristic.PROPERTY_WRITE
                | BluetoothGattCharacteristic.PROPERTY_WRITE_NO_RESPONSE)) != 0;
    }

    /// Posters
    private void postConnectionFailure() {
        stopTimer();
        Log.e(TAG, "postConnectionFailure");
        PivoEventBus.publish(PivoEventBus.CONNECTION_FAILURE, new PivoEvent.ConnectionFailure());
    }

    private void postVersionUpdate(Version version){
        Log.e(TAG, "postVersionUpdate:"+version.getVersion());
        if (version.isAboveV1()){
            authenticatePivo();
        }else {
            startTimer();
            PivoEventBus.publish(PivoEventBus.CONNECTION_COMPLETED, new PivoEvent.ConnectionComplete());
        }
    }

    private void postMacAddress(byte[] bytes){
        if (bytes.length>=8){
            PivoEventBus.publish(PivoEventBus.MAC_ADDRESS, new PivoEvent.MacAddress(printData(Arrays.copyOfRange(bytes, 3, 9), "MacAddress: ")));
        }
    }

    private void postNotifyPivoAuthResult(int status) {
        if (status == 2){
            startTimer();
            PivoEventBus.publish(PivoEventBus.CONNECTION_COMPLETED, new PivoEvent.ConnectionComplete());
        }else {
            disconnect();
        }
    }

    private void postConnectionReceived() {
        Log.d(TAG, "postConnectionReceived");
        handler.postDelayed(() -> {
            checkConnection();
            // unsubscribe scanning
            unSubscribeScanning();
        }, 2000);
    }

    private void postNotificationReceived() {
        PivoEventBus.publish(PivoEventBus.PIVO_NOTIFICATION, new PivoEvent());
    }

    /**
     * This callback is triggered if there's battery changes in Pivo
     */
    private void postBatteryLevelChanged(int level) {
        PivoEventBus.publish(PivoEventBus.PIVO_NOTIFICATION, new PivoEvent.BatteryChanged(level));
    }

    /**
     * RemoteControllerClick callback handler
     */
    private void postRemoteControllerEvent(RemoteButtonEvent event) {
        switch (event){
            case MODE:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCMode(event.state));
                break;
            case CAMERA:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCCamera(event.state));
                break;
            case STOP:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCStop(event.state));
                break;
            case LEFT:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCLeft(event.state));
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCLeftPressed());
                break;
            case RIGHT:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCRight(event.state));
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCRightPressed());
                break;
            case LEFT_CONTINUOUS:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCLeftContinuous(event.state));
                break;
            case RIGHT_CONTINUOUS:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCRightContinuous(event.state));
                break;
            case SPEED_CHANGE:
                int speedValue = getSpeedValue(event.speed);
                if (instance.getVersion().isV0()){
                    PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCSpeed(speedValue, event.state));
                }else {
                    PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCSpeed(speedValue, event.direction, event.state));
                    if (event.direction == 1){
                        PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCSpeedUp(speedValue, event.state));
                    }else if (event.direction == 0){
                        PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCSpeedDown(speedValue, event.state));
                    }
                }
                break;
        }
    }

    private int getSpeedValue(Rotator.Speed speed){
        switch (speed){
            case SPEED_1:return 10;
            case SPEED_2:return 20;
            case SPEED_3:return 30;
            case SPEED_4:return 50;
            case SPEED_5:return 60;
            case SPEED_6:return 120;
            case SPEED_7:return 300;
            case SPEED_8:return 600;
            case SPEED_9:return 1200;
            case SPEED_10:return 1800;
            case SPEED_11:return 3600;
            case SPEED_12:return 7200;
            case SPEED_13:return 10800;
            case SPEED_14:return 21600;
            case SPEED_15:return 43200;
            case SPEED_16:return 64800;
            case SPEED_17:return 86400;
            default:return 10;
        }
    }
}
