package app.pivo.android.prosdk;

import android.content.Context;
import android.graphics.PointF;
import android.graphics.Rect;
import android.media.Image;
import android.text.TextUtils;
import android.util.Log;

import app.pivo.android.prosdk.exceptions.*;
import app.pivo.android.prosdk.tracking.FrameMetadata;
import app.pivo.android.prosdk.util.IPivoSdk;
import app.pivo.android.prosdk.util.ITrackingListener;
import app.pivo.android.prosdk.util.PivoDevice;
import app.pivo.android.prosdk.util.Version;

public class PivoProSdk implements IPivoSdk {

    private final String TAG = this.getClass().getSimpleName();
    static PivoProSdk instance;
    private IPivoController c;
    private IFrameProcessingTask f;


    private FrameMetadata frameMetadata;

    private SensitivityMode sensitivityMode = SensitivityMode.NORMAL;


    public static PivoProSdk getInstance(){
        return instance;
    }

    public static void init(Context context){
        instance = new PivoProSdk(context);
    }

    PivoProSdk(Context context){
        // pivo controller
        c = new PivoController
                .Builder()
                .setContext(context)
                .build();

        // frame processing tasks
        f = new FrameProcessingTask
                .Builder()
                .setContext(context)
                .build();

       PivoSpeedController.getInstance().setController(c);

    }

    @Override
    public void unlockWithLicenseKey(String licenseContent) throws PackageNameException, LicenseDateException, LicenseNotProvidedException, InvalidLicenseException, SdkPlanTypeException {
        if (licenseContent == null || TextUtils.isEmpty(licenseContent)){
            throw new LicenseNotProvidedException("License is not provided.");
        }
        c.getLicenseContent(licenseContent);
    }

    @Override
    public void scan(){
        c.scan();
    }

    @Override
    public void connectTo(PivoDevice pivoDevice) {
        c.connectTo(pivoDevice);
    }

    @Override
    public void stopScan() {
        c.stopScan();
    }

    @Override
    public void setSpeedBySecondsPerRound(int speedInSecondsPerRound) {
        c.changeSpeed(speedInSecondsPerRound);
    }

    @Override
    public int[] getSupportedSpeedsInSecondsPerRound() {
        return c.getSupportedSpeedsInSecondsPerRound();
    }

    @Override
    public int[] getSupportedSpeedsByRemoteInSecondsPerRound() {
        return c.getSupportedSpeedsByRemoteInSecondsPerRound();
    }

    @Override
    public void disconnect() {
        c.disconnect();
    }

    @Override
    public void turnLeft(int angle) {
        c.turnLeft(angle);
    }

    @Override
    public void turnLeft(int angle, int speed) {
        c.turnLeft(angle, speed);
    }

    @Override
    public void turnRight(int angle) {
        c.turnRight(angle);
    }

    @Override
    public void turnRight(int angle, int speed) {
        c.turnRight(angle, speed);
    }

    @Override
    public void turnRightContinuously() {
        c.turnRightContinuously();
    }

    @Override
    public void turnRightContinuously(int speed) {
        c.turnRightContinuously(speed);
    }

    @Override
    public void turnLeftContinuously() {
        c.turnLeftContinuously();
    }

    @Override
    public void turnLeftContinuously(int speed) {
        c.turnLeftContinuously(speed);
    }

    @Override
    public void requestBatteryLevel() {
        c.requestBatteryLevel();
    }

    @Override
    public void stop() {
        c.stop();
    }

    @Override
    public void changeName(String name) {
        c.changeName(name);
    }

    @Override
    public boolean isPivoConnected() {
        return c.isPivoConnected();
    }

    @Override
    public String getPivoVersion() {
        return c.getPivoVersion();
    }

    @Override
    public Version getVersion() {
        return c.getVersion();
    }

    @Override
    public void getMacAddress() {
        c.getMacAddress();
    }

    @Override
    public String[] getSupportedTrackingModes() {
        return new String[]{"Action", "Person", "Horse"};
    }

    @Override
    public void startActionTracking(FrameMetadata metadata, Rect region, Image image,PivoSensitivity sensitivity, ITrackingListener listener) {
        if (image==null || region==null || listener == null || metadata == null || sensitivity == null){
            Log.e(TAG, "startActionTracking() ->  Some of the arguments you're sending is null.");
            return;
        }
        switch (sensitivity){
            case NONE:
                sensitivityMode = SensitivityMode.NONE;
                break;
            case NORMAL:
                sensitivityMode = SensitivityMode.NORMAL;
                break;
            case SLOW:
                sensitivityMode = SensitivityMode.SLOW;
                break;
            case FAST:
                sensitivityMode = SensitivityMode.FAST;
                break;
        }
        frameMetadata = metadata;
        this.outListener = listener;
        if (c.isVerified()){
            f.startActionTracking(metadata, region, image, mListener);
        }
    }

    @Override
    public void startActionTracking(FrameMetadata metadata, Rect region, byte[] image, PivoSensitivity sensitivity, ITrackingListener listener) {
        if (image==null || region==null || listener == null || metadata == null || sensitivity == null){
            Log.e(TAG, "startActionTracking() ->  Some of the arguments you're sending is null.");
            return;
        }
        switch (sensitivity){
            case NONE:
                sensitivityMode = SensitivityMode.NONE;
                break;
            case NORMAL:
                sensitivityMode = SensitivityMode.NORMAL;
                break;
            case SLOW:
                sensitivityMode = SensitivityMode.SLOW;
                break;
            case FAST:
                sensitivityMode = SensitivityMode.FAST;
                break;
        }
        frameMetadata = metadata;
        this.outListener = listener;
        if (c.isVerified()){
            f.startActionTracking(metadata, region, image, mListener);
        }
    }

    @Override
    public void starPersonTracking(FrameMetadata metadata, Image image,PivoSensitivity sensitivity, ITrackingListener listener) {
        if (image==null || metadata==null || listener == null|| sensitivity == null){
            Log.e(TAG, "starPersonTracking() -> Some of the arguments you're sending is null.");
            return;
        }
        switch (sensitivity){
            case NONE:
                sensitivityMode = SensitivityMode.NONE;
                break;
            case NORMAL:
                sensitivityMode = SensitivityMode.NORMAL;
                break;
            case SLOW:
                sensitivityMode = SensitivityMode.SLOW;
                break;
            case FAST:
                sensitivityMode = SensitivityMode.FAST;
                break;
        }
        frameMetadata = metadata;
        this.outListener = listener;
        if (c.isVerified()){
            f.startPersonTracking(metadata, image, mListener);
        }
    }

    @Override
    public void starPersonTracking(FrameMetadata metadata, byte[] image, PivoSensitivity sensitivity, ITrackingListener listener) {
        if (image==null || metadata==null || listener == null|| sensitivity == null){
            Log.e(TAG, "starPersonTracking() -> Some of the arguments you're sending is null.");
            return;
        }
        switch (sensitivity){
            case NONE:
                sensitivityMode = SensitivityMode.NONE;
                break;
            case NORMAL:
                sensitivityMode = SensitivityMode.NORMAL;
                break;
            case SLOW:
                sensitivityMode = SensitivityMode.SLOW;
                break;
            case FAST:
                sensitivityMode = SensitivityMode.FAST;
                break;
        }
        frameMetadata = metadata;
        this.outListener = listener;
        if (c.isVerified()){
            f.startPersonTracking(metadata, image, mListener);
        }
    }

    @Override
    public void startHorseTracking(FrameMetadata metadata, Image image,PivoSensitivity sensitivity, ITrackingListener listener) {
        if (image==null || listener == null || metadata==null || sensitivity == null){
            Log.e(TAG, "startHorseTracking() -> Some of the arguments you're sending is null.");
            return;
        }
        switch (sensitivity){
            case NONE:
                sensitivityMode = SensitivityMode.NONE;
                break;
            case NORMAL:
                sensitivityMode = SensitivityMode.NORMAL;
                break;
            case SLOW:
                sensitivityMode = SensitivityMode.SLOW;
                break;
            case FAST:
                sensitivityMode = SensitivityMode.FAST;
                break;
        }
        frameMetadata = metadata;
        this.outListener = listener;
        if (c.isVerified()){
            f.startHorseTracking(metadata, image, mListener);
        }
    }

    @Override
    public void startHorseTracking(FrameMetadata metadata, byte[] image, PivoSensitivity sensitivity, ITrackingListener listener) {
        if (image==null || listener == null || metadata==null || sensitivity == null){
            Log.e(TAG, "startHorseTracking() -> Some of the arguments you're sending is null.");
            return;
        }
        switch (sensitivity){
            case NONE:
                sensitivityMode = SensitivityMode.NONE;
                break;
            case NORMAL:
                sensitivityMode = SensitivityMode.NORMAL;
                break;
            case SLOW:
                sensitivityMode = SensitivityMode.SLOW;
                break;
            case FAST:
                sensitivityMode = SensitivityMode.FAST;
                break;
        }
        frameMetadata = metadata;
        this.outListener = listener;
        if (c.isVerified()){
            f.startHorseTracking(metadata, image, mListener);
        }
    }

    @Override
    public void stopTracking(){
        f.stopTracking();
    }

    @Override
    public void updateTrackingFrame(Image image, FrameMetadata metadata) {
        if (image==null){
            Log.e(TAG, "updateTrackingFrame()-> Image object is null");
            return;
        }
        if (c.isVerified()){
            frameMetadata = metadata;
            f.update(image, metadata);
        }
    }

    @Override
    public void updateTrackingFrame(byte[] image, FrameMetadata metadata) {
        if (image==null){
            Log.e(TAG, "updateTrackingFrame()-> Image object is null");
            return;
        }
        if (c.isVerified()){
            frameMetadata = metadata;
            f.update(image, metadata);
        }
    }

    private ITrackingListener outListener;
    private ITrackingListener mListener = new ITrackingListener() {

        @Override
        public void onTracking(int x, int y, int width, int height, int frameWidth, int frameHeight) {
            Rect rect = new Rect(x, y, x + width, y + height);
            f.setTargetToTrack(rect);

            if (f.getTracker() != null) {
                outListener.onTracking(f.getTracker().draw());
            }

            outListener.onTracking(x, y, width, height, frameWidth, frameHeight);

            PivoSpeedController.getInstance()
                    .setCameraFacing(frameMetadata.isFrontCamera())
                    .setSensitivityMode(sensitivityMode)
                    .setOrientation(frameMetadata.getRotation())
                    .setOrientationLocked(frameMetadata.isOrientationLocked())
                    .setPositionType(PositionType.MIDDLE)
                    .setPreviewSize(frameWidth,frameHeight)
                    .track(new PointF(x + width / 2.f, y + height / 2.f));
        }

        @Override
        public void onClear() {
            outListener.onClear();
        }
    };
}
