package app.pivo.android.prosdk;

import java.util.Random;
import java.util.UUID;
import app.pivo.android.prosdk.util.Degree;
import app.pivo.android.prosdk.util.Direction;
import app.pivo.android.prosdk.util.PivoType;
import app.pivo.android.prosdk.util.Sound;
import app.pivo.android.prosdk.events.*;
import app.pivo.android.prosdk.util.Version;

/**
 * Created by murodjon on 2020/03/31
 */
class Rotator {

    private byte[]stop = new byte[]{0x05, (byte) 0xAF, 0x00, 0x00, 0x00, 0x00};
    private byte[]current = new byte[]{0x05, (byte) 0xAF, 0x21, 0x00, 0x00, 0x00};

    private Version version = new Version(0, PivoType.RED);
    private int speed = 10;
    private Mode mode = Mode.NORMAL;
    private Sound sound = Sound.SWITCH_ON;
    private Degree angle_step = Degree.DEGREE_4;
    private boolean remoteIsOn = true;

    public Rotator() { }

    static Rotator getInstance(UUID uuid, String name, String macAddress) {
        Rotator rotator = null;
        if (PivoUUID.serviceUUID.equals(uuid)){
            rotator = new Rotator();
        }
        return rotator;
    }

    public Version getVersion(){
        return version;
    }

    boolean isRemoteOn(){
        return remoteIsOn;
    }

    void setRemoteOn(boolean on){
        this.remoteIsOn = on;
    }

    enum Mode {
        NORMAL,
        HYBRID_VIDEO,
        TIMELAPSE
    }

    enum Speed {
        SPEED_1(10),
        SPEED_2(20),
        SPEED_3(30),
        SPEED_4(50),
        SPEED_5(60),
        SPEED_6(120),
        SPEED_7(300),
        SPEED_8(600),
        SPEED_9(1200),
        SPEED_10(1800),
        SPEED_11(3600),
        //the following speed list is supported by V1
        SPEED_12(7200),
        SPEED_13(10800),
        SPEED_14(21600),
        SPEED_15(43200),
        SPEED_16(64800),
        SPEED_17(86400);

        Speed(int val){
            value = val;
        }
        int value;
        int getValue(){
            return value;
        }

        static Speed speed(int value){
            switch (value){
                case 1:return SPEED_1;
                case 2:return SPEED_2;
                case 3:return SPEED_3;
                case 4:return SPEED_4;
                case 5:return SPEED_5;
                case 6:return SPEED_6;
                case 7:return SPEED_7;
                case 8:return SPEED_8;
                case 9:return SPEED_9;
                case 10:return SPEED_10;
                case 11:return SPEED_11;
                case 12:return SPEED_12;
                case 13:return SPEED_13;
                case 14:return SPEED_14;
                case 15:return SPEED_15;
                case 16:return SPEED_16;
                case 17:return SPEED_17;
            }
            return SPEED_1;
        }

        static Speed speed(byte data){
            switch (data){
                case 0x11:
                    return SPEED_1;
                case 0x12:
                    return SPEED_2;
                case 0x13:
                    return SPEED_3;
                case 0x14:
                    return SPEED_4;
                case 0x15:
                    return SPEED_5;
                case 0x16:
                    return SPEED_6;
                case 0x17:
                    return SPEED_7;
                case 0x18:
                    return SPEED_8;
                case 0x19:
                    return SPEED_9;
                case 0x1A:
                    return SPEED_10;
                case 0x1B:
                    return SPEED_11;
                case 0x1C:
                    return SPEED_12;
                case 0x1D:
                    return SPEED_13;
                case 0x1E:
                    return SPEED_14;
                case 0x1F:
                    return SPEED_15;
                case 0x20:
                    return SPEED_16;
                case 0x21:
                    return SPEED_17;
            }
            return SPEED_1;
        }
    }

    public RemoteButtonEvent getRemoteControllerButtonEvent(byte[] data){
        if (data.length>=6 && data[2] == 0x05){
            if (version.isV0()){
                switch (data[5]){
                    case 0x0A:
                        return RemoteButtonEvent.MODE.update(data[3] == 0x01?1:0, -1);
                    case 0x01:
                        return RemoteButtonEvent.CAMERA.update(data[3] == 0x01?1:0, -1);
                    case 0x00:
                        return RemoteButtonEvent.STOP.update(data[3] == 0x01?1:0, -1);
                    case 0x26:
                        return RemoteButtonEvent.LEFT_CONTINUOUS.update(data[3] == 0x01?1:0, -1);
                    case 0x24:
                        return RemoteButtonEvent.RIGHT_CONTINUOUS.update(data[3] == 0x01?1:0, -1);
                    case 0x23:
                        return RemoteButtonEvent.LEFT.update(data[3] == 0x01?1:0, -1);
                    case 0x21:
                        return RemoteButtonEvent.RIGHT.update(data[3] == 0x01?1:0, -1);
                    case (byte) 0xFF:
                        return RemoteButtonEvent.OFF.update(data[3] == 0x01?1:0, -1);
                    default:
                        return RemoteButtonEvent.SPEED_CHANGE.update(data[3] == 0x01?1:0, -1, Speed.speed(data[5]));
                }
            }else if (version.isAboveV0()){
                switch (data[5]){
                    case 0x0A:
                        return RemoteButtonEvent.MODE.update(data[3] == 0x01?1:0, -1);
                    case 0x01:
                        return RemoteButtonEvent.CAMERA.update(data[3] == 0x01?1:0, -1);
                    case 0x00:
                        return RemoteButtonEvent.STOP.update(data[3] == 0x01?1:0, -1);
                    case 0x51:
                        return RemoteButtonEvent.LEFT_CONTINUOUS.update(data[3] == 0x01?1:0, -1);
                    case 0x53:
                        return RemoteButtonEvent.RIGHT_CONTINUOUS.update(data[3] == 0x01?1:0, -1);
                    case 0x54:
                        return RemoteButtonEvent.LEFT.update(data[3] == 0x01?1:0, -1);
                    case 0x56:
                        return RemoteButtonEvent.RIGHT.update(data[3] == 0x01?1:0, -1);
                    case (byte) 0xFF:
                        return RemoteButtonEvent.OFF.update(data[3] == 0x01?1:0, -1);
                    default:
                        return RemoteButtonEvent.SPEED_CHANGE.update(data[3] == 0x01?1:0, data[4]==0x01?1:0, Speed.speed(data[5]));
                }
            }
        }
        return null;
    }

    int getBatteryLevel(byte[] data) {
        if (data[2] == 0x04 && data[4] == 0x01) {
            return data[5];
        }
        return -1;
    }

    PivoEvent.RCSpeed getSpeedType(byte[] data){
        if ((this.version.isAboveV0()) && data[2] == 0x05){
            int speed = getSpeedLevel(data[5]);
            if (speed!=-1){
                return new PivoEvent.RCSpeed(speed,
                        data[4] == (byte)0x01?ParentSpeed.UP:ParentSpeed.DOWN,
                        data[3]==(byte)0x00? ParentSpeed.PRESSED:ParentSpeed.RELEASE);
            }
        }
        return null;
    }

    private int getSpeedLevel(byte data){
        switch (data){
            case (byte) 0x11:return 10;
            case (byte) 0x12:return 20;
            case (byte) 0x13:return 30;
            case (byte) 0x14:return 50;
            case (byte) 0x15:return 60;
            case (byte) 0x16:return 120;
            case (byte) 0x17:return 300;
            case (byte) 0x18:return 600;
            case (byte) 0x19:return 1200;
            case (byte) 0x1A:return 1800;
            case (byte) 0x1B:return 3600;
            case (byte) 0x1C:return 7200;
            case (byte) 0x1D:return 10800;
            case (byte) 0x1E:return 21600;
            case (byte) 0x1F:return 43200;
            case (byte) 0x20:return 64800;
            case (byte) 0x21:return 86400;
        }
        return -1;
    }

    int checkRemoteOnOff(byte[] data){
        if (version.isAboveV0()){
            if (data[2] == 0x60 && data[4] == 0x01){
                if (data[3] == (byte)0xA1){
                    remoteIsOn = true;
                    return 1;
                }else {
                    remoteIsOn = false;
                    return 0;
                }
            }
            return -1;
        }
        return -1;
    }

    Version verifyVersion(byte[] data){
        if (data[2] == 0x03 && data[4] == 0x01) {
            switch (data[3]){
                case 0:
                    version = new Version(0, PivoType.RED);
                    break;
                case 1:
                    version = new Version(1, PivoType.RED);
                    break;
                case 2:
                    version = new Version(2, PivoType.SILVER);
                    break;
                case 3:
                    version = new Version(3, PivoType.RED);
                    break;
                case 4:
                    version = new Version(4, PivoType.BLACK);
                    break;
                case 16://0x10
                    version = new Version(data[5], PivoType.RED);
                    break;
                case 32://0x20
                    version = new Version(data[5], PivoType.SILVER);
                    break;
                case 48://0x
                    version = new Version(data[5], PivoType.BLACK);
                    break;
            }
            return version;
        }else{
            return null;
        }
    }

    private int inquiryForAuth;

    // added by jw.jung, 2020.03.26.
    byte[] authenticatePivo(byte seqNum, int answerFromPivo, int inquryFromPivo) {
        if (seqNum == 0){
            Random randInquiry = new Random(System.currentTimeMillis());
            int a = randInquiry.nextInt();

//            Log.e("TTT","inquriy: " + a);

            this.current = new byte[]{0x05, (byte)0xAF, 0x6F, 0x00, 0x00, 0x00,
                    (byte)(a>>24), (byte)(a>>16), (byte)(a>>8), (byte)a, 0x00, 0x00, 0x00, 0x00};
            this.inquiryForAuth = a;
        } else if (seqNum == 2){
            PivoAuthentication authMode = new PivoAuthentication(version.getVersion());

            if (!authMode.verifyAuth(this.inquiryForAuth, answerFromPivo)) {
//                Log.e(TAG, "Error: Pivo Authenticatin Fail!");

                this.current = new byte[]{0x05, (byte)0xAF, 0x6F, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
            } else {
                int code = authMode.getAnswer(inquryFromPivo);

//                Log.e(TAG, "Inform : Pivo Authenticatin Succeed!");

                this.current = new byte[]{0x05, (byte)0xAF, 0x6F, 0x02, 0x00, 0x01,
                        0x00, 0x00, 0x00, 0x00, (byte)(code>>24), (byte)(code>>16), (byte)(code>>8), (byte)code};
            }
        } else {
//            Log.e(TAG, "Error: Invalid sequence number: " + seqNum);
            return null;
        }
        return this.current;
    }

    byte[] getTimeoutByteData(int time){
        if (version.getVersion()>=5){
            this.current = new byte[]{0x05, (byte) 0xAF, (byte) 0xA1, 0x00, 0x00, getByte(time/10)};
            return this.current;
        }else {
            return null;
        }
    }

    byte[] getMacAddress(){
        this.current = new byte[]{0x05, (byte) 0xAF, 0x1B, 0x00, 0x00, 0x00};
        return this.current;
    }

    byte[] checkConnection(){
        this.current = new byte[]{0x05, (byte) 0xAF, (byte) 0x03, 0x00, 0x00, 0x00};
        return this.current;
    }

    byte[] battery_Level() {
        this.current = new byte[]{0x05, (byte) 0xAF, 0x04, 0x00, 0x00, 0x00};
        return this.current;
    }

    byte[] switchRemote(boolean on){
        byte[] data=new byte[]{0x05, (byte)0xAF, (byte)0x60, on ? (byte) 0xA1 : (byte) 0xA0, 0x00, 0x00};
        return data;
    }

    byte[] turnOnNotifier(){
        byte[] data=new byte[]{0x05, (byte)0xAF, (byte)0x09, (byte) 0xA1, 0x00, 0x00};
        return data;
    }

    byte[] turnOffNotifier(){
        byte[]data=new byte[]{0x05, (byte)0xAF, (byte)0x09, (byte)0xA0, 0x00, 0x00};
        return data;
    }

    byte[] startPairing(){
        byte[]data=new byte[]{0x05, (byte)0xAF, (byte)0x51, (byte)0x00, 0x00, 0x00};
        return data;
    }

    byte[] cancelPairing(){
        byte[]data=new byte[]{0x05, (byte)0xAF, (byte)0x52, (byte)0x00, 0x00, 0x00};
        return data;
    }

    int getNotification(byte[] data){
        if (data[2] == (byte)0xFA && data[5] == (byte)0x21){
            return 1;
        }else if (data[2] == (byte)0xFA && data[5] == (byte)0x23){
            return -1;
        }else if (data[2] == (byte)0X7F && data[4] == (byte)0x01){
            return 2;
        }else if (data[2] == (byte)0x51 && data[4] == (byte)0x01){
            return 3;//pair started
        }else if (data[2] == (byte)0x52 && data[4] == (byte)0x01){
            return 4;//pair cancelled
        }else if (data[2] == (byte)0x55 && data[4] == (byte)0x01 && data[5] == (byte)0x01){
            return 5;//pair succeed
        }else if (data[2] == (byte)0x53 && data[4] == (byte)0x01 && data[5] == (byte)0x00){
            return 6;//pair failed
        }else if (data[2] == (byte)0x05 && data[4] == (byte)0x01 && data[5] == (byte)0xff){
            return 7;
        }else if (data[2] == (byte)0x00 && data[4] == (byte)0x01 && data[5] == (byte)0x00){
            return 8;
        }else if (data[2] == (byte)0x54 && data[4] == (byte)0x01 && data[5] == (byte)0x01){
            return 9;
        }else if (data[2] == (byte)0x54 && data[4] == (byte)0x01 && data[5] == (byte)0x00){
            return 10;
        }else if (data[2] == (byte)0x55 && data[4] == (byte)0x01 && data[5] == (byte)0x00){
            return 11;
        }else return -1;
    }

    /**
     * this function is used for changing rotator name
     * @param name is new name for rotator
     * @return an array of byte values is returned from function
     */
    byte[] changeRotatorName(String name){
        // 5 elements are needed for changing name
        int length=name.length();
        // final byte array size for changing name is 5+(length of name)
        byte[] data=new byte[length+5];
        data[0] = (byte)0x05;
        data[1] = (byte)0xAF;
        data[2] = (byte)0x7F;
        data[3] = (byte)length;
        data[4] = (byte)0x00;

        for (int i=0;i<name.length();i++){
            data[5+i]=(byte)name.charAt(i);
        }
        return data;
    }

    /**
     * This function is used to generate mode commands, such as timelapse, hybrid, normal...
     * @return an array of byte values is returned from function
     */
    private byte[] getModeBytes(){
        byte[] data=this.current.clone();

        data[2] = (byte) 0x07;
        data[3] = getModeByte();

        return data;
    }

    /**
     * Return specific byte according to mode
     */
    private byte getModeByte(){
        byte data = (byte) 0x00;

        switch (mode) {
            case NORMAL:
                data = (byte) 0xD0;
                break;
            case HYBRID_VIDEO:
                data = (byte) 0xD1;
                break;
            case TIMELAPSE:
                data = (byte) 0xD2;
                break;
        }
        return data;
    }

    /**
     * This functions generates byte array for sound
     * @return an array of byte values is returned from function
     */
    byte[] getSoundBytes(){
        byte[] data=this.current.clone();

        data[2] = (byte) 0x08;
        data[3] = getSoundByte();
        return data;
    }

    /**
     * Return specific byte according to sound type
     */
    byte getSoundByte(){
        byte data=(byte) 0x00;

        switch (sound){
            case SWITCH_ON:
                data = (byte)0xB0;
                break;
            case SWITCH_OFF:
                data = (byte)0xB1;
                break;
            case SUCCESS:
                data = (byte)0xB2;
                break;
            case DISCONNECTED:
                data = (byte)0xB3;
                break;
            case OTHER_1:
                data = (byte)0xB4;
                break;
            case OTHER_2:
                data = (byte)0xB5;
                break;
        }
        return data;
    }

    byte[] getRotatingData(int angle, Direction direction, int speed){
        if (version.isV0()){
            return getRotationBytes(angle, direction, speed);
        }else if (version.isAboveV0()){
            if (contains(getV1RemoteSpeeds(), speed)){
                return getRotationBytes(angle, direction, speed);
            }else
            if (contains(getV1SupportedSpeeds(), speed)){
                return getV1RotationBytes(angle, direction, speed);
            }
        }
        return null;
    }

    /**
     * This function should be called if the pivo version is v1,1
     * @param angle rotation angle
     * @param direction is pivo direction
     * @return an array of byte values is returned from function
     */
    byte[] getV1RotationBytes(int angle, Direction direction){
        return getV1RotationBytes(angle, direction, speed);
    }

    /**
     * This function should be called if the pivo version is v1,1
     * @param angle rotation angle
     * @param speed speed of
     * @param direction is pivo direction
     * @return an array of byte values is returned from function
     */
    byte[] getV1RotationBytes(int angle, Direction direction, int speed){
        byte[] data = this.current.clone();

        if (angle_step == Degree.DEGREE_1){
            //min rotation degree is 1.
            if (angle != -1){
                data[2] = (byte)(direction == Direction.RIGHT ? 0x26 : 0x28);
                data[3] = getV1SpeedByte(speed);
                data[5] = (byte) angle;
            }else {
                data[2] = (byte)(direction == Direction.RIGHT ? 0x26 : 0x28);
                data[3] = getV1SpeedByte(speed);
                data[5] = (byte) 0x00;
            }
        }else{
            //min rotation degree is 4.
            if (angle != -1){
                data[2] = (byte) (direction == Direction.RIGHT ? 0x22 : 0x24);
                data[3] = getV1SpeedByte(speed);
                data[5] = (byte) Math.floor(angle/4);
            } else {
                data[2] = (byte) (direction == Direction.RIGHT ? 0x22 : 0x24);
                data[3] = getV1SpeedByte(speed);
                data[5] = (byte) 0x00;
            }
        }
        return data;
    }

    byte[] getRotationBytes(int angle, Direction direction){
        return getRotationBytes(angle, direction, speed);
    }

    private byte[] getRotationBytes(int angle, Direction direction, int speed) {

        byte[] data = this.current.clone();

        if (angle_step == Degree.DEGREE_1){
            //min rotation degree is 1.
            if (angle != -1){
                data[2] = (byte)(direction == Direction.RIGHT ? 0x25 : 0x27);
                data[3] = getSpeedByte(speed);
                data[5] = (byte) angle;
            }else {
                data[2] = (byte)(direction == Direction.RIGHT ? 0x25 : 0x27);
                data[3] = getSpeedByte(speed);
                data[5] = (byte) 0x00;
            }
        }else{
            //min rotation degree is 4.
            if (angle != -1){
                data[2] = (byte) (direction == Direction.RIGHT ? 0x21 : 0x23);
                data[3] = getSpeedByte(speed);
                data[5] = (byte) Math.floor(angle/4);
            } else {
                data[2] = (byte) (direction == Direction.RIGHT ? 0x21 : 0x23);
                data[3] = getSpeedByte(speed);
                data[5] = (byte) 0x00;
            }
        }
        return data;
    }

    byte[] getStopBytes() {
        return this.stop;
    }

    byte[] getSpeedData(int speed){
        this.speed = speed;
        if (version.isV0()){
            return getSpeedBytes(speed);
        }else if (version.isAboveV0()){
            if (contains(getV1RemoteSpeeds(), speed)){
                return getSpeedBytes(speed);
            }else if (contains(getV1SupportedSpeeds(), speed)){
                return getV1SpeedBytes(speed/2);
            }else {
                this.speed = 10;
            }
        }
        return new byte[0];
    }

    boolean isSpeedSupported(int speed){
        return ((version.isV0() && contains(getV0SupportedSpeeds(), speed)) ||
                ((version.isAboveV0())
                        && contains(getV1SupportedSpeeds(), speed)));
    }

    boolean isSupportedAngle(int angle){
        return angle>=-1;
    }


    private boolean contains(int[] speeds, int speed){
        for (int i=0; i<speeds.length; i++){
            if (speeds[i] == speed)return true;
        }
        return false;
    }

    /**
     * This function generates speed for V1.
     * @return an array of byte values is returned from function
     */
    private byte[] getV1SpeedBytes(int speed){
        byte[] data = this.current.clone();

        data[2] = (byte) 0x10;
        data[3] = getV1SpeedByte(speed);

        return data;
    }

    /**
     * This function generates speed for V0 and V1.
     * @return an array of byte values is returned from function
     */
    private byte[] getSpeedBytes(int speed) {
        byte[] data = this.current.clone();

        data[2] = (byte) 0x06;
        data[3] = getSpeedByte(speed);

        return data;
    }

    /**
     * This function returns byte for specific speed level, it is supported only by V1
     * @return a byte value is returned from function
     */
    private byte getV1SpeedByte(int speed){
        if (speed>=2&&speed<=255){
            String hexCode = Integer.toHexString(speed);
            return (byte) Integer.parseInt(hexCode, 16);
        }
        return (byte)0x01;
    }

    byte getByte(int time){
        if (time>=1&&time<=255){
            String hexCode = Integer.toHexString(time);
            return (byte) Integer.parseInt(hexCode, 16);
        }
        return (byte)0x01;
    }

    /**
     * This function returns byte for specific speed level, it is supported by V1 and V0
     * @return a byte value is returned from function
     */
    /**
     * This function returns byte for specific speed level, it is supported by V1 and V0
     * @return a byte value is returned from function
     */
    private byte getSpeedByte(int speed) {
        byte data = (byte) 0x00;

        switch (speed){
            case 10:
                data = (byte) 0x11;
                break;
            case 20:
                data = (byte) 0x12;
                break;
            case 30:
                data = (byte) 0x13;
                break;
            case 50:
                data = (byte) 0x14;
                break;
            case 60:
                data = (byte) 0x15;
                break;
            case 120:
                data = (byte) 0x16;
                break;
            case 300:
                data = (byte) 0x17;
                break;
            case 600:
                data = (byte) 0x18;
                break;
            case 1200:
                data = (byte) 0x19;
                break;
            case 1800:
                data = (byte) 0x1A;
                break;
            case 3600:
                data = (byte) 0x1B;
                break;
        }
        if (version.isAboveV0()){
            switch (speed){
                case 7200:
                    data = (byte)0x1C;
                    break;
                case 10800:
                    data = (byte)0x1D;
                    break;
                case 21600:
                    data = (byte)0x1E;
                    break;
                case 43200:
                    data = (byte)0x1F;
                    break;
                case 64800:
                    data = (byte)0x20;
                    break;
                case 86400:
                    data = (byte)0x21;
                    break;
            }
        }
        return data;
    }

    int[] getRemoteSupportedSpeedList(){
        if (version.isV0()){
            return getV0RemoteSpeeds();
        }else if (version.isAboveV0()){
            return getV1RemoteSpeeds();
        }else {
            return new int[0];
        }
    }

    int[] getSupportedSpeedList(){
        if (version.isV0()){
            return getV0SupportedSpeeds();
        }else if (version.isAboveV0()){
            return getV1SupportedSpeeds();
        }else {
            return new int[0];
        }
    }

    private int[] getV0RemoteSpeeds(){
        return new int[]{10, 20, 30, 50, 60, 120, 300, 600, 1200, 1800, 3600};
    }

    private int[] getV1RemoteSpeeds(){
        return new int[]{10, 20, 30, 50, 60, 120, 300, 600, 1200, 1800, 3600, 7200, 10800, 21600, 43200, 64800, 86400};
    }

    private int[] getV0SupportedSpeeds(){
        return new int[]{10, 20, 30, 50, 60, 120, 300, 600, 1200, 1800, 3600};
    }

    private int[] getV1SupportedSpeeds(){
        int[] supportedSpeed = new int[255-5+1+/*remote supported speeds(start from 600 spr and 86400 spr)*/+10];
        int count = 0;
        for (int i=5; i<=255; i++){
            supportedSpeed[count] = i*2;
            count++;
        }
        supportedSpeed[count] = 600;
        supportedSpeed[count+1] = 1200;
        supportedSpeed[count+2] = 1800;
        supportedSpeed[count+3] = 3600;
        supportedSpeed[count+4] = 7200;
        supportedSpeed[count+5] = 10800;
        supportedSpeed[count+6] = 21600;
        supportedSpeed[count+7] = 43200;
        supportedSpeed[count+8] = 64800;
        supportedSpeed[count+9] = 86400;
        return supportedSpeed;
    }

    void setDegree(Degree angle){
        this.angle_step=angle;
    }

    Degree getDegree(){
        return this.angle_step;
    }

    void setSound(Sound sound){
        this.sound = sound;
    }

    Sound getSound(){
        return sound;
    }

    void setMode(Mode mode){
        this.mode = mode;
    }

    Mode getMode(){
        return this.mode;
    }

    UUID getServiceUUID() {
        return PivoUUID.serviceUUID;
    }

    UUID getRxUUID() {
        return PivoUUID.rxUUID;
    }

    UUID getTxUUID() {
        return PivoUUID.txUUID;
    }
}
