package app.pivo.android.prosdk.events;
import android.util.Log;
import android.util.SparseArray;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;

import java.lang.annotation.Retention;
import java.util.HashMap;
import java.util.Map;

import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.CompositeDisposable;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;
import io.reactivex.subjects.PublishSubject;

import static java.lang.annotation.RetentionPolicy.SOURCE;

/**
 * PivoEventBus allows to listen and send Pivo events {@link PivoEvent} in different components of the application.
 */
public final class PivoEventBus {

    private static SparseArray<PublishSubject<Object>> sSubjectMap = new SparseArray<>();
    private static Map<Object, CompositeDisposable> sSubscriptionsMap = new HashMap<>();

    /**
     * Connection completed subject.
     */
    public static final int CONNECTION_COMPLETED = 0;

    /**
     * Connection failed or an app is disconnected from Pivo.
     */
    public static final int CONNECTION_FAILURE = 1;

    /**
     * Scan device subject.
     */
    public static final int SCAN_DEVICE = 2;

    /**
     * Remote controller subject.
     */
    public static final int REMOTE_CONTROLLER = 3;

    /**
     * Name is changed subject.
     */
    public static final int NAME_CHANGED = 4;

    /**
     * Pivo notification subject
     */
    public static final int PIVO_NOTIFICATION = 5;

    /**
     * MacAddress subject
     */
    public static final int MAC_ADDRESS = 6;

    @Retention(SOURCE)
    @IntDef({CONNECTION_COMPLETED, CONNECTION_FAILURE, SCAN_DEVICE, REMOTE_CONTROLLER, NAME_CHANGED, PIVO_NOTIFICATION, MAC_ADDRESS})
    @interface Subject {
    }

    private PivoEventBus() {
        // hidden constructor
    }

    /**
     * Get the subject or create it if it's not already in memory.
     */
    @NonNull
    private static PublishSubject<Object> getSubject(@Subject int subjectCode) {
        PublishSubject<Object> subject = sSubjectMap.get(subjectCode);
        if (subject == null) {
            subject = PublishSubject.create();
            subject.subscribeOn(AndroidSchedulers.mainThread());
            sSubjectMap.put(subjectCode, subject);
        }

        return subject;
    }

    /**
     * Get the CompositeDisposable or create it if it's not already in memory.
     */
    @NonNull
    private static CompositeDisposable getCompositeDisposable(@NonNull Object object) {
        CompositeDisposable compositeDisposable = sSubscriptionsMap.get(object);
        if (compositeDisposable == null) {
            compositeDisposable = new CompositeDisposable();
            sSubscriptionsMap.put(object, compositeDisposable);
        }

        return compositeDisposable;
    }

    /**
     * Subscribe to the specified subject and listen for updates on that subject. Pass in an object to associate
     * your registration with, so that you can unsubscribe later.
     * <br/><br/>
     * <b>Note:</b> Make sure to call {@link PivoEventBus#unregister(Object)} to avoid memory leaks.
     */
    public static void subscribe(@Subject int subject, @NonNull Object lifecycle, @NonNull Consumer<Object> action) {
        Disposable disposable = getSubject(subject).subscribe(action, throwable -> Log.e("PivoEventBus", "Exception"));//.subscribe(action);
        getCompositeDisposable(lifecycle).add(disposable);
    }

    /**
     * Unregisters this object from the bus, removing all subscriptions.
     * This should be called when the object is going to go out of memory.
     */
    public static void unregister(@NonNull Object lifecycle) {
        //We have to remove the composition from the map, because once you dispose it can't be used anymore
        CompositeDisposable compositeDisposable = sSubscriptionsMap.remove(lifecycle);
        if (compositeDisposable != null) {
            compositeDisposable.dispose();
        }
    }

    /**
     * Publish an object to the specified subject for all subscribers of that subject.
     */
    public static void publish(@Subject int subject, @NonNull Object message) {
        getSubject(subject).onNext(message);
    }
}
