package app.pivo.android.prosdk;

import android.app.ActivityManager;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattService;
import android.content.Context;
import android.os.Handler;
import android.os.ParcelUuid;
import android.text.TextUtils;
import android.util.Log;
import android.widget.Toast;

import com.polidea.rxandroidble.NotificationSetupMode;
import com.polidea.rxandroidble.RxBleClient;
import com.polidea.rxandroidble.RxBleConnection;
import com.polidea.rxandroidble.RxBleDevice;
import com.polidea.rxandroidble.RxBleDeviceServices;
import com.polidea.rxandroidble.internal.RxBleLog;
import com.polidea.rxandroidble.scan.ScanFilter;
import com.polidea.rxandroidble.scan.ScanSettings;
import com.polidea.rxandroidble.utils.ConnectionSharingAdapter;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.UUID;

import app.pivo.android.prosdk.events.*;
import app.pivo.android.prosdk.exceptions.*;
import app.pivo.android.prosdk.util.Degree;
import app.pivo.android.prosdk.util.Direction;
import app.pivo.android.prosdk.util.PivoDevice;
import app.pivo.android.prosdk.util.PivoDeviceInfo;
import app.pivo.android.prosdk.util.Sound;
import app.pivo.android.prosdk.util.Version;
import rx.Observable;
import rx.Subscription;
import rx.android.schedulers.AndroidSchedulers;
import rx.internal.util.SubscriptionList;
import rx.subjects.PublishSubject;

import static app.pivo.android.prosdk.events.PivoEventBus.NAME_CHANGED;


class PivoController extends IPivoController {

    private Context context;

    private RxBleDevice bleDevice;
    private RxBleClient rxBleClient;
    private final PublishSubject<Void> disconnectTriggerSubject = PublishSubject.create();
    private Observable<RxBleConnection> connectionObservable;

    private Rotator instance;

    private String macAddress = "";
    private byte[] serialNumberBytes;

    private boolean waitingForData = false;
    private boolean isTimerRunning = false;
    private boolean keepAlive = true;
    private boolean isBypassOn = false;
    private boolean isInternalApp = false;
    private boolean isLogEnabled = false;
    private PivoDevice pivoDevice;
    private PivoDeviceInfo pivoDeviceInfo;

    private IVerify verifier;

    static class Builder {
        PivoController pivoController = new PivoController();

        Builder setContext(Context context) {
            pivoController.context = context;
            pivoController.verifier = new Verifier(context);

            RxBleClient.setLogLevel(RxBleLog.DEBUG);
            pivoController.setBleClient(pivoController.getRxBleClient());
            return this;
        }

        PivoController build() {
            return pivoController;
        }
    }

    private void setBleClient(RxBleClient client) {
        rxBleClient = client;
    }

    private void setBleDevice(RxBleDevice bleDevice) {
        this.bleDevice = bleDevice;
    }

    private void setMacAddress(String macAddress) {
        this.macAddress = macAddress;
        setBleDevice(getRxBleClient().getBleDevice(macAddress));
    }

    private PublishSubject<Void> getDisconnectTriggerSubject() {
        return disconnectTriggerSubject;
    }

    private RxBleClient getRxBleClient() {
        if (rxBleClient == null) {
            rxBleClient = RxBleClient.create(context);
        }

        return rxBleClient;
    }

    private RxBleDevice getBleDevice() {
        if (TextUtils.isEmpty(macAddress)) return null;

        setBleDevice(getRxBleClient().getBleDevice(macAddress));
        return bleDevice;
    }

    private void connect() {
        if (getBleDevice() != null) {
            connectionObservable = prepareConnectionObservable();
            subscriptionList.add(
                    connectionObservable.subscribe(this::onConnectionReceived, this::onConnectionFailure));
        }
    }

    private Observable<RxBleConnection> prepareConnectionObservable() {
        return bleDevice
                .establishConnection(false)
                .takeUntil(disconnectTriggerSubject)
                .doOnUnsubscribe(this::updateUI)
                .compose(new ConnectionSharingAdapter());
    }

    private void onConnectionFailure(Throwable throwable) {
        postConnectionFailure();
    }

    private void onConnectionReceived(RxBleConnection connection) {
        Subscription serviceSubscription = connection.discoverServices()
                .observeOn(AndroidSchedulers.mainThread())
                .doOnUnsubscribe(this::updateUI)
                .subscribe(this::swapScanResult, this::onConnectionFailure);
        subscriptionList.add(serviceSubscription);
    }

    private Boolean bothChannelsSetup = false;

    private void onNotificationChannelSetupSuccess(Observable<byte[]> notificationObservable) {
        Subscription notificationSub = notificationObservable
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(this::onNotificationReceived, this::onNotificationSetupFailure);

        subscriptionList.add(notificationSub);

        if (bothChannelsSetup)
            postConnectionReceived();

        bothChannelsSetup = true;
    }

    private void onNotificationReceived(byte[] bytes) {
        if (bytes.length < 6) {
            printData("Incomplete bytes: ", bytes);
            return;
        }

        // Authentication Notification
        if (bytes[2] == 0x6F) {
            if (bytes[3] == 0x01) {// Pivo HW to App with answer
                int inquiry = (bytes[6] & 0xff) << 24 | (bytes[7] & 0xff) << 16 | (bytes[8] & 0xff) << 8 | (bytes[9] & 0xff);
                int answer = (bytes[10] & 0xff) << 24 | (bytes[11] & 0xff) << 16 | (bytes[12] & 0xff) << 8 | (bytes[13] & 0xff);

                byte[] sendBytes = instance.authenticatePivo((byte) 0x02, answer, inquiry);

                if (sendBytes == null) { // Error
                    return;
                }
                //printData(sendBytes, "3rd Handshake Stream : ");
                write(sendBytes);

                if (sendBytes[5] == 0) {
                    // Status 0: Pivo HW Authentication Fail!
                    postNotifyPivoAuthResult(0);

                    // Disconnect the connection
                    if (isPivoConnected()) {
                        disconnectAll();
                    }
                }
            } else if (bytes[3] == 0x03) {    // Verification notificatoin from Pivo HW
                if (bytes[5] == 1) {// Check authentication result from Pivo
                    // Status 2: Mutual Authentication Sucess
                    postNotifyPivoAuthResult(2);
                } else {
                    // Status 1: Pivo HW rejected this app
                    postNotifyPivoAuthResult(1);
                }
            }
        } else if (!(instance.getBatteryLevel(bytes) == -1)) {
            postBatteryLevelChanged(instance.getBatteryLevel(bytes));
        } else if (instance.verifyVersion(bytes) != null) {//check Pivo version handler, after receive version, trigger "Connection_Completed" event
            postVersionUpdate(instance.getVersion());
        } else if (bytes[2] == 5) {// Remote Controller events from a device to App
            postRemoteControllerEvent(instance.getRemoteControllerButtonEvent(bytes));
        } else if (bytes[2] == (byte) 0x1C) {
            postSerialNumber(bytes);
        } else if (bytes[2] == (byte) 0x1B) {
            postMacAddress(bytes);
        } else if (bytes[2] == (byte) 0xA1) {
            postTimeoutSet(bytes);
        } else if (bytes[2] == (byte) 0x60) {
            onBypassReceived(bytes[3] == (byte) 0xA0);
        } else if (bytes[2] == (byte) 0x10 || bytes[2] == (byte) 0x06) {
            // TODO: 2020/11/26 New Speed is set
        } else if (bytes[2] == (byte) 0xFA && bytes[5] == (byte) 0x21) {
            // if Notification receiving is enabled, it sends rotated to 1 to right direction
            onRotatedRight();
        } else if (bytes[2] == (byte) 0xFA && bytes[5] == (byte) 0x23) {
            // if Notification receiving is enabled, it sends rotated to 1 to left direction
            onRotatedLeft();
        } else if (bytes[2] == (byte) 0X7F && bytes[4] == (byte) 0x01) {// if the name is changed
            PivoEventBus.publish(NAME_CHANGED, new PivoEvent.NameChanged(getName(bytes)));
        } else if (bytes[2] == (byte) 0x51 && bytes[4] == (byte) 0x01) {
            // if pairing started
        } else if (bytes[2] == (byte) 0x52 && bytes[4] == (byte) 0x01) {
            //pair cancelled
        } else if (bytes[2] == (byte) 0x55 && bytes[4] == (byte) 0x01 && bytes[5] == (byte) 0x01) {
            //pair succeed
        } else if (bytes[2] == (byte) 0x53 && bytes[4] == (byte) 0x01 && bytes[5] == (byte) 0x00) {
            //pair failed
        } else if (bytes[2] == (byte) 0x54 && bytes[4] == (byte) 0x01 && bytes[5] == (byte) 0x01) {
            // pressing power button 3 times quickly
        } else if (bytes[2] == (byte) 0x54 && bytes[4] == (byte) 0x01 && bytes[5] == (byte) 0x00) {
            // Pressing power button once within 3 s in pairing mode
        } else if (bytes[2] == (byte) 0x55 && bytes[4] == (byte) 0x01 && bytes[5] == (byte) 0x00) {
            // going from pairing
        } else {
            if (!waitingForData) return;
            waitingForData = false;
            postNotificationReceived();
        }
    }

    private String getName(byte[] bytes) {
        StringBuilder name = new StringBuilder();
        //0X05	0XAF	0X7F	LEN	0X00	CHAR0	CHAR1	CHAR2	CHAR3	CHAR4	CHAR5	CHAR6
        //Command To App: (AA 55 81 00) 05 AF 7F 03 01 67 67 67
        if (bytes[0] == (byte) 0x05 && bytes[2] == (byte) 0x7F && bytes.length >= 5) {
            for (int i = 5; i < bytes.length; i++) {
                name.append((char) bytes[i]);
            }
        } else if (bytes[0] == (byte) 0xAA && bytes[1] == (byte) 0x55 && bytes.length >= 10) {
            for (int i = 9; i < bytes.length; i++) {
                name.append((char) bytes[i]);
            }
        }
        return name.toString();
    }

    private void onNotificationSetupFailure(Throwable throwable) {
        printLog(Log.ERROR, "Setting up notifications failed: " + throwable);
    }

    private void onWriteSuccess(byte[] bytes) {
        printLog(Log.INFO, "Command sent successfully.");

        waitingForData = true;

        try {
            writeSubscriptions.remove().unsubscribe();
        } catch (NoSuchElementException e) {
            e.printStackTrace();
        }
    }

    private void onWriteFailure(Throwable throwable) {
        printLog(Log.ERROR, "Sending command failed.");

        postConnectionFailure();
        try {
            writeSubscriptions.remove().unsubscribe();
        } catch (NoSuchElementException e) {
            e.printStackTrace();
        }
    }

    private void swapScanResult(RxBleDeviceServices services) {
        BluetoothGattService mBTService = null;

        for (BluetoothGattService service : services.getBluetoothGattServices()) {
            instance = Rotator.getInstance(service.getUuid(), true/*verifier.checkAllowedApp()*/);
            if (instance != null) {
                mBTService = service;
                break;
            }
        }

        if (instance == null) {
            postConnectionFailure();
            return;
        }

        if (mBTService == null || mBTService.getUuid() == null) {
            postConnectionFailure();
            return;
        }

        for (BluetoothGattCharacteristic characteristic : mBTService.getCharacteristics()) {
//            Log.e(TAG,"------------ Characteristic: " + characteristic.getUuid());
//            Log.e(TAG, "Read: " + isCharacteristicReadable(characteristic));
//            Log.e(TAG, "Notify: " + isCharacteristicNotifiable(characteristic));
//            Log.e(TAG, "Write: " + isCharacteristicWriteable(characteristic));
//            Log.e(TAG, "getDescriptors: " + characteristic.getDescriptors().size());
            writeNotificationCharacteristic(characteristic);
        }

        //postConnectionReceived();
    }

    private final Handler handler = new Handler();
    private final Runnable runnable = this::afficher;

    private void afficher() {
        printLog(Log.INFO, "Timer »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»» afficher  »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»");
        if (isTimerRunning && isAppOnForeground()) {
            requestBatteryLevel();
        }
        handler.postDelayed(runnable, 20000);
    }

    private boolean isAppOnForeground() {
        ActivityManager activityManager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
        List<ActivityManager.RunningAppProcessInfo> appProcesses = activityManager.getRunningAppProcesses();
        if (appProcesses == null) {
            return false;
        }
        final String packageName = context.getPackageName();
        for (ActivityManager.RunningAppProcessInfo appProcess : appProcesses) {
            if (appProcess.importance == ActivityManager.RunningAppProcessInfo.IMPORTANCE_FOREGROUND && appProcess.processName.equals(packageName)) {
                return true;
            }
        }
        return false;
    }

    private void startTimer() {
        if (isTimerRunning) return;
        printLog(Log.INFO, "Timer »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»» startTimer  »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»");
        isTimerRunning = true;
        runnable.run();
    }

    private void stopTimer() {
        printLog(Log.INFO, "Timer »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»» stopTimer  »»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»»");
        handler.removeCallbacks(runnable);
        isTimerRunning = false;
    }

    private void write(byte[] data) {
        printData("Sent data: ", data);
        if (isPivoConnected() && connectionObservable != null) {
            Subscription subscription = connectionObservable
                    .flatMap(rxBleConnection -> rxBleConnection.writeCharacteristic(instance.getRxUUID(), data))
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(this::onWriteSuccess, this::onWriteFailure);
            writeSubscriptions.add(subscription);
        } else {
            postConnectionFailure();
        }
    }

    private void writeNotificationCharacteristic(BluetoothGattCharacteristic chr) {
        if (isPivoConnected()) {

            Subscription subscription = connectionObservable
                    .flatMap(rxBleConnection -> rxBleConnection.setupNotification(chr.getUuid(), NotificationSetupMode.COMPAT))
//                    .flatMap(notificationObservable -> notificationObservable)
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(this::onNotificationChannelSetupSuccess, this::onNotificationSetupFailure);
            subscriptionList.add(subscription);
        }
    }

    void authenticatePivo() {
        if (instance == null) {
            printLog(Log.ERROR, "There is no connected device. (authenticatePivo())");
            postConnectionFailure();
            return;
        }
        byte[] arrAuthInit = instance.authenticatePivo((byte) 0, 0, 0);
        write(arrAuthInit);
    }

    private void checkConnection() {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        write(instance.checkConnection());
    }

    private void getMacAddress() {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        if (getVersion().getVersion() < 5) {
            printLog(Log.ERROR, "There is no connected device.");
        }
        write(instance.getMacAddress());
    }

    private void requestSerialNumber() {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        write(instance.getSerialNumber());
    }

    @Override
    void setTimeout(int sec) {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        write(instance.getTimeoutByteData(sec));
    }

    @Override
    void getLicenseContent(String content) throws LicenseDateException, PackageNameException, InvalidLicenseException, SdkPlanTypeException {
        if (verifier != null) {
            verifier.readLicense(content);
        }
    }

    @Override
    void enableLog(boolean enabled) {
        this.isLogEnabled = enabled;
    }

    @Override
    boolean isVerified() {
        if (verifier != null) {
            isInternalApp = verifier.isValid();
            return isInternalApp;
        } else {
            return false;
        }
    }

    @Override
    void turnLeft(int angle) {
        rotate(angle, Direction.LEFT);
    }

    @Override
    void turnLeft(int angle, int speed) throws UnsupportedSpeedException {
        rotate(angle, Direction.LEFT, speed);
    }

    @Override
    void turnRight(int angle) {
        rotate(angle, Direction.RIGHT);
    }

    @Override
    void turnRight(int angle, int speed) throws UnsupportedSpeedException {
        rotate(angle, Direction.RIGHT, speed);
    }

    @Override
    void turnRightContinuously() {
        rotate(-1, Direction.RIGHT);
    }

    @Override
    void turnRightContinuously(int speed) throws UnsupportedSpeedException {
        rotate(-1, Direction.RIGHT, speed);
    }

    @Override
    void turnLeftContinuously() {
        rotate(-1, Direction.LEFT);
    }

    @Override
    void turnLeftContinuously(int speed) throws UnsupportedSpeedException {
        rotate(-1, Direction.LEFT, speed);
    }

    @Override
    void turnLeftWithFeedback(int angle, int speed) throws UnsupportedSpeedException {
        rotateWithFeedback(angle, Direction.LEFT, speed);
    }

    @Override
    void turnRightWithFeedback(int angle, int speed) throws UnsupportedSpeedException {
        rotateWithFeedback(angle, Direction.RIGHT, speed);
    }

    @Override
    void turnHold() {
        if (instance == null) {
            postConnectionFailure();
            return;
        }

        try {
            if (instance.getVersion() != null) {
                if (instance.getVersion().isAboveV0()) {
                    rotate(1, Direction.LEFT, 510);
                } else {
                    rotateWithFeedback(1, Direction.LEFT, 3600);
                }
            }
        } catch (UnsupportedSpeedException e) {
            e.printStackTrace();
        }
    }

    @Override
    void stop() {
        if (instance == null) {
            printLog(Log.ERROR, "There is no connected device.");
            postConnectionFailure();
            return;
        }

        write(instance.getStopBytes());
        waitingForData = false;
    }

    @Override
    void changeName(String name) {
        if (instance != null) {
            if (isPivoConnected() && connectionObservable != null) {
                write(instance.changeRotatorName(name));
            }
        }
    }

    @Override
    void changeSpeed(int speed) {
        if (instance == null) {
            printLog(Log.ERROR, "There is no connected device.");
            postConnectionFailure();
            return;
        }
        if (instance.isSupportedSpeed(speed)) {
            write(instance.getSpeedData(speed));
        } else {
            printLog(Log.ERROR, "The given speed is not supported, please call to get supported speeds from getSupportedSpeeds()");
        }
    }

    @Override
    boolean isPivoConnected() {
        return bleDevice != null && bleDevice.getConnectionState() == RxBleConnection.RxBleConnectionState.CONNECTED;
    }

    @Override
    Version getVersion() {
        if (instance == null) {
            postConnectionFailure();
            return null;
        }

        return instance.getVersion();
    }

    @Override
    String getPivoVersion() {
        if (instance == null) {
            postConnectionFailure();
            return "UNKNOWN";
        }

        return "VERSION_" + instance.getVersion().getVersion();
    }

    @Override
    PivoDeviceInfo getDeviceInfo() {
        if (instance == null) {
            postConnectionFailure();
            return null;
        }

        return pivoDeviceInfo;
    }

    @Override
    void turnOnNotification() {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        write(instance.turnOnNotifier());
    }

    @Override
    void turnOffNotification() {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        write(instance.turnOffNotifier());
    }

    @Override
    void changeMode() {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        // TODO: 2020/04/01 add mode
//        rotator.setMode(mode);
//        write(rotator.getModeBytes());
    }

    @Override
    void startPairingMode() {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        write(instance.startPairing());
    }

    @Override
    void cancelPairingMode() {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        write(instance.cancelPairing());
    }

    @Override
    void makeBeepSound(Sound sound) {
        if (instance == null) {
            postConnectionFailure();
            return;
        }

        instance.setSound(sound);
        write(instance.getSoundBytes());
    }

    @Override
    void keepAlive(boolean alive) {
        this.keepAlive = alive;
        if (keepAlive) {
            stopTimer();
            startTimer();
        } else {
            stopTimer();
        }
    }

    @Override
    void changeMinRotationDegree(Degree degree) {
        if (instance == null) {
            postConnectionFailure();
            return;
        }

        instance.setDegree(degree);
    }

    @Override
    void scan() {
        if (isScanning()) return;
        if (!isVerified()) {
            printLog(Log.ERROR, "******************* Exception: Invalid license is provided. *********************");
            return;
        }
        //disconnect first, if the phone's connected to the device
        while (getDisconnectTriggerSubject().hasObservers()) {
            getDisconnectTriggerSubject().onNext(null);
        }

        //clear results if they're scanned earlier
        unsubscribe();

        scanSubscription = rxBleClient.scanBleDevices(
                new ScanSettings.Builder()
                        .setScanMode(ScanSettings.SCAN_MODE_LOW_LATENCY)
                        .setCallbackType(ScanSettings.CALLBACK_TYPE_ALL_MATCHES)
                        .build(),
                new ScanFilter.Builder()
                        .setServiceUuid(new ParcelUuid(PivoUUID.serviceUUID))
                        .build()
        )
                .observeOn(AndroidSchedulers.mainThread())
                .doOnUnsubscribe(this::unsubscribe)
                .subscribe(this::addDevice, throwable -> {
                    printLog(Log.ERROR, "No device found.");
                });
    }

    @Override
    void connectTo(PivoDevice pivoDevice) {
        if (!isVerified()) {
            printLog(Log.ERROR, "******************* Exception: Invalid license is provided. *********************");
            return;
        }
        setMacAddress(pivoDevice.getMacAddress());
        this.pivoDevice = pivoDevice;
        bothChannelsSetup = false;
        disconnectAll();
        connect();
    }

    @Override
    void stopScan() {
        disconnectAll();
    }

    @Override
    void disconnect() {
        disconnectAll();

        postConnectionFailure();
    }

    private void disconnectAll() {
        while (getDisconnectTriggerSubject().hasObservers()) {
            getDisconnectTriggerSubject().onNext(null);
        }

        unsubscribe();
    }

    @Override
    int[] getSupportedSpeeds() {
        if (instance == null) {
            postConnectionFailure();
            return new int[0];
        }
        return instance.getSupportedSpeeds();
    }

    @Override
    int[] getSupportedSpeedsWithFeedback() {
        if (instance == null) {
            postConnectionFailure();
            return new int[0];
        }
        return instance.getSupportedSpeedWithFeedback();
    }

    @Override
    void requestBatteryLevel() {
        if (instance != null) {
            printLog(Log.INFO, "Battery level command is sent.");

            if (isPivoConnected() && connectionObservable != null) {
                write(instance.battery_Level());
            } else {
                stopTimer();
            }
        }
    }

    @Override
    void enableBypass(boolean activate) {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        if (isBypassSupported()) {
            write(instance.enableBypass(activate));
        }
    }

    @Override
    boolean isBypassOn() {
        return isBypassOn;
    }

    @Override
    boolean isBypassSupported() {
        if (instance == null) {
            postConnectionFailure();
            return false;
        }
        return instance.getVersion().isAboveV1();
    }

    private Subscription scanSubscription;
    private SubscriptionList subscriptionList = new SubscriptionList();
    private final LinkedList<Subscription> writeSubscriptions = new LinkedList();

    private boolean isScanning() {
        return scanSubscription != null;
    }

    private void addDevice(com.polidea.rxandroidble.scan.ScanResult result) {
        if (result.getBleDevice() != null) {
            PivoEventBus.publish(PivoEventBus.SCAN_DEVICE, new PivoEvent.Scanning(new PivoDevice(result.getBleDevice().getName(), result.getBleDevice().getMacAddress())));
        }
    }

    //clear the results and unsubscribe
    private void unsubscribe() {
        unSubscribeScanning();
        /* Remove all the subscription that we have */
        subscriptionList.unsubscribe();
        /* Refresh the list, because it is one time use only */
        subscriptionList = new SubscriptionList();

        while (!writeSubscriptions.isEmpty()) {
            try {
                writeSubscriptions.remove().unsubscribe();
            } catch (NoSuchElementException e) {
                e.printStackTrace();
            }
        }
    }

    private void unSubscribeScanning() {
        if (scanSubscription != null) {
            scanSubscription.unsubscribe();
            scanSubscription = null;
        }
    }

    private void rotateWithFeedback(int angle, Direction direction, int speed) throws UnsupportedSpeedException {
        if (instance == null) {
            postConnectionFailure();
            return;
        }

        if (!instance.isSupportedAngle(angle)) {
            printLog(Log.ERROR, "The given rotation angle is unsupported, please give positive numbers");
            return;
        }

        if (!instance.isSupportedSpeedWithFeedback(speed)) {
            throw new UnsupportedSpeedException(String.format("The given speed %s is not supported, please call to get supported speeds from getSupportedSpeedsWithFeedback().", speed));
        }

        write(instance.getRotationWithFeedbackBytes(angle, direction, speed));
    }

    private void rotate(int angle, Direction direction, int speed) throws UnsupportedSpeedException {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        if (!instance.isSupportedAngle(angle)) {
            printLog(Log.ERROR, "The given rotation angle is unsupported, please give positive numbers.");
            return;
        }

        if (!instance.isSupportedSpeed(speed)) {
            throw new UnsupportedSpeedException(String.format("The given speed %s is not supported, please call to get supported speeds from getSupportedSpeeds().", speed));
        }

        write(instance.getRotatingData(angle, direction, speed));
    }

    private void rotate(int angle, Direction direction) {
        if (instance == null) {
            postConnectionFailure();
            return;
        }
        if (!instance.isSupportedAngle(angle)) {
            printLog(Log.ERROR, "The given rotation angle is unsupported, please give positive numbers.");
            return;
        }

        write(instance.getRotatingData(angle, direction));
    }

    private void updateUI() {
    }

    private boolean isCharacteristicNotifiable(BluetoothGattCharacteristic characteristic) {
        return (characteristic.getProperties() & BluetoothGattCharacteristic.PROPERTY_NOTIFY) != 0;
    }

    private boolean isCharacteristicReadable(BluetoothGattCharacteristic characteristic) {
        return ((characteristic.getProperties() & BluetoothGattCharacteristic.PROPERTY_READ) != 0);
    }

    private boolean isCharacteristicWriteable(BluetoothGattCharacteristic characteristic) {
        return (characteristic.getProperties() & (BluetoothGattCharacteristic.PROPERTY_WRITE
                | BluetoothGattCharacteristic.PROPERTY_WRITE_NO_RESPONSE)) != 0;
    }

    private PivoDeviceInfo getPivoDeviceInfo() {
        String serialNumber = PivoSerialNumberUtil
                .getSerialNumber(getVersion(), serialNumberBytes, pivoDevice.getMacAddress());

        return new PivoDeviceInfo(
                pivoDevice.getName(),
                pivoDevice.getMacAddress(),
                getVersion().getVersion(),
                getVersion().getPivoType(),
                serialNumber);
    }

    private void onBypassReceived(boolean on) {
        isBypassOn = on;
        PivoEventBus.publish(PivoEventBus.BYPASS, new PivoEvent.BypassEvent(isBypassOn));
    }

    /// Posters
    private void postConnectionFailure() {
        this.pivoDevice = null;
        this.pivoDeviceInfo = null;
        isBypassOn = false;
        stopTimer();
        PivoEventBus.publish(PivoEventBus.CONNECTION_FAILURE, new PivoEvent.ConnectionFailure());
        printLog(Log.ERROR, "There is no connected rotator!");
    }

    private void postVersionUpdate(Version version) {
        if (version.isAboveV1()) {
            authenticatePivo();
        } else {
            onConnectionEstablished();
        }
    }

    private void postSerialNumber(byte[] serialNumBytes) {
        this.serialNumberBytes = serialNumBytes;//= convertHexToSerial(bytes);
        onConnectionEstablished();
    }

    private void postTimeoutSet(byte[] bytes) {
        PivoEventBus.publish(PivoEventBus.PIVO_NOTIFICATION, new PivoEvent.Timeout(bytes[5]));
    }

    private void postMacAddress(byte[] bytes) {
        if (bytes.length >= 8) {
            PivoEventBus.publish(PivoEventBus.MAC_ADDRESS, new PivoEvent.MacAddress("MacAddress: " + convertDataToString(Arrays.copyOfRange(bytes, 3, 9))));
        }
    }

    private void postNotifyPivoAuthResult(int status) {
        if (status == 2) {
            if (getVersion().getVersion() >= 5) {
                requestSerialNumber();
            } else {
                onConnectionEstablished();
            }
        } else {
            disconnectAll();
        }
    }

    private void onConnectionEstablished() {
        pivoDeviceInfo = getPivoDeviceInfo();
        PivoEventBus.publish(PivoEventBus.CONNECTION_COMPLETED, new PivoEvent.ConnectionComplete(pivoDeviceInfo));
        startTimer();
    }

    private void postConnectionReceived() {
        printLog(Log.INFO, "postConnectionReceived");
        handler.postDelayed(() -> {
            checkConnection();
            // unsubscribe scanning
            unSubscribeScanning();
        }, 2000);
    }

    private void onRotatedRight() {
        PivoEventBus.publish(PivoEventBus.PIVO_NOTIFICATION, new PivoEvent.Rotated(Direction.RIGHT));
    }

    private void onRotatedLeft() {
        PivoEventBus.publish(PivoEventBus.PIVO_NOTIFICATION, new PivoEvent.Rotated(Direction.LEFT));
    }

    private void postNotificationReceived() {
        PivoEventBus.publish(PivoEventBus.PIVO_NOTIFICATION, new PivoEvent());
    }

    /**
     * This callback is triggered if there's battery changes in Pivo
     */
    private void postBatteryLevelChanged(int level) {
        PivoEventBus.publish(PivoEventBus.PIVO_NOTIFICATION, new PivoEvent.BatteryChanged(level));
    }

    /**
     * RemoteControllerClick callback handler
     */
    private void postRemoteControllerEvent(RemoteButtonEvent event) {
        switch (event) {
            case MODE:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCMode(event.state));
                break;
            case CAMERA:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCCamera(event.state));
                break;
            case STOP:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCStop(event.state));
                break;
            case LEFT:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCLeft(event.state));
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCLeftPressed());
                break;
            case RIGHT:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCRight(event.state));
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCRightPressed());
                break;
            case LEFT_CONTINUOUS:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCLeftContinuous(event.state));
                break;
            case RIGHT_CONTINUOUS:
                PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCRightContinuous(event.state));
                break;
            case SPEED_CHANGE:
                int speedValue = getSpeedValue(event.speed);
                if (instance.getVersion().isV0()) {
                    PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCSpeed(speedValue, event.state));
                } else {
                    PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCSpeed(speedValue, event.direction, event.state));
                    if (event.direction == 1) {
                        PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCSpeedUp(speedValue, event.state));
                    } else if (event.direction == 0) {
                        PivoEventBus.publish(PivoEventBus.REMOTE_CONTROLLER, new PivoEvent.RCSpeedDown(speedValue, event.state));
                    }
                }
                break;
        }
    }

    private int getSpeedValue(Rotator.Speed speed) {
        switch (speed) {
            case SPEED_2:
                return 20;
            case SPEED_3:
                return 30;
            case SPEED_4:
                return 50;
            case SPEED_5:
                return 60;
            case SPEED_6:
                return 120;
            case SPEED_7:
                return 300;
            case SPEED_8:
                return 600;
            case SPEED_9:
                return 1200;
            case SPEED_10:
                return 1800;
            case SPEED_11:
                return 3600;
            case SPEED_12:
                return 7200;
            case SPEED_13:
                return 10800;
            case SPEED_14:
                return 21600;
            case SPEED_15:
                return 43200;
            case SPEED_16:
                return 64800;
            case SPEED_17:
                return 86400;
            default:
                return 10;
        }
    }

    private void printLog(int priority, String logContent) {
        if (isLogEnabled) {
            Log.println(priority, "PivoSdk", logContent);
        }
    }

    private void printData(String logContent, byte[] data) {
        if (isLogEnabled && isInternalApp) {
            Log.println(Log.INFO, "PivoSdk", logContent + ": " + convertDataToString(data));
        }
    }

    private String convertDataToString(byte[] data) {
        final StringBuilder stringBuilder = new StringBuilder(data.length);
        for (byte byteChar : data)
            stringBuilder.append(String.format("%02X ", byteChar));
        return stringBuilder.toString();
    }
}