package app.pivo.android.prosdk;
import android.graphics.PointF;
import android.util.Log;

import app.pivo.android.prosdk.exceptions.UnsupportedSpeedException;

class PivoSpeedController implements  IPivoSpeedController {

    private static PivoSpeedController instance;

    private final PivoRotatorConfig mPivoRotatorConfig = PivoRotatorConfig.getInstance();

    private IPivoController mController;


    private int mWidth;
    private int mHeight;
    private int mOrientation;
    private double mZoom = 1.f;

    private boolean isFront = true;
    private boolean isOrientationLocked;

    private PositionType positionType = PositionType.MIDDLE;

    private SensitivityMode mode = SensitivityMode.NORMAL;


    private PointF lastPoint = new PointF(0,0);

    public static PivoSpeedController getInstance (){
        if (instance ==null){
            instance = new PivoSpeedController();
            return instance;
        }
        return instance;
    }

    @Override
    public PivoSpeedController setController(IPivoController pivoSpeedController) {
        mController = pivoSpeedController;
        return this;
    }

    @Override
    public PivoSpeedController setPositionType(PositionType positionType) {
        this.positionType = positionType;
        return this;
    }

    @Override
    public PivoSpeedController setPreviewSize(int width, int height) {
        mWidth = width;
        mHeight = height;
        return this;
    }

    @Override
    public PivoSpeedController setCameraFacing(boolean isFront) {
        this.isFront = isFront;
        return this;
    }

    @Override
    public PivoSpeedController setOrientation(int orientation) {
        mOrientation = orientation;
        return this;
    }

    @Override
    public PivoSpeedController setZoomFactor(double zoom) {
        mZoom = zoom;
        return this;
    }

    @Override
    public PivoSpeedController setOrientationLocked(boolean locked) {
        isOrientationLocked = locked;
        return this;
    }

    @Override
    public PivoSpeedController setSensitivityMode(SensitivityMode sensitivityMode) {
        mode = sensitivityMode;
        return this;
    }

    @Override
    public void track(PointF currentPoint) {
        if(mode.equals(SensitivityMode.NONE)){
            mController.turnHold();
            return;
        }
        mPivoRotatorConfig.initConfig(mZoom , mode, mController.getVersion()); // initialize with zoom and Sensitivity
        PointF diff = new PointF(getDesiredPoint().x - currentPoint.x, getDesiredPoint().y - currentPoint.y);


        if (lastPoint.equals(new PointF(0, 0))) {
            lastPoint = currentPoint;
        }
        if (isOrientationLocked){
            if ((mOrientation % 2 == 1)){
                handleTracking(currentPoint,mWidth,diff.x, true);
            }else {
                handleTracking(currentPoint,mHeight,diff.y,false);
            }
        }else{
            if ((mOrientation % 2 == 1)){
                handleTracking(currentPoint,mWidth,diff.x, true);
            }else {
                handleTracking(currentPoint,mHeight,diff.x,true);
            }
        }
    }

    private PointF getDesiredPoint() {

        if (isOrientationLocked) {
            if (mOrientation % 2 == 1) {
                if (positionType == PositionType.LEFT) {
                    return new PointF(mWidth * 3 / 4.f, 0);
                } else if (positionType == PositionType.RIGHT) {
                    return new PointF(mWidth / 4.f, 0);
                } else {
                    return new PointF(mWidth / 2.f, 0);
                }
            } else {
                if (positionType == PositionType.LEFT) {
                    return new PointF(0, mHeight * 3 / 4.f);
                } else if (positionType == PositionType.RIGHT) {
                    return new PointF(0, mHeight / 4.f);
                } else {
                    return new PointF(0, mHeight / 2.f);
                }
            }
        }else{ // Orientation not locked
            if (mOrientation % 2 == 1) {
                if (positionType == PositionType.LEFT) {
                    return new PointF(mWidth * 3 / 4.f, 0);
                } else if (positionType == PositionType.RIGHT) {
                    return new PointF(mWidth / 4.f, 0);
                } else {
                    return new PointF(mWidth / 2.f, 0);
                }
            } else {
                if (positionType == PositionType.LEFT) {
                    return new PointF(mHeight * 3 / 4.f, 0);
                } else if (positionType == PositionType.RIGHT) {
                    return new PointF(mHeight / 4.f, 0);
                } else {
                    return new PointF(mHeight / 2.f, 0);
                }
            }
        }
    }


    private void handleTracking(PointF currentPoint, int dimension, float coordinate, boolean isPortrait){
        final float scaleFactor = (dimension / 2.f) / mPivoRotatorConfig.scaleFactor;
        final float pdPixel;
        if (isPortrait){
            pdPixel = Math.abs( mPivoRotatorConfig.getKP()* (currentPoint.x - getDesiredPoint().x)
                    + mPivoRotatorConfig.getKD() * (currentPoint.x - lastPoint.x));
        }else{
            pdPixel = Math.abs( mPivoRotatorConfig.getKP()* (currentPoint.y - getDesiredPoint().y)
                    + mPivoRotatorConfig.getKD() * (currentPoint.y - lastPoint.y));
        }

        final float pd_angle = pdPixel / scaleFactor;

        float PD_ANGLE_THRESHOLD = 2.5f;
        if (pd_angle< PD_ANGLE_THRESHOLD || Math.abs(coordinate)<= mPivoRotatorConfig.MARGIN * dimension){
            mController.turnHold();
            return;
        }

        int s;
        if (mController.getVersion().getVersion()==0){
            s = SpeedConstraint.getInstance().constraint(pd_angle).getValue();
        }else {
            s = 2 * SpeedConstraint.getInstance().constraintAngle(pd_angle);
        }

        lastPoint = currentPoint;
        try{
            if (coordinate <0){
                if (isFront){
                    mController.turnLeftContinuously(s);
                }else{
                    mController.turnRightContinuously(s);
                }
            }else {
                if (isFront){
                    mController.turnRightContinuously(s);
                }else{
                    mController.turnLeftContinuously(s);
                }
            }
        }
        catch (UnsupportedSpeedException e) {
            e.printStackTrace();
        }
    }
}
