package app.pivo.android.prosdk.util;

import android.graphics.Rect;
import android.media.Image;

import app.pivo.android.prosdk.PivoSensitivity;
import app.pivo.android.prosdk.exceptions.*;
import app.pivo.android.prosdk.tracking.FrameMetadata;

/**
 * PivoSDK interface
 */
public interface IPivoSdk {

    /**
     * This function is used to put license content. Please get license content and pass as parameter to this function without modifying it
     *
     * @param licenseContent - license file
     * @throws PackageNameException this exception is thrown if the license is expired
     * @throws LicenseDateException this exception is thrown if the package name in the package name of the application and license files are different
     */
    void unlockWithLicenseKey(String licenseContent) throws PackageNameException, LicenseDateException, LicenseNotProvidedException, InvalidLicenseException, SdkPlanTypeException;

    /**
     * This function is used to scan for Pivo devices
     */
    void scan();

    /**
     * This function is used to connect to the device.
     *
     * @param pivoDevice is the device, you want to connect to
     */
    void connectTo(PivoDevice pivoDevice);

    /**
     * Call this function to stop scanning.
     */
    void stopScan();

    /**
     * This function is used to set default speed with seconds per round for Pivo. You need to send Pivo supported speed, to get
     * Pivo supported speeds call getSupportedSpeedsInSecondsPerRound() function.
     *
     * @param speedInSecondsPerRound
     */
    void setSpeedBySecondsPerRound(int speedInSecondsPerRound);

    void setSpeedBySecondsPerRoundRemote(int speedInSecondsPerRound);

    /**
     * This function is used to get all supported speeds for application.
     *
     * @return returns an array of speeds(unit is seconds per round)
     */
    int[] getSupportedSpeedsInSecondsPerRound();

    /**
     * This function is used to get all supported speeds using remote controller.
     *
     * @return returns array of speeds(unit is seconds per round).
     */
    int[] getSupportedSpeedsByRemoteInSecondsPerRound();



    /**
     * This function is called to disconnect from the Pivo.
     */
    void disconnect();

    /**
     * Turn Pivo to left at the current speed.
     *
     * @param angle is angle to rotate
     */
    void turnLeft(int angle);

    /**
     * This function is used to turn Pivo to left.
     *
     * @param angle is angle to rotate
     * @param speed is specified speed
     */
    void turnLeft(int angle, int speed) throws UnsupportedSpeedException;

    /**
     * This function is used to turn Pivo to right.
     *
     * @param angle is angle to rotate
     */
    void turnRight(int angle);

    /**
     * This function is used to turn Pivo to right.
     *
     * @param angle is angle to rotate
     * @param speed is specified speed
     */
    void turnRight(int angle, int speed) throws UnsupportedSpeedException;

    /**
     * This function is used to rotate continuously to right with the current speed.
     */
    void turnRightContinuously();

    /**
     * This function is used to rotate continuously to right with the specified speed.
     *
     * @param speed is Pivo speed
     */
    void turnRightContinuously(int speed) throws UnsupportedSpeedException;

    /**
     * This function is used to rotate continuously to left with the current speed.
     */
    void turnLeftContinuously();

    /**
     * This function is used to rotate continuously to left with the specified speed.
     *
     * @param speed is Pivo speed
     */
    void turnLeftContinuously(int speed) throws UnsupportedSpeedException;

    void turnLeftRemote(int angle, int speed) throws UnsupportedSpeedException;
    void turnRightRemote(int angle, int speed) throws UnsupportedSpeedException;

    /**
     * This function is used to request Pivo battery level.
     */
    void requestBatteryLevel();

    /**
     * This function is used to stop Pivo rotation
     */
    void stop();

    /**
     * This function is used to change Pivo name.
     *
     * @param name is desired name
     */
    void changeName(String name);

    /**
     * This function is used to check connectivity of Pivo to the app.
     *
     * @return is boolean, if it's connected returns true, otherwise it returns false.
     */
    boolean isPivoConnected();

    /**
     * This method is deprecated starting from version 1.0.1-beta. Please use getVersion instead of this method.
     * This function is used to get Pivo version.
     *
     * @return version of Pivo
     */
    @Deprecated
    String getPivoVersion();

    /**
     * This function returns all supported tracking modes by Pivo
     *
     * @return an array of supported tracking methods.
     */
    String[] getSupportedTrackingModes();

    /**
     * This function is used to start Action Tracking.
     *
     * @param metadata is a frame metadata which contains orientation, rotation, width, height, cameraFacing.
     * @param region   is region of interest. The minimum supported width and height should be bigger than 100.
     * @param image    is image data which is acquired from camera.
     * @param listener is a callback of action tracking.
     */
    void startActionTracking(FrameMetadata metadata, Rect region, Image image, PivoSensitivity sensitivity, ITrackingListener listener);

    /**
     * This function is used to start Action Tracking.
     *
     * @param metadata  is a frame metadata which contains orientation, rotation, width, height, cameraFacing.
     * @param region    is region of interest. The minimum supported width and height should be bigger than 100.
     * @param byteArray is NV21 image byteArray.
     * @param listener  is a callback of action tracking.
     */
    void startActionTracking(FrameMetadata metadata, Rect region, byte[] byteArray, PivoSensitivity sensitivity, ITrackingListener listener);

    /**
     * This function is used to start human tracking.
     *
     * @param metadata    metadata is a frame metadata which contains orientation, rotation, width, height, cameraFacing.
     * @param image       is image data which is acquired from camera.
     * @param sensitivity is enum set to control Pivo movement.
     * @param listener    is a callback of person tracking.
     */
    void starPersonTracking(FrameMetadata metadata, Image image, PivoSensitivity sensitivity, ITrackingListener listener);

    /**
     * This function is used to start human tracking.
     *
     * @param metadata    metadata is a frame metadata which contains orientation, rotation, width, height, cameraFacing.
     * @param byteArray   is NV21 image data.
     * @param sensitivity is enum set to control Pivo movement.
     * @param listener    is a callback of person tracking.
     */
    void starPersonTracking(FrameMetadata metadata, byte[] byteArray, PivoSensitivity sensitivity, ITrackingListener listener);

    /**
     * This function is used to start horse tracking.
     *
     * @param metadata    is a frame metadata which contains orientation, rotation, width, height, cameraFacing.
     * @param image       is image data which is acquired from camera.
     * @param sensitivity is enum set to control Pivo movement.
     * @param listener    is a callback of horse tracking.
     */
    void startHorseTracking(FrameMetadata metadata, Image image, PivoSensitivity sensitivity, ITrackingListener listener);

    /**
     * This function is used to start horse tracking.
     *
     * @param metadata    is a frame metadata which contains orientation, rotation, width, height, cameraFacing.
     * @param byteArray   is NV21 byte array
     * @param sensitivity is enum set to control Pivo movement.
     * @param listener    is a callback of horse tracking.
     */
    void startHorseTracking(FrameMetadata metadata, byte[] byteArray, PivoSensitivity sensitivity, ITrackingListener listener);

    /**
     * This function is used to update frames for processing. Please make sure calling startPersonTracking, startActionTracking, or startHorseTracking,
     * and call updateTrackingFrame function.
     *
     * @param image    is image data which is acquired from camera.
     * @param metadata is a frame metadata which contains orientation, rotation, width, height, cameraFacing.
     */
    void updateTrackingFrame(Image image, FrameMetadata metadata);

    /**
     * This function is used to update frames for processing. Please make sure calling startPersonTracking, startActionTracking, or startHorseTracking,
     * and call updateTrackingFrame function.
     *
     * @param byteArray is an NV21 image byte array.
     * @param metadata  is a frame metadata which contains orientation, rotation, width, height, cameraFacing.
     */
    void updateTrackingFrame(byte[] byteArray, FrameMetadata metadata);

    /**
     * This function is used to stop tracking.
     */
    void stopTracking();

    /**
     * This method is used to get Pivo {@link Version} and type.
     *
     * @return pivo version
     */
    Version getVersion();

    /**
     * Returns device info {@link PivoDeviceInfo}
     *
     * @return device info
     */
    PivoDeviceInfo getDeviceInfo();

    void enableBypass(boolean activate);
}