package app.pivo.android.prosdk;

import app.pivo.android.basicsdk.util.Version;
import app.pivo.android.podsdk.model.PivoDeviceCategory;
import app.pivo.android.podsdk.model.PodModel;

class PivoRotatorConfig {

    final float MARGIN = 0.05f; // 5% of Preview width (it can be adjust to increase the sensitivity later)
    private float KP; //Increase the responsiveness
    private float KD; //Prevent the overshooting

    private double mZoom;

    float minSpeed;
    float maxSpeed;

    float maxDegree;

    Version version = new Version(
            0,
            new PivoDeviceCategory.PivoPod(0, PodModel.RED)
    );

    private int lowerIdx, upperIdx;
    float scaleFactor;

    private static PivoRotatorConfig instance;

    private final float [] angularSpeeds = {36, 18, 12, 7.2f, 6, 3, 1.2f, 0.6f}; // For version 0 of Pivo
    private final Speed[] mSpeeds = {
            Speed.SPEED_1,
            Speed.SPEED_2,
            Speed.SPEED_3,
            Speed.SPEED_4,
            Speed.SPEED_5,
            Speed.SPEED_6,
            Speed.SPEED_7,
            Speed.SPEED_8
    };


    private SensitivityMode sensitivityMode;

    void initConfig(double zoom, SensitivityMode mode, Version version){
        sensitivityMode = mode;
        mZoom = zoom;
        this.version = version;

        if(version.getVersion() == 0) {
            setSpeedBoundariesV0(); // For sensitivity modes speed boundaries for V0

            KD = mode.getKDValue();
            KP = mode.getKPValue();
            lowerIdx = mode.getLowerIndex();
            upperIdx = mode.getUpperIndex();

            minSpeed = angularSpeeds[upperIdx];
            maxSpeed = angularSpeeds[lowerIdx];
            scaleFactor = maxSpeed - minSpeed;
        }
        else if(version.isAboveV0()) {
            setSpeedBoundariesV1(); // For sensitivity modes speed boundaries for V1

            KD = mode.getKDValue();
            KP = mode.getKPValue();

            maxSpeed = mode.getV1MaxHalfSpeed();
            maxDegree = 180.0f / maxSpeed;
            scaleFactor = maxDegree - (180.0f/mode.getV1MinHalfSpeed()); // (90 - 3)degPerSec // 360/4 - 360/120
        }
    }

    public static PivoRotatorConfig getInstance () {
        if (instance == null) {
            instance = new PivoRotatorConfig();
            return instance;
        }
        return instance;
    }


    /***Getters***/
    SensitivityMode getSensitivityMode(){
        return sensitivityMode;
    }

    int getLowerIdx() {
        return lowerIdx;
    }

    int getUpperIdx() {
        return upperIdx;
    }

    int getMaxSpeedIndex() {
        return lowerIdx;
    } // Same with getLowerIdx but different Naming

    int getMinSpeedIndex() {
        return upperIdx;
    }

    Speed[] getSpeeds() {
        return mSpeeds;
    }

    float[] getAngularSpeeds() {
        return angularSpeeds;
    }

    float getKD() {
        return KD;
    }

    float getKP() {
        return KP;
    }

    public double getZoom (){
        return mZoom;
    }

    public Version getVersion() {
        return version;
    }

    /***Setters***/
    void setKD(float KD) {
        this.KD = KD;
    }

    void setKP(float KP) {
        this.KP = KP;
    }



    /*******/

    private void setSpeedBoundariesV0() {
        if (mZoom < 2.0) {
            SensitivityMode.FAST.setLowerIndex(0);
            SensitivityMode.FAST.setUpperIndex(4);
            SensitivityMode.FAST.setKPValue(2.8f);
            SensitivityMode.FAST.setKDValue(0.5f);

            SensitivityMode.NORMAL.setLowerIndex(0);
            SensitivityMode.NORMAL.setUpperIndex(5);
            SensitivityMode.NORMAL.setKPValue(1.8f);
            SensitivityMode.NORMAL.setKDValue(0.4f);

            SensitivityMode.SLOW.setLowerIndex(1);
            SensitivityMode.SLOW.setUpperIndex(6);
            SensitivityMode.SLOW.setKPValue(1.0f);
            SensitivityMode.SLOW.setKDValue(0.4f);
        }
        else if (mZoom < 3.0) {
            SensitivityMode.FAST.setLowerIndex(1);
            SensitivityMode.FAST.setUpperIndex(5);
            SensitivityMode.FAST.setKPValue(2.6f);
            SensitivityMode.FAST.setKDValue(0.5f);

            SensitivityMode.NORMAL.setLowerIndex(1);
            SensitivityMode.NORMAL.setUpperIndex(6);
            SensitivityMode.NORMAL.setKPValue(1.8f);
            SensitivityMode.NORMAL.setKDValue(0.4f);

            SensitivityMode.SLOW.setLowerIndex(1);
            SensitivityMode.SLOW.setUpperIndex(7);
            SensitivityMode.SLOW.setKPValue(1.0f);
            SensitivityMode.SLOW.setKDValue(0.4f);
        }
        else {
            SensitivityMode.FAST.setLowerIndex(1);
            SensitivityMode.FAST.setUpperIndex(6);
            SensitivityMode.FAST.setKPValue(2.4f);
            SensitivityMode.FAST.setKDValue(0.6f);

            SensitivityMode.NORMAL.setLowerIndex(1);
            SensitivityMode.NORMAL.setUpperIndex(6);
            SensitivityMode.NORMAL.setKPValue(1.8f);
            SensitivityMode.NORMAL.setKDValue(0.5f);

            SensitivityMode.SLOW.setLowerIndex(2);
            SensitivityMode.SLOW.setUpperIndex(7);
            SensitivityMode.SLOW.setKPValue(1.0f);
            SensitivityMode.SLOW.setKDValue(0.5f);
        }
    }

    private void setSpeedBoundariesV1() {
        if (mZoom < 2.0) {
            SensitivityMode.FAST.setKPValue(2.8f);
            SensitivityMode.FAST.setKDValue(0.4f);
            SensitivityMode.FAST.setV1MaxHalfSpeed(5); // 10 sec/rnd --> 36 deg/sec
            SensitivityMode.FAST.setV1MinHalfSpeed(40);

            SensitivityMode.NORMAL.setKPValue(1.8f);
            SensitivityMode.NORMAL.setKDValue(0.4f);
            SensitivityMode.NORMAL.setV1MaxHalfSpeed(5);
            SensitivityMode.NORMAL.setV1MinHalfSpeed(60);

            SensitivityMode.SLOW.setKPValue(1.0f);
            SensitivityMode.SLOW.setKDValue(0.4f);
            SensitivityMode.SLOW.setV1MaxHalfSpeed(10); // 20 sec/rnd --> 18 deg/sec
            SensitivityMode.SLOW.setV1MinHalfSpeed(140);
        }
        else if (mZoom < 3.0) {
            SensitivityMode.FAST.setKPValue(2.6f);
            SensitivityMode.FAST.setKDValue(0.4f);
            SensitivityMode.FAST.setV1MaxHalfSpeed(8);
            SensitivityMode.FAST.setV1MinHalfSpeed(80);

            SensitivityMode.NORMAL.setKPValue(1.8f);
            SensitivityMode.NORMAL.setKDValue(0.4f);
            SensitivityMode.NORMAL.setV1MaxHalfSpeed(8);
            SensitivityMode.NORMAL.setV1MinHalfSpeed(90);

            SensitivityMode.SLOW.setKPValue(1.0f);
            SensitivityMode.SLOW.setKDValue(0.4f);
            SensitivityMode.SLOW.setV1MaxHalfSpeed(12);
            SensitivityMode.SLOW.setV1MinHalfSpeed(170);
        }
        else {
            SensitivityMode.FAST.setKPValue(2.4f);
            SensitivityMode.FAST.setKDValue(0.5f);
            SensitivityMode.FAST.setV1MaxHalfSpeed(10);
            SensitivityMode.FAST.setV1MinHalfSpeed(120);

            SensitivityMode.NORMAL.setKPValue(1.8f);
            SensitivityMode.NORMAL.setKDValue(0.5f);
            SensitivityMode.NORMAL.setV1MaxHalfSpeed(11);
            SensitivityMode.NORMAL.setV1MinHalfSpeed(240);

            SensitivityMode.SLOW.setKPValue(1.0f);
            SensitivityMode.SLOW.setKDValue(0.5f);
            SensitivityMode.SLOW.setV1MaxHalfSpeed(14);
            SensitivityMode.SLOW.setV1MinHalfSpeed(200);
        }
    }

    private void setSpeedBoundariesV2() {
        if (mZoom < 2.0) {
            SensitivityMode.FAST.setKPValue(2.0f);
            SensitivityMode.FAST.setKDValue(0.6f);
            SensitivityMode.FAST.setV1MaxHalfSpeed(3); // 4 sec/rnd --> 90 deg/sec
            SensitivityMode.FAST.setV1MinHalfSpeed(30);

            SensitivityMode.NORMAL.setKPValue(1.8f);
            SensitivityMode.NORMAL.setKDValue(0.4f);
            SensitivityMode.NORMAL.setV1MaxHalfSpeed(5);
            SensitivityMode.NORMAL.setV1MinHalfSpeed(60);

            SensitivityMode.SLOW.setKPValue(1.0f);
            SensitivityMode.SLOW.setKDValue(0.4f);
            SensitivityMode.SLOW.setV1MaxHalfSpeed(10); // 20 sec/rnd --> 18 deg/sec
            SensitivityMode.SLOW.setV1MinHalfSpeed(140);
        }
        else if (mZoom < 3.0) {
            SensitivityMode.FAST.setKPValue(1.9f);
            SensitivityMode.FAST.setKDValue(0.7f);
            SensitivityMode.FAST.setV1MaxHalfSpeed(5);
            SensitivityMode.FAST.setV1MinHalfSpeed(50);

            SensitivityMode.NORMAL.setKPValue(1.8f);
            SensitivityMode.NORMAL.setKDValue(0.4f);
            SensitivityMode.NORMAL.setV1MaxHalfSpeed(8);
            SensitivityMode.NORMAL.setV1MinHalfSpeed(90);

            SensitivityMode.SLOW.setKPValue(1.0f);
            SensitivityMode.SLOW.setKDValue(0.4f);
            SensitivityMode.SLOW.setV1MaxHalfSpeed(12);
            SensitivityMode.SLOW.setV1MinHalfSpeed(170);
        }
        else {
            SensitivityMode.FAST.setKPValue(1.6f);
            SensitivityMode.FAST.setKDValue(0.8f);
            SensitivityMode.FAST.setV1MaxHalfSpeed(6);
            SensitivityMode.FAST.setV1MinHalfSpeed(70);

            SensitivityMode.NORMAL.setKPValue(1.8f);
            SensitivityMode.NORMAL.setKDValue(0.5f);
            SensitivityMode.NORMAL.setV1MaxHalfSpeed(11);
            SensitivityMode.NORMAL.setV1MinHalfSpeed(240);

            SensitivityMode.SLOW.setKPValue(1.0f);
            SensitivityMode.SLOW.setKDValue(0.5f);
            SensitivityMode.SLOW.setV1MaxHalfSpeed(14);
            SensitivityMode.SLOW.setV1MinHalfSpeed(200);
        }
    }

    enum Speed {
        SPEED_1(10),
        SPEED_2(20),
        SPEED_3(30),
        SPEED_4(50),
        SPEED_5(60),
        SPEED_6(120),
        SPEED_7(300),
        SPEED_8(600),
        SPEED_9(1200),
        SPEED_10(1800),
        SPEED_11(3600),
        //the following speed list is supported by V1
        SPEED_12(7200),
        SPEED_13(10800),
        SPEED_14(21600),
        SPEED_15(43200),
        SPEED_16(64800),
        SPEED_17(86400);

        Speed(int val) {
            value = val;
        }

        int value;

        int getValue() {
            return value;
        }

        static Speed speed(int value) {
            switch (value) {
                case 1:
                    return SPEED_1;
                case 2:
                    return SPEED_2;
                case 3:
                    return SPEED_3;
                case 4:
                    return SPEED_4;
                case 5:
                    return SPEED_5;
                case 6:
                    return SPEED_6;
                case 7:
                    return SPEED_7;
                case 8:
                    return SPEED_8;
                case 9:
                    return SPEED_9;
                case 10:
                    return SPEED_10;
                case 11:
                    return SPEED_11;
                case 12:
                    return SPEED_12;
                case 13:
                    return SPEED_13;
                case 14:
                    return SPEED_14;
                case 15:
                    return SPEED_15;
                case 16:
                    return SPEED_16;
                case 17:
                    return SPEED_17;
            }
            return SPEED_1;
        }

        static Speed speed(byte data) {
            switch (data) {
                case 0x11:
                    return SPEED_1;
                case 0x12:
                    return SPEED_2;
                case 0x13:
                    return SPEED_3;
                case 0x14:
                    return SPEED_4;
                case 0x15:
                    return SPEED_5;
                case 0x16:
                    return SPEED_6;
                case 0x17:
                    return SPEED_7;
                case 0x18:
                    return SPEED_8;
                case 0x19:
                    return SPEED_9;
                case 0x1A:
                    return SPEED_10;
                case 0x1B:
                    return SPEED_11;
                case 0x1C:
                    return SPEED_12;
                case 0x1D:
                    return SPEED_13;
                case 0x1E:
                    return SPEED_14;
                case 0x1F:
                    return SPEED_15;
                case 0x20:
                    return SPEED_16;
                case 0x21:
                    return SPEED_17;
            }
            return SPEED_1;
        }
    }
}
