package app.valuationcontrol.multimodule.library.entities;

import java.security.Principal;

import lombok.Getter;
import lombok.extern.log4j.Log4j2;
import org.springframework.context.ApplicationEvent;
import org.springframework.core.ResolvableType;
import org.springframework.core.ResolvableTypeProvider;

@Log4j2
public class Event<T> extends ApplicationEvent implements ResolvableTypeProvider {

  private final T old;
  private final T updated;
  @Getter private final Principal principal;
  @Getter private final Model model;
  private final Class<?> parameterClass;

  public Event(
      Object source, T old, T updated, Principal principal, Class<?> parameterClass, Model model) {
    super(source);
    this.old = old;
    this.updated = updated;
    this.principal = principal;
    this.parameterClass = parameterClass;
    this.model = model;
  }

  public T getOldVersion() {
    return old;
  }

  public T getNewVersion() {
    return updated;
  }

  @Override
  public ResolvableType getResolvableType() {
    return ResolvableType.forClassWithGenerics(getClass(), parameterClass);
  }

  public enum EVENT_TYPE {
    CREATED,
    DELETED,
    UPDATED,
    METADATA_UPDATE
  }

  public String getParameterClass() {
    return this.parameterClass.getSimpleName();
  }

  public EVENT_TYPE getType() {
    if (isCreate()) {
      return EVENT_TYPE.CREATED;
    } else if (isDelete()) {
      return EVENT_TYPE.DELETED;
    } else if (isMetadataUpdate()) {
      return EVENT_TYPE.METADATA_UPDATE;
    }
    return EVENT_TYPE.UPDATED;
  }

  public T effectedEntity() {
    return this.isCreate() || this.isUpdate() ? this.getNewVersion() : this.getOldVersion();
  }

  public boolean isCreate() {
    return old == null && updated != null;
  }

  public boolean isUpdate() {
    return old != null && updated != null;
  }

  public boolean isMetadataUpdate() {
    return old != null && old.equals(updated);
  }

  public boolean isDelete() {
    return old != null && updated == null;
  }

  public static <M> Event<M> created(
      Object source, M created, Principal principal, Class<?> parameterClass, Model model) {
    return new Event<>(source, null, created, principal, parameterClass, model);
  }

  public static <M> Event<M> updated(
      Object source, M old, M updated, Principal principal, Class<?> parameterClass, Model model) {
    return new Event<>(source, old, updated, principal, parameterClass, model);
  }

  public static <M> Event<M> lightUpdated(
      Object source, M updated, Principal principal, Class<?> parameterClass, Model model) {
    return new Event<>(source, updated, updated, principal, parameterClass, model);
  }

  public static <M> Event<M> deleted(
      Object source, M old, Principal principal, Class<?> parameterClass, Model model) {
    return new Event<>(source, old, null, principal, parameterClass, model);
  }
}
