package app.valuationcontrol.multimodule.library.xlhandler;

import app.valuationcontrol.multimodule.library.entities.Model;
import app.valuationcontrol.multimodule.library.entities.Sensitivity;
import app.valuationcontrol.multimodule.library.entities.SensitivityResult;
import app.valuationcontrol.multimodule.library.entities.Variable;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import lombok.extern.log4j.Log4j2;

@Log4j2
public record SensitivityRunner(
        CalcDocument calcDocument, List<Sensitivity> sensitivities, SCENARIO scenario, Model model) {

  static final String SENSITIVITY_SHEET =
      POICalcDocument.getScenarioSheetName(SCENARIO.SENSITIVITY);

  public List<SensitivityResult> runSensitivities() {

    List<SensitivityResult> sensitivityResults = new ArrayList<>();
    for (Sensitivity sensitivity : sensitivities) {
      log.info(
          "Running sentivity " + sensitivity.getSensitivityName() + "for scenario  " + scenario);

      Optional<Variable> variable1 =
          model.getVariableWithID(sensitivity.getSensitivityVariable1Id());
      Optional<Variable> variable2 =
          model.getVariableWithID(sensitivity.getSensitivityVariable2Id());
      Optional<Variable> measurementVariable =
          model.getVariableWithID(sensitivity.getSensitivityMeasurementVariableId());

      if (variable1.isEmpty() || variable2.isEmpty() || measurementVariable.isEmpty()) {
        throw new IllegalArgumentException("One of the variables is not set");
      }

      double[] originalVariable1Value =
          new double
              [variable1.get().isModelledAtSegment() ? Math.max(1, model.getSegments().size()) : 1];
      double[] originalVariable2Value =
          new double
              [variable2.get().isModelledAtSegment() ? Math.max(1, model.getSegments().size()) : 1];
      double[] originalMeasurementValue =
          new double
              [measurementVariable.get().isModelledAtSegment()
                  ? Math.max(
                      1,
                      model.getSegments().size()
                          + 1) // +1 to include the measurement at aggregated level
                  : 1];

      int column = 0;
      int variable1Offset;
      int variable2Offset;

      GenericSheet sensitivityGenericSheet;
      try {
        GenericSheet scenarioGenericSheet = calcDocument.getSheet(scenario);
        sensitivityGenericSheet = calcDocument.copy(scenarioGenericSheet, SENSITIVITY_SHEET);

        int numberOfMeasurementSegments =
            measurementVariable.get().isModelledAtSegment()
                ? Math.max(1, model.getSegments().size() + 1)
                : 1;

        // Initiating a sensitivity result object
        SensitivityResult sensitivityResult =
            new SensitivityResult(
                sensitivity.getId(),
                scenario.ordinal(),
                sensitivity.getSensitivityVariable1Steps(),
                sensitivity.getSensitivityVariable2Steps(),
                numberOfMeasurementSegments);

        if (variable1.get().isModelledAtSegment() && !model.getSegments().isEmpty()) {
          for (int k = 0; k < model.getSegments().size(); k++) {
            originalVariable1Value[k] =
                getValueForVariableAtPeriod(
                    variable1.get(),
                    variable1
                        .get()
                        .columnOfSegmentAndPeriod(k, sensitivity.getSensitivityVariable1Period()),
                    sensitivityGenericSheet,
                    false);
          }
        } else {
          originalVariable1Value[0] =
              getValueForVariableAtPeriod(
                  variable1.get(),
                  variable1
                      .get()
                      .columnOfSegmentAndPeriod(-1, sensitivity.getSensitivityVariable1Period()),
                  sensitivityGenericSheet,
                  false);
        }

        // Storing initial values for variable2
        if (variable2.get().isModelledAtSegment() && !model.getSegments().isEmpty()) {
          for (int k = 0; k < model.getSegments().size(); k++) {
            originalVariable2Value[k] =
                getValueForVariableAtPeriod(
                    variable2.get(),
                    variable2
                        .get()
                        .columnOfSegmentAndPeriod(k, sensitivity.getSensitivityVariable2Period()),
                    sensitivityGenericSheet,
                    false);
          }
        } else {
          originalVariable2Value[0] =
              getValueForVariableAtPeriod(
                  variable2.get(),
                  variable2
                      .get()
                      .columnOfSegmentAndPeriod(-1, sensitivity.getSensitivityVariable2Period()),
                  sensitivityGenericSheet,
                  false);
        }

        if (measurementVariable.get().isModelledAtSegment() && !model.getSegments().isEmpty()) {
          for (int k = 0;
              k <= model.getSegments().size();
              k++) { // Setting limiter to <= since we want to include the aggregated value as
            // well
            int segmentIndex = k - 1;
            originalMeasurementValue[k] =
                getValueForVariableAtPeriod(
                    measurementVariable.get(),
                    measurementVariable
                        .get()
                        .columnOfSegmentAndPeriod(
                            segmentIndex, sensitivity.getSensitivityMeasurementVariablePeriod()),
                    sensitivityGenericSheet,
                    false);
          }
        } else {
          originalMeasurementValue[0] =
              getValueForVariableAtPeriod(
                  measurementVariable.get(),
                  measurementVariable
                      .get()
                      .columnOfSegmentAndPeriod(
                          -1, sensitivity.getSensitivityMeasurementVariablePeriod()),
                  sensitivityGenericSheet,
                  false);
        }

        variable1Offset = sensitivity.getSensitivityVariable1Steps() / 2;
        variable2Offset = sensitivity.getSensitivityVariable2Steps() / 2;

        for (int h = 0; h < numberOfMeasurementSegments; h++) {

          log.debug(
              "Original value: {} from column: {} from row: {} ",
              originalMeasurementValue[h],
              column,
              measurementVariable.get().getRow());

          for (int i = 0; i < sensitivity.getSensitivityVariable1Steps(); i++) {

            // If modelled at segment we only show the relative offset and not the actual
            // original value
            double variable1Value =
                (i - variable1Offset) * sensitivity.getSensitivityVariable1StepSize();
            if (!variable1.get().isModelledAtSegment() || model.getSegments().isEmpty()) {
              variable1Value =
                  originalVariable1Value[0]
                      + (i - variable1Offset) * sensitivity.getSensitivityVariable1StepSize();
            }
            sensitivityResult.setVariable1Values(i, variable1Value);

            for (int j = 0; j < sensitivity.getSensitivityVariable2Steps(); j++) {

              List<GenericSheet.CellValue> cellsToBeUpdated = new ArrayList<>();

              double variable2Value =
                  (j - variable2Offset) * sensitivity.getSensitivityVariable2StepSize();
              if (!variable2.get().isModelledAtSegment() || model.getSegments().isEmpty()) {
                variable2Value =
                    originalVariable2Value[0]
                        + (j - variable2Offset) * sensitivity.getSensitivityVariable2StepSize();
              }

              if (i == 0) {
                sensitivityResult.setVariable2Values(j, variable2Value);
              }

              if (variable1.get().isModelledAtSegment() && !model.getSegments().isEmpty()) {
                for (int k = 0; k < model.getSegments().size(); k++) {
                  variable1Value =
                      originalVariable1Value[k]
                          + (i - variable1Offset) * sensitivity.getSensitivityVariable1StepSize();
                  cellsToBeUpdated.add(
                      new GenericSheet.CellValue(
                          variable1Value,
                          variable1
                              .get()
                              .columnOfSegmentAndPeriod(
                                  k, sensitivity.getSensitivityVariable1Period()),
                          variable1.get().getRow()));
                }
              } else {
                cellsToBeUpdated.add(
                    new GenericSheet.CellValue(
                        variable1Value,
                        variable1
                            .get()
                            .columnOfSegmentAndPeriod(
                                -1, sensitivity.getSensitivityVariable1Period()),
                        variable1.get().getRow()));
              }

              if (variable2.get().isModelledAtSegment() && !model.getSegments().isEmpty()) {
                for (int k = 0; k < model.getSegments().size(); k++) {
                  variable2Value =
                      originalVariable2Value[k]
                          + (j - variable2Offset) * sensitivity.getSensitivityVariable2StepSize();
                  cellsToBeUpdated.add(
                      new GenericSheet.CellValue(
                          variable2Value,
                          variable2
                              .get()
                              .columnOfSegmentAndPeriod(
                                  k, sensitivity.getSensitivityVariable2Period()),
                          variable2.get().getRow()));
                }
              } else {
                cellsToBeUpdated.add(
                    new GenericSheet.CellValue(
                        variable2Value,
                        variable2
                            .get()
                            .columnOfSegmentAndPeriod(
                                -1, sensitivity.getSensitivityVariable2Period()),
                        variable2.get().getRow()));
              }

              GenericSheet.CellValue[] cellsToBeUpdatedArray =
                  new GenericSheet.CellValue[cellsToBeUpdated.size()];
              cellsToBeUpdatedArray = cellsToBeUpdated.toArray(cellsToBeUpdatedArray);

              sensitivityGenericSheet.setCellValues(cellsToBeUpdatedArray);
              sensitivityGenericSheet.computeCellValues(cellsToBeUpdatedArray);

              final double measuredValue =
                  getValueForVariableAtPeriod(
                      measurementVariable.get(),
                      measurementVariable
                          .get()
                          .columnOfSegmentAndPeriod(
                              h - 1, sensitivity.getSensitivityMeasurementVariablePeriod()),
                      sensitivityGenericSheet,
                      true);
              sensitivityResult.setMyResultArray(h, i, j, measuredValue);
            }
          }
        }

        // If all steps are completed, we validate the sensitivityResult
        sensitivityResult.setValid(true);

        sensitivityResults.add(sensitivityResult);
        calcDocument.remove(sensitivityGenericSheet);

      } catch (Exception e) {
        log.error("Error when running sensitivity", e);
      }
    }
    return sensitivityResults;
  }

  private double getValueForVariableAtPeriod(
      Variable variable, int column, GenericSheet genericSheet, boolean computeCell) {
    if (computeCell) genericSheet.computeCell(column, variable.getRow());
    return genericSheet.getValueWithoutRecalculation(column, variable.getRow());
  }
}
