package app.valuationcontrol.multimodule.library.xlhandler;

import app.valuationcontrol.multimodule.library.entities.Model;
import app.valuationcontrol.multimodule.library.helpers.poiudf.DateDif;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import lombok.extern.log4j.Log4j2;
import org.apache.poi.ss.formula.WorkbookEvaluator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

/** Manager of xl connections Will keep a map of modelids to its XLInstance and */
@Log4j2
@Component
public class XLHandleManager {

  private final Map<Long, XLInstance> xlInstanceByModelId = new ConcurrentHashMap<>();
  private final ModelChangeNotifier modelChangeNotifier;
  private final Duration inactivityThreshold;
  private final CalcDocumentFactoryImpl calcDocumentFactory;

  private boolean isSystemShuttingDown;

  @Autowired
  public XLHandleManager(
      ModelChangeNotifier modelChangeNotifier,
      @Value("${xlinstance.unused.threshold}") int secondsOfInactivityAllowed,
      CalcDocumentFactoryImpl calcDocumentFactory) {
    this.modelChangeNotifier = modelChangeNotifier;
    this.inactivityThreshold = Duration.ofSeconds(secondsOfInactivityAllowed);
    this.calcDocumentFactory = calcDocumentFactory;

    log.info("Registering additional excel function");
    WorkbookEvaluator.registerFunction("DATEDIF", new DateDif());

    Runtime.getRuntime().addShutdownHook(new Thread(() -> this.isSystemShuttingDown = true));
  }

  @Scheduled(fixedRate = 30, timeUnit = TimeUnit.SECONDS)
  private synchronized void closeUnusedDocuments() {

    xlInstanceByModelId.forEach(
        (modelId, xlInstance) -> {
          if (xlInstance.isUnusedFor(inactivityThreshold)
              && modelChangeNotifier.fetchScenariosFromUserRegistry(modelId).isEmpty()) {
            log.debug("Closing model {}", modelId);
            xlInstance.sleep();
            xlInstanceByModelId.remove(modelId);
          }
        });
  }

  protected XLInstance createModelInstance(Model model) {
    return new XLInstance(model, calcDocumentFactory.getDocument(), modelChangeNotifier);
  }

  public synchronized ScenarioDataProvider getXLInstanceForModel(Model model) {
    if (isSystemShuttingDown) {
      return null;
    }

    XLInstance xlInstance =
        xlInstanceByModelId.computeIfAbsent(model.getId(), l -> createModelInstance(model));
    xlInstance.setAttachedModel(model);
    return xlInstance;
  }
}
