package app.valuationcontrol.multimodule.library.entities;

import app.valuationcontrol.multimodule.library.xlhandler.ScenarioDataProvider;
import app.valuationcontrol.multimodule.library.xlhandler.XLHandleManager;
import java.security.Principal;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentLinkedQueue;
import lombok.Getter;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.stereotype.Component;

@Component
@Log4j2
public class Events {

  @Getter private final ConcurrentLinkedQueue<Event<?>> eventsToTreat;

  private final ApplicationEventPublisher applicationEventPublisher;

  private final XLHandleManager xlHandleManager;

  @Autowired
  public Events(
      ApplicationEventPublisher applicationEventPublisher, XLHandleManager xlHandleManager) {
    this.applicationEventPublisher = applicationEventPublisher;
    this.xlHandleManager = xlHandleManager;
    this.eventsToTreat = new ConcurrentLinkedQueue<>();
  }

  public void publishCustomEvent(Event<?> customEvent) {
    log.debug("Publishing custom event. " + customEvent.getParameterClass());
    eventsToTreat.add(customEvent);
    applicationEventPublisher.publishEvent(customEvent);
  }

  public void processEvents(Principal p) {
    log.debug("processing events ");
    // If there are several events (import several values)

    List<Event<?>> principalEvents =
        this.getEventsToTreat().stream()
            .filter(e -> e.getPrincipal().getName().equals(p.getName()))
            .toList();

    if (Objects.equals(principalEvents.size(), 1)) {
      log.debug(
          "Start - Single events detected -> Remaining events"
              + (long) this.getEventsToTreat().size());

      Event<?> e = principalEvents.get(0);
      log.debug("Event was removed" + this.getEventsToTreat().remove(e));
      if (e != null) {
        if (e.isMetadataUpdate()) {
          lightReload(e.getModel());
        } else {
          switch (e.getParameterClass()) {
            case "VariableValue" -> onVariableValueEvent(e.getModel(), e);
            case "SubArea" -> onSubAreaEvent(e.getModel(), e);
            case "Sensitivity" -> onSensitivityEvent(e.getModel(), e);
            case "Segment" -> onSegmentEvent(e.getModel(), e);
            default -> fullReload(e.getModel());
          }
        }
      }
      log.debug(
          "Stop - Single event detected -> Remaining events" + this.getEventsToTreat().size());
    } else if (principalEvents.size() > 1) {
      log.debug("Multiple events detected -> Remaining events" + this.getEventsToTreat().size());
      principalEvents.stream()
          .map(
              e -> {
                this.getEventsToTreat().remove(e);
                return e.getModel();
              })
          .distinct()
          .forEach(
              m -> {
                log.debug("Reloading model  " + m.getId());
                getInstance(m).clearCacheAndReloadAndUpdateClients();
              });
      log.debug("Multiple events detected -> Remaining events" + this.getEventsToTreat().size());
    }
  }

  public ScenarioDataProvider getInstance(Model model) {
    return this.xlHandleManager.getXLInstanceForModel(model);
  }

  private void lightReload(Model model) {
    getInstance(model).reloadAndUpdateClients();
  }

  private void fullReload(Model model) {
    getInstance(model).clearCacheAndReloadAndUpdateClients();
  }

  private void onVariableValueEvent(Model model, Event<?> event) {
    if (!event.isDelete()) {
      getInstance(model)
          .refreshValue(
              ((VariableValue) event.getNewVersion()).getAttachedVariable(),
              (VariableValue) event.getNewVersion());
    } else {
      getInstance(model)
          .reloadVariableAndUpdateCache(
              ((VariableValue) event.getOldVersion()).getAttachedVariable());
    }
    // The above do no send update, so we perform a light update
    lightReload(model);
  }

  private void onSubAreaEvent(Model model, Event<?> event) {
    if (event.isUpdate()) {
      final SubArea oldVersion = (SubArea) event.getOldVersion();
      final SubArea newVersion = (SubArea) event.getNewVersion();

      if (oldVersion.isModelledAtSegment() != newVersion.isModelledAtSegment()) {
        fullReload(model);
      } else {
        lightReload(model);
      }
    } else {
      lightReload(model);
    }
  }

  private void onSensitivityEvent(Model model, Event<?> sensitivityEvent) {
    if (sensitivityEvent.isDelete()) {
      getInstance(model)
          .getSensitivityResults()
          .removeIf(
              sensitivityResult ->
                  ((Sensitivity) sensitivityEvent.effectedEntity())
                      .getId()
                      .equals(sensitivityResult.getId()));
    }
    lightReload(model);
  }

  private void onSegmentEvent(Model model, Event<?> tEvent) {
    if (tEvent.isDelete() || tEvent.isCreate()) {
      fullReload(model);
    } else {
      lightReload(model);
    }
  }
}
