package app.valuationcontrol.multimodule.library.entities;

import app.valuationcontrol.multimodule.library.helpers.DataTransformer;
import app.valuationcontrol.multimodule.library.helpers.ModelProvider;
import app.valuationcontrol.multimodule.library.records.ModelGraphData;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.GenerationType;
import jakarta.persistence.Id;
import jakarta.persistence.ManyToOne;
import java.util.Objects;
import java.util.Optional;
import lombok.Getter;
import lombok.Setter;

@Getter
@Entity
public class ModelGraph implements DataTransformer<ModelGraphData>, ModelProvider {
  /** */
  @Id
  @GeneratedValue(strategy = GenerationType.IDENTITY)
  @Column(name = "graph_id")
  private Long id;

  @ManyToOne private Model attachedModel;

  private String graphName;
  private String graphDescription;

  private Long graphVariable1Id;
  private Long graphVariable2Id;
  private Long graphVariable3Id;

  private String graphVariable1Type;
  private String graphVariable2Type;
  private String graphVariable3Type;

  @Setter private String graphVariable1Axis;
  @Setter private String graphVariable2Axis;
  @Setter private String graphVariable3Axis;

  private Boolean graphIncludeHistoricals;

  public ModelGraph() {}

  public ModelGraph(ModelGraphData modelGraphData, Model model) {
    this.updateFrom(modelGraphData);
    this.attachedModel = model;
  }

  public ModelGraph(ModelGraph existingGraph) {
    this(existingGraph.asData(), existingGraph.attachedModel);
  }

  public void updateFrom(ModelGraphData graphData) {
    this.graphName = graphData.graphName();
    this.graphDescription = graphData.graphDescription();
    this.graphVariable1Id = graphData.graphVariable1Id();
    this.graphVariable2Id = graphData.graphVariable2Id();
    this.graphVariable3Id = graphData.graphVariable3Id();
    this.graphVariable1Type = graphData.graphVariable1Type();
    this.graphVariable2Type = graphData.graphVariable2Type();
    this.graphVariable3Type = graphData.graphVariable3Type();
    this.graphVariable1Axis = graphData.graphVariable1Axis();
    this.graphVariable2Axis = graphData.graphVariable2Axis();
    this.graphVariable3Axis = graphData.graphVariable3Axis();
    this.graphIncludeHistoricals = graphData.graphIncludeHistoricals();
  }

  public boolean useSegmentAxis() {
    Optional<Variable> variable1 =
        this.attachedModel.getVariables().stream()
            .filter(variable -> Objects.equals(variable.getId(), this.graphVariable1Id))
            .findFirst();

    return variable1.isPresent()
        && variable1.get().isSingleOrConstantValue()
        && variable1.get().isModelledAtSegment();
  }

  @Override
  public ModelGraphData asData() {
    return new ModelGraphData(
        id,
        graphName,
        graphDescription,
        Objects.isNull(graphVariable1Id) ? -1 : graphVariable1Id,
        Objects.isNull(graphVariable2Id) ? -1 : graphVariable2Id,
        Objects.isNull(graphVariable3Id) ? -1 : graphVariable3Id,
        graphVariable1Type,
        graphVariable2Type,
        graphVariable3Type,
        graphVariable1Axis,
        graphVariable2Axis,
        graphVariable3Axis,
        graphIncludeHistoricals);
  }

  @Override
  public Long getModelId() {
    return this.getAttachedModel().getId();
  }

  public ModelGraph setAttachedModel(Model attachedModel) {
    this.attachedModel = attachedModel;
    return this;
  }

  public void setGraphVariable1Id(Long graphVariable1Id) {
    this.graphVariable1Id = graphVariable1Id;
  }

  public void setGraphVariable2Id(Long graphVariable2Id) {
    this.graphVariable2Id = graphVariable2Id;
  }

  public void setGraphVariable3Id(Long graphVariable3Id) {
    this.graphVariable3Id = graphVariable3Id;
  }

  public int getNumberOfVariables() {
    return (Objects.requireNonNullElse(graphVariable1Id, -1L) > 0 ? 1 : 0)
        + (Objects.requireNonNullElse(graphVariable2Id, -1L) > 0 ? 1 : 0)
        + (Objects.requireNonNullElse(graphVariable3Id, -1L) > 0 ? 1 : 0);
  }

  public int getNumberOfDifferentAxis() {
    int returnInt = 1;
    if ((graphVariable1Axis.equalsIgnoreCase("secondary")
            && (graphVariable2Axis.equalsIgnoreCase("primary")
                || graphVariable3Axis.equalsIgnoreCase("primary")))
        || (graphVariable2Axis.equalsIgnoreCase("secondary")
            && (graphVariable1Axis.equalsIgnoreCase("primary")
                || graphVariable3Axis.equalsIgnoreCase("primary")))
        || (graphVariable3Axis.equalsIgnoreCase("secondary")
            && (graphVariable1Axis.equalsIgnoreCase("primary")
                || graphVariable2Axis.equalsIgnoreCase("primary")))) {
      returnInt = returnInt + 1;
    }
    return returnInt;
  }

  public boolean useTwoAxis() {
    return Integer.valueOf(Math.min(getNumberOfDifferentAxis(), getNumberOfVariables())).equals(2);
  }
}
