package app.valuationcontrol.multimodule.library.xlhandler;

import app.valuationcontrol.multimodule.library.helpers.exceptions.ResourceException;
import java.io.*;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import lombok.extern.log4j.Log4j2;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.*;

@Log4j2
public class POICalcDocument implements CalcDocument {

  public static final String NUMBER_FORMAT ="#,##0.0;(#,##0.0)";
  public static final String PERCENTAGE_FORMAT ="#,##0.0%;(#,##0.0)%";
  public static final String DATE_FORMAT ="DD.MM.YYYY";
  private static final String SCENARIO_SHEET_PREFIX = "scenario_";
  private final Map<String, GenericSheet> sheetsByName = new ConcurrentHashMap<>();
  private final HashMap<String, XSSFCellStyle> stylesMap = new HashMap<>();
  protected XSSFWorkbook _notAccessibleWorkbook;

  public POICalcDocument() {
    initiateWorkbook();
  }

  private XSSFCellStyle addStyleTosStylesMap(String variableType, String format, boolean isValue) {
    XSSFCellStyle style = _notAccessibleWorkbook.createCellStyle();
    style.cloneStyleFrom(_notAccessibleWorkbook.getCellStyleAt(0));

    // Setting font and border
    int fontIndex = 0;
    if (variableType.contains("total") || variableType.contains("header")) {
      fontIndex = 1;
      if(variableType.contains("total")) style.setBorderBottom(BorderStyle.DOUBLE);
    } else if (variableType.equals("kpi")) {
      fontIndex = 2;
    } else if (variableType.contains("input") && isValue) {
      fontIndex = 4;
    }
    XSSFFont font = _notAccessibleWorkbook.getFontAt(fontIndex);
    style.setFont(font);

    // Setting format for values
    if (isValue) {
      style.setDataFormat((short) 1);
      if(format!=null){
        if (format.equals("percent")) style.setDataFormat((short) 2);
        if (format.equals("date")) style.setDataFormat((short) 3);
      }

      // Setting background for values
      if (variableType.contains("input")) {
        style.setFillForegroundColor(IndexedColors.LIGHT_ORANGE.getIndex());
        style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        style.setBorderBottom(BorderStyle.THIN);
        style.setBorderTop(BorderStyle.THIN);
        style.setBorderLeft(BorderStyle.THIN);
        style.setBorderRight(BorderStyle.THIN);
      }
    }

    return style;
  }

  public XSSFCellStyle getStyle(String variableType, String format, boolean isValue) {
    return stylesMap.computeIfAbsent(
        variableType + "_" + format + "_" + isValue,
        style -> addStyleTosStylesMap(variableType, format, isValue));
  }

  // Additional functions are added in WebserviceApplication
  private synchronized void initiateWorkbook() {
    this.sheetsByName.clear();
    this._notAccessibleWorkbook = new XSSFWorkbook();

    //Creating fonts
    XSSFFont font = _notAccessibleWorkbook.getFontAt((short) 0);
    font.setFontHeightInPoints((short) 8);
    font.setFontName("Arial");
    font.setFamily(3);
    font.setScheme(FontScheme.NONE);
    font.setItalic(false);
    font.setBold(false);

    XSSFFont fontBold = _notAccessibleWorkbook.createFont();
    fontBold.setFontHeightInPoints((short) 8);
    fontBold.setFontName("Arial");
    fontBold.setFamily(3);
    fontBold.setScheme(FontScheme.NONE);
    fontBold.setItalic(false);
    fontBold.setBold(true);

    XSSFFont fontItalic = _notAccessibleWorkbook.createFont();
    fontItalic.setFontHeightInPoints((short) 8);
    fontItalic.setFontName("Arial");
    fontItalic.setFamily(3);
    fontItalic.setScheme(FontScheme.NONE);
    fontItalic.setItalic(true);
    fontItalic.setBold(false);

    XSSFFont fontBoldUnderline = _notAccessibleWorkbook.createFont();
    fontBoldUnderline.setFontHeightInPoints((short) 8);
    fontBoldUnderline.setFontName("Arial");
    fontBoldUnderline.setFamily(3);
    fontBoldUnderline.setScheme(FontScheme.NONE);
    fontBoldUnderline.setItalic(false);
    fontBoldUnderline.setBold(true);
    fontBoldUnderline.setUnderline(FontUnderline.SINGLE);

    XSSFFont fontInput = _notAccessibleWorkbook.createFont();
    fontInput.setFontHeightInPoints((short) 8);
    fontInput.setFontName("Arial");
    fontInput.setFamily(3);
    fontInput.setScheme(FontScheme.NONE);
    fontInput.setItalic(false);
    fontInput.setBold(false);
    fontInput.setColor(IndexedColors.BLUE.getIndex());

    //Creating format
    XSSFDataFormat decimalFormat = _notAccessibleWorkbook.createDataFormat();
    decimalFormat.putFormat((short) 1, NUMBER_FORMAT);
    decimalFormat.putFormat((short) 2, PERCENTAGE_FORMAT);
    decimalFormat.putFormat((short) 3, DATE_FORMAT);

    //Creating cellStyle
    XSSFCellStyle style = _notAccessibleWorkbook.getCellStyleAt(0);
    style.setVerticalAlignment(VerticalAlignment.CENTER);
    style.setWrapText(false);
    _notAccessibleWorkbook
        .getStylesSource()
        .getCTStylesheet()
        .addNewCellStyles()
        .addNewCellStyle()
        .setXfId(0);
  }

  private synchronized <T> T doOneByOne(Callable<T> callable) {
    try {
      return callable.call();
    } catch (Exception exception) {
      this.initiateWorkbook();
      try {
        return callable.call();
      } catch (Exception e) {
        throw new IllegalStateException(e);
      }
    }
  }

  private synchronized XSSFWorkbook getWorkbook() {
    return _notAccessibleWorkbook;
  }

  public static String getScenarioSheetName(SCENARIO scenario) throws IllegalArgumentException {
    if (scenario != null) {
      return SCENARIO_SHEET_PREFIX + scenario.ordinal();
    }
    throw new java.lang.IllegalArgumentException(
        "Cannot produce sheet name since scenarioNumber is null");
  }

  @Override
  public void assertConnected(Runnable resetData) {
    this.doOneByOne(
        () -> {
          try {
            if (_notAccessibleWorkbook == null)
              throw new RuntimeException("Spreadsheet pointer is null");
            XSSFSheet sheet = this.getWorkbook().getSheet(getScenarioSheetName(SCENARIO.BASE));
            log.debug("Found in assertConnected for scenario sheet : {}",sheet.getSheetName());
            if (sheet.getPhysicalNumberOfRows() == 0) {
              throw new RuntimeException("Number of rows is nil so we reload");
            }
          } catch (Exception e) {
            log.debug("NOT Found in assertConnected");
            this.initiateWorkbook();
            resetData.run();
          }
          return null;
        });
  }

  @Override
  public GenericSheet getSheet(SCENARIO scenario) {
    final String scenarioSheetName = getScenarioSheetName(scenario);
    return sheetsByName.computeIfAbsent(scenarioSheetName, s -> copyBaseTo(scenario));
  }

  public GenericSheet createPrettySheet(String name) {
    return new POISheet(name, this.getWorkbook().createSheet(name));
  }

  @Override
  public void preventScreenUpdating(boolean prevent) {
    log.debug(prevent); // not in use in POI
  }

  @Override
  public void close() {

    this.doOneByOne(
        () -> {
          if (_notAccessibleWorkbook == null) {
            log.debug("Workbook was null");
            return null;
          }

          try {
            log.debug("Closing workbook");
            this.getWorkbook().close();
            _notAccessibleWorkbook = null;
            this.sheetsByName.clear();
          } catch (IOException e) {
            throw new RuntimeException(e);
          }
          return null;
        });
  }

  @Override
  public InputStream saveAs() {
    try {
      ByteArrayOutputStream outputStream1 = new ByteArrayOutputStream();
      this.getWorkbook().write(outputStream1);
      return new ByteArrayInputStream(outputStream1.toByteArray());
    } catch (IOException e) {
      throw new java.lang.RuntimeException(e);
    }
  }

  @Override
  public void remove(GenericSheet genericSheet) {
    this.doOneByOne(
        () -> {
          this.sheetsByName.remove(genericSheet.getName());
          this.getWorkbook().getSheetIndex(genericSheet.getName());
          this.getWorkbook()
              .removeSheetAt(this.getWorkbook().getSheetIndex(genericSheet.getName()));
          return null;
        });
  }

  @Override
  public GenericSheet copy(GenericSheet sourceGenericSheet, String newSheetName) {
    return this.doOneByOne(
        () -> {
          XSSFSheet sourceSheet = this.getWorkbook().getSheet(sourceGenericSheet.getName());
          if (sourceSheet != null) {
            int sourceGenericSheetIndex =
                this.getWorkbook().getSheetIndex(sourceSheet.getSheetName());
            XSSFSheet _newSheet =
                this.getWorkbook().cloneSheet(sourceGenericSheetIndex, newSheetName);

            final GenericSheet genericSheet = new POISheet(newSheetName, _newSheet);

            sheetsByName.put(newSheetName, genericSheet);
            log.info("Copied sheet successfully to {}", newSheetName);
            return genericSheet;
          } else {
            return null;
          }
        });
  }

  private GenericSheet copyBaseTo(SCENARIO scenario) {
    final String scenarioSheetName = getScenarioSheetName(scenario);

    return doOneByOne(
        () -> {
          if (SCENARIO.BASE.equals(scenario)) {
            // Create empty sheet.

            // Remove old base
            try {
              if (this.getWorkbook().getSheet(scenarioSheetName) == null) {
                this.getWorkbook().createSheet(scenarioSheetName);
              } else {
                this.getWorkbook().removeSheetAt(getSheetIndexByName(scenarioSheetName));
                this.getWorkbook().createSheet(scenarioSheetName);
              }
            } catch (ResourceException e) {
              log.error("Wrapped target exception: ", e);
            }
          } else {

            // Create the sheet only if it does not exist
            try {
              getSheetIndexByName(scenarioSheetName);
            } catch (IllegalStateException e) {
              this.getWorkbook()
                  .cloneSheet(
                      getSheetIndexByName(getScenarioSheetName(SCENARIO.BASE)), scenarioSheetName);
            }
          }

          final XSSFSheet byName = this.getWorkbook().getSheet(scenarioSheetName);

          return (GenericSheet) new POISheet(scenarioSheetName, byName);
        });
  }

  private int getSheetIndexByName(String sheetName) throws IllegalStateException {
    int sheetIndex = this.getWorkbook().getSheetIndex(sheetName);
    if (sheetIndex == -1) throw new IllegalStateException("Sheet was not found");
    return sheetIndex;
  }

  @Override
  public void closeAllButBase() {
    // Remove and close all sheets but the base one
    this.doOneByOne(
        () -> {
          final Iterator<Map.Entry<String, GenericSheet>> iterator =
              sheetsByName.entrySet().iterator();

          while (iterator.hasNext()) {
            final Map.Entry<String, GenericSheet> next = iterator.next();
            if (getScenarioSheetName(SCENARIO.BASE).equals(next.getKey())) {
              continue;
            }
            iterator.remove();
            this.getWorkbook()
                .removeSheetAt(this.getWorkbook().getSheetIndex(next.getValue().getName()));
          }
          return null;
        });
  }

  @Override
  public void closeAllSheets() { // More efficient to create a new document
    this.doOneByOne(
        () -> {
          initiateWorkbook();
          return null;
        });
  }
}
