package app.valuationcontrol.multimodule.library.xlhandler;

import static app.valuationcontrol.multimodule.library.xlhandler.CellValueHelper.generateAddress;
import static java.lang.Double.parseDouble;
import static java.lang.Integer.parseInt;
import static java.util.Collections.singletonList;

import app.valuationcontrol.multimodule.library.entities.Variable;
import app.valuationcontrol.multimodule.library.helpers.CellError;
import app.valuationcontrol.multimodule.library.helpers.DataPeriod;
import app.valuationcontrol.multimodule.library.helpers.StringReplacer;
import app.valuationcontrol.multimodule.library.helpers.VariableReplacementOccurrence;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import lombok.extern.log4j.Log4j2;

@Log4j2
public class VariableEvaluator {
  private VariableEvaluator() {}

  private static final String CREATE_TRIANGLE_GROUPS = "(createtriangle\\(([^\\(\\)]*(?:\\(.*\\))*);(.*)\\))";
  private static final String CONVERGE_TO_GROUPS = ".?(convergeto\\()(.+)(\\))";
  public static final String EQUALS = "=";
  private static final String REGEX_STRING_INTERVAL = "t[\\-|\\+]?\\d*";
  private static final String ALL_PERIODS = "allperiods";
  private static final String HISTORICAL_PERIOD = "historicalperiod";
  private static final String PROJECTION_PERIOD = "projectionperiod";
  private static final String LAST_PERIOD = "lastperiod";
  private static final String FIRST_PERIOD = "firstperiod";
  private static final Pattern PATTERN = Pattern.compile(REGEX_STRING_INTERVAL);
  private static final String LEFT_BRACKET = "[";
  private static final String RIGHT_BRACKET = "]";
  private static final String LEFT_CURLY = "{";
  private static final String RIGHT_CURLY = "}";
  public static final int AGGREGATION_SEGMENT = -1;

  private static final String[] PREDEFINED_MODIFIERS =
      new String[] {ALL_PERIODS, HISTORICAL_PERIOD, PROJECTION_PERIOD, LAST_PERIOD, FIRST_PERIOD};

  public static String evaluateVariable(
      Variable variable, DataPeriod dataPeriod, Map<Long, List<CellError>> formulaErrors) {
    String currentEvaluatedFormula = variable.getEvaluatedVariableFormula();

    // Checking if the formula equals to a double or an integer, if so just return the formula
    // =<value>
    try {
      // Double can parse both integers and double so if we can get a double out of the formula it
      // is a correct double or integer.
      parseDouble(currentEvaluatedFormula);
      return asFormula(currentEvaluatedFormula);
    } catch (Exception e) {
      // expected for formulas that are not
    }

    if (currentEvaluatedFormula.toLowerCase().matches(CREATE_TRIANGLE_GROUPS)) {
      // Addressing interpolate to
      log.info("Found a triangle");
      currentEvaluatedFormula = makeTriangleFunction(variable,dataPeriod,currentEvaluatedFormula);
    }

    if (currentEvaluatedFormula.toLowerCase().matches(CONVERGE_TO_GROUPS)) {
      // Addressing interpolate to
      currentEvaluatedFormula = makeConvergeFunction(variable, dataPeriod, currentEvaluatedFormula);
    }

    // Replace period indicator and tests
    currentEvaluatedFormula =
        replacePeriodIndicatorsAndPeriodChecks(variable, dataPeriod, currentEvaluatedFormula);

    if (variable.getReplacementOccurrences().isEmpty()) {
      evaluateVariableReplacers(variable, formulaErrors);
    }

    if (variable.getReplacementOccurrences() == null
        || variable.getReplacementOccurrences().isEmpty()) {
      return asFormula(currentEvaluatedFormula);
    }

    // Scan for variables and their position
    // Dependencies need to be sorted by Variable::variableName length desc
    for (VariableReplacementOccurrence variableReplacement : variable.getReplacementOccurrences()) {

      currentEvaluatedFormula =
          replace(
              currentEvaluatedFormula,
              getReplacementString(variable, dataPeriod, variableReplacement, formulaErrors),
              variableReplacement);
    }

    return asFormula(currentEvaluatedFormula);
  }

  /**
   * In formula replace the search string from variableReplacement with the replacementString passed
   * in
   *
   * @param formula the formula
   * @param replacementString the replacement string
   * @param variableReplacement the variable replacement o
   * @return the amended formula
   */
  private static String replace(
      String formula, String replacementString, VariableReplacementOccurrence variableReplacement) {
    return formula.replace(variableReplacement.searchString(), replacementString);
  }

  static String asFormula(String currentEvaluatedFormula) {
    return EQUALS + currentEvaluatedFormula;
  }

  /**
   * Generate the replacement string for this variableReplacement
   *
   * @param variable the variable
   * @param dataPeriod the data period
   * @param variableReplacement variable replacement
   * @return the string to replace the one in the formula
   */
  private static String getReplacementString(
      Variable variable,
      DataPeriod dataPeriod,
      VariableReplacementOccurrence variableReplacement,
      Map<Long, List<CellError>> formulaErrors) {
    String replacementString = variableReplacement.searchString();
    String prefixSearchString =
        LEFT_CURLY + variableReplacement.searchVariable().getId() + RIGHT_CURLY;
    String targetReplacementString = replacementString.substring(prefixSearchString.length());

    if (!variableReplacement.searchVariable().isModelledAtSegment()) {
      dataPeriod =
          new DataPeriod(
              variable.getAttachedModel(),
              dataPeriod.periodOffset(),
              0,
              variableReplacement.searchVariable().isSingleOrConstantValue(),dataPeriod.triangleOffset());
    }

    for (StringReplacer myReplacer : variableReplacement.stringReplacers()) {

      if (myReplacer.modifier() == null) { // if there is no modifer
        targetReplacementString =
            switch (myReplacer.searchString()) {
              case ALL_PERIODS -> variableReplacement
                  .searchVariable()
                  .generateAllPeriodsRange(dataPeriod);
              case HISTORICAL_PERIOD -> variableReplacement
                  .searchVariable()
                  .generateHistoricalRange(dataPeriod);
              case PROJECTION_PERIOD -> variableReplacement
                  .searchVariable()
                  .generateProjectionRange(dataPeriod);
              case FIRST_PERIOD -> variableReplacement
                  .searchVariable()
                  .generateFirstPeriodRange(dataPeriod);
              case LAST_PERIOD -> variableReplacement
                  .searchVariable()
                  .generateLastPeriodRange(dataPeriod);
              default -> generateAddress(variableReplacement.searchVariable(), dataPeriod);
            };

      } else {
        if (dataPeriod.isWithinBound(dataPeriod.periodOffset() + myReplacer.modifier())) {

          DataPeriod newDataPeriod =
              new DataPeriod(
                  variable.getAttachedModel(),
                  myReplacer.modifier() + dataPeriod.periodOffset(),
                  dataPeriod.segmentOffsetFactor(),
                  variableReplacement.searchVariable().isSingleOrConstantValue(),dataPeriod.triangleOffset());
          String addressReplacementString =
              generateAddress(variableReplacement.searchVariable(), newDataPeriod);
          targetReplacementString =
              targetReplacementString.replace(myReplacer.searchString(), addressReplacementString);
          targetReplacementString =
              targetReplacementString.replace(LEFT_BRACKET, "").replace(RIGHT_BRACKET, "");
        } else {
          targetReplacementString = "-1";
          CellError e =
              new CellError(
                  dataPeriod.periodOffset(),
                  "OutOfBounds error",
                  "You are trying to access an unsafe area using "
                      + myReplacer.searchString()
                      + " in formula "
                      + variable.getVariableFormula());

          formulaErrors.computeIfAbsent(variable.getId(), l -> new ArrayList<>()).add(e);
        }
      }
    }
    return targetReplacementString;
  }

  private static String replacePeriodIndicatorsAndPeriodChecks(
      Variable myVariable, DataPeriod dataPeriod, String currentEvaluatedFormula) {
    currentEvaluatedFormula =
        currentEvaluatedFormula.replace(
            "periodindicator", Integer.toString(dataPeriod.periodOffset() + 1));

    currentEvaluatedFormula =
        currentEvaluatedFormula.replace(
            "islastperiod",
            dataPeriod.periodOffset() + 1 == myVariable.getAttachedModel().getNbProjectionPeriod()
                ? "1"
                : "0");
    currentEvaluatedFormula =
        currentEvaluatedFormula.replace(
            "isfirstperiod", dataPeriod.periodOffset() == 0 ? "1" : "0");
    return currentEvaluatedFormula;
  }

  private static String makeTriangleFunction(
          Variable variable, DataPeriod dataPeriod, String originalFormula) {


    Pattern p = Pattern.compile(CREATE_TRIANGLE_GROUPS);
    Matcher m = p.matcher(originalFormula);

    String initialValueFormula;
    String followingValueFormula;
    if (m.find()) {
      initialValueFormula = m.group(2);
     followingValueFormula = m.group(3);
    } else {
      initialValueFormula = originalFormula.replace("createTriangle(", "");
      initialValueFormula = initialValueFormula.replace(")", "");
      followingValueFormula=initialValueFormula;
    }

    int triangleOffset = dataPeriod.triangleOffset().length >=1 ? dataPeriod.triangleOffset()[0] : 0;
    if(dataPeriod.periodOffset()==triangleOffset){
      return initialValueFormula;
    }else if (dataPeriod.periodOffset()>triangleOffset){
      return followingValueFormula;
    }else{
      return "";
    }

  }


  private static String makeConvergeFunction(
      Variable variable, DataPeriod dataPeriod, String originalFormula) {

    int remainingPeriod =
        variable.getAttachedModel().getNbProjectionPeriod() - dataPeriod.periodOffset();

    String previousCellAddress = generateAddress(variable, dataPeriod.previousPeriod());

    Pattern p = Pattern.compile(CONVERGE_TO_GROUPS);
    Matcher m = p.matcher(originalFormula);

    String target;
    if (m.find()) {
      target = m.group(2);
    } else {
      target = originalFormula.replace("convergeto(", "");
      target = target.replace(")", "");
    }

    return "if(isNumber("
        + previousCellAddress
        + ");"
        + previousCellAddress
        + "+("
        + target
        + "-"
        + previousCellAddress
        + ")/"
        + remainingPeriod
        + ";"
        + target
        + "/"
        + remainingPeriod
        + ")";
  }

  /**
   * Computes to Natural Language logic for the variable
   *
   * @param variable Variable to be assessed, the transient Replacers will be updated
   */
  private static void evaluateVariableReplacers(
      Variable variable, Map<Long, List<CellError>> formulaErrors) {
    int modifier;

    String currentEvaluatedFormula = variable.getEvaluatedVariableFormula();
    String tempEvaluatedFormula;
    String tempModifierString;
    String middleStringBeforeReplacements;

    for (Variable searchVariable : variable.getVariableDependencies()) {
      int stringIndex;
      int stringIndexMiddleStart;
      int stringIndexStop;
      String searchString;

      // Checking for predefined modifiers {<variableid>}[<predefinedmodifier>
      currentEvaluatedFormula =
          checkForPredefinedModifiers(variable, currentEvaluatedFormula, searchVariable);

      // looking for variables with modifiers => example [t:t+1]
      searchString = LEFT_CURLY + searchVariable.getId() + RIGHT_CURLY + LEFT_BRACKET;
      do {
        stringIndex = currentEvaluatedFormula.indexOf(searchString);

        if (stringIndex != AGGREGATION_SEGMENT) {
          stringIndexMiddleStart = stringIndex + searchString.length() - 1;
          tempEvaluatedFormula = currentEvaluatedFormula;

          /*checking for a terminating bracket for our interval*/
          if (!tempEvaluatedFormula.contains(RIGHT_BRACKET)) {

            CellError e =
                new CellError(
                    AGGREGATION_SEGMENT,
                    "FormulaError",
                    "Couldn't find terminating bracket for formula "
                        + variable.getVariableFormula());
            formulaErrors.computeIfAbsent(variable.getId(), l -> new ArrayList<>()).add(e);
            break;
          } else {
            List<StringReplacer> replacers = new ArrayList<>();

            // add 1 to include the terminating bracket in our substring ; seach from string_index
            // to find nearrest closing braket
            stringIndexStop = tempEvaluatedFormula.indexOf(RIGHT_BRACKET, stringIndex) + 1;

            // Splitting the string in 3 with the middle string being the one with the modifiers
            String startString = tempEvaluatedFormula.substring(0, stringIndex);

            String middleString =
                tempEvaluatedFormula.substring(stringIndexMiddleStart, stringIndexStop);

            String endString = tempEvaluatedFormula.substring(stringIndexStop);

            Matcher m = PATTERN.matcher(middleString);

            while (m.find()) {
              tempModifierString = m.group();
              // If the modfier is t, then there is no period modifier, setting modifier to 0
              if (tempModifierString.equals("t")) {
                modifier = 0;
              } else {
                // Remove the t and find the offset as an integer
                modifier = parseInt(tempModifierString.replace("t", ""));
              }

              StringReplacer replacer = new StringReplacer(tempModifierString, modifier);
              replacers.add(replacer);
            }
            // sorting the stringReplacers in descending length to tacle (t being includig in t+1)
            // or (t+1
            // being in t+10)
            replacers.sort(StringReplacer.StringReplacerComparatorDesc);

            // Evaluating the formula by replacing modifiers
            middleStringBeforeReplacements = middleString;
            for (StringReplacer temp_replacer : replacers) {
              middleString = middleString.replace(temp_replacer.searchString(), "");
            }
            middleString = middleString.replace(LEFT_BRACKET, "");
            middleString = middleString.replace(RIGHT_BRACKET, "");

            currentEvaluatedFormula = startString + middleString + endString;

            // Adding a replacement occurence
            VariableReplacementOccurrence vro =
                new VariableReplacementOccurrence(
                    searchVariable,
                    LEFT_CURLY
                        + searchVariable.getId()
                        + RIGHT_CURLY.concat(middleStringBeforeReplacements),
                    replacers);
            variable.getReplacementOccurrences().add(vro);

            // Example formula = average(searchvariable[t:t+1])+2
            // Startstring is the start of the formula without the searchvariable => average(
            // middlestring is the content between the brackets searchvariable[t:t+1] => [t:t+1]
            // endString is remaining content after the brackets => )+2
          }
        }
      } while (stringIndex != AGGREGATION_SEGMENT);
      // End of loop for variables with modifiers

      // looking for variables without modifiers
      searchString = LEFT_CURLY + searchVariable.getId() + RIGHT_CURLY;
      do {
        stringIndex = currentEvaluatedFormula.indexOf(searchString);
        if (stringIndex != AGGREGATION_SEGMENT) {

          List<StringReplacer> replacers = new ArrayList<>();
          StringReplacer replacer = new StringReplacer(searchString, null);
          replacers.add(replacer);
          VariableReplacementOccurrence vro =
              new VariableReplacementOccurrence(searchVariable, searchString, replacers);
          variable.getReplacementOccurrences().add(vro);
          currentEvaluatedFormula = currentEvaluatedFormula.replace(searchString, "");
        }
      } while (stringIndex != AGGREGATION_SEGMENT);
      // End of loop for variables without modifiers
    }
  }

  /**
   * Go through our the predefined modifiers and search for the searchVariable, adds the
   * searchVariable to the variable dependencies
   *
   * @param variable the variable
   * @param currentEvaluatedFormula the formula currently evaluated
   * @param searchVariable the search variable
   * @return the updated formula with the searchVariable
   */
  private static String checkForPredefinedModifiers(
      Variable variable, String currentEvaluatedFormula, Variable searchVariable) {
    String searchString;

    for (String predefinedModifier : PREDEFINED_MODIFIERS) {
      searchString =
          LEFT_CURLY
              + searchVariable.getId()
              + RIGHT_CURLY
              + LEFT_BRACKET
              + predefinedModifier
              + RIGHT_BRACKET;

      if (currentEvaluatedFormula.contains(searchString)) {

        StringReplacer replacer = new StringReplacer(predefinedModifier, null);

        VariableReplacementOccurrence vro =
            new VariableReplacementOccurrence(
                searchVariable, searchString, singletonList(replacer));

        variable.getReplacementOccurrences().add(vro);

        currentEvaluatedFormula = currentEvaluatedFormula.replace(searchString, "");
      }
    }
    return currentEvaluatedFormula;
  }
}
