package app.valuationcontrol.webservice.enin;

import app.valuationcontrol.webservice.enin.records.*;
import app.valuationcontrol.webservice.helpers.exceptions.ResourceException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;

@Service
public class EninAPIService {
  @Autowired private WebClient webClientAnalysis;

  @Autowired private WebClient webClientDataset;

  private final String API_PREFIX = "https://api.enin.ai/analysis/v1/";

  @SuppressWarnings("FieldCanBeLocal")
  private final String API_DATASET_PREFIX = "https://api.enin.ai/datasets/v1/";

  private <M> M getAnalysisRecords(Class<M> parameterClass, String requestUri) {
    try {
      return webClientAnalysis
          .get()
          .uri(requestUri)
          .accept(MediaType.APPLICATION_JSON)
          .retrieve()
          .bodyToMono(parameterClass)
          .block();
    } catch (Exception e) {
      throw new ResourceException(
          HttpStatus.BAD_REQUEST, "Couldn't reach ENIN api for " + requestUri);
    }
  }

  private <M> M getDatasetRecords(
      @SuppressWarnings("SameParameterValue") Class<M> parameterClass, String requestUri) {
    try {
      return webClientDataset
          .get()
          .uri(requestUri)
          .accept(MediaType.APPLICATION_JSON)
          .retrieve()
          .bodyToMono(parameterClass)
          .block();
    } catch (Exception e) {
      throw new ResourceException(
          HttpStatus.BAD_REQUEST, "Couldn't reach ENIN api for " + requestUri);
    }
  }

  public CompanyNameRecord[] getCompanies(String companyName) {
    String requestUri =
        API_DATASET_PREFIX
            + "dataset/company-name-composite?company_name.name=ILIKE:%"
            + companyName
            + "%&keep_only_fields=company.name,company.org_nr,company.org_nr_schema,company_details.latest_accounting_year&limit=15";
    return getDatasetRecords(CompanyNameRecord[].class, requestUri);
  }

  public IncomeStatementRecord[] getIncomeStatement(
      String companyNumber, boolean doImportGroupFigures) {
    String requestUri =
        API_PREFIX
            + "company/"
            + companyNumber
            + "/accounts-income-statement?accounts_type_identifier=annual_corporate_group_accounts";
    String requestUriSolo = API_PREFIX + "company/" + companyNumber + "/accounts-income-statement";
    IncomeStatementRecord[] incomeStatementRecords =
        getAnalysisRecords(
            IncomeStatementRecord[].class, doImportGroupFigures ? requestUri : requestUriSolo);
    if (incomeStatementRecords.length == 0 && doImportGroupFigures) // if Group figures not found
    {
      incomeStatementRecords = getAnalysisRecords(IncomeStatementRecord[].class, requestUriSolo);
    }
    return incomeStatementRecords;
  }

  public BalanceSheetRecord[] getBalanceSheet(String companyNumber, boolean doImportGroupFigures) {
    String requestUri =
        API_PREFIX
            + "company/"
            + companyNumber
            + "/accounts-balance-sheet?accounts_type_identifier=annual_corporate_group_accounts";
    String requestUriSolo = API_PREFIX + "company/" + companyNumber + "/accounts-balance-sheet";
    BalanceSheetRecord[] balanceSheetRecord =
        getAnalysisRecords(
            BalanceSheetRecord[].class, doImportGroupFigures ? requestUri : requestUriSolo);
    if (balanceSheetRecord.length == 0 && doImportGroupFigures) // if Group figures not found
    {
      balanceSheetRecord = getAnalysisRecords(BalanceSheetRecord[].class, requestUriSolo);
    }
    return balanceSheetRecord;
  }

  public EninCompanyDescriptionRecord[] getDescription(String companyNumber) {
    String requestUri =
        API_PREFIX
            + "company/"
            + companyNumber
            + "/company-description?company_description_type_identifier=brreg_official_purpose";
    return getAnalysisRecords(EninCompanyDescriptionRecord[].class, requestUri);
  }

  public EninCompanyRecord getCompanyRecord(String companyNumber, boolean doImportGroupFigures) {
    if (companyNumber.toUpperCase().matches("NO[0-9]{9}")) {
      return new EninCompanyRecord(
          companyNumber,
          getDescription(companyNumber)[0],
          getIncomeStatement(companyNumber, doImportGroupFigures),
          getBalanceSheet(companyNumber, doImportGroupFigures));
    } else {
      throw new ResourceException(HttpStatus.BAD_REQUEST, "Company number is not valid");
    }
  }
}
