package app.valuationcontrol.webservice.model;

import static java.util.stream.Collectors.toMap;
import static org.springframework.http.HttpStatus.CREATED;

import app.valuationcontrol.webservice.EntityService;
import app.valuationcontrol.webservice.enin.EninAPIService;
import app.valuationcontrol.webservice.enin.EninMappingToTemplate;
import app.valuationcontrol.webservice.enin.records.BalanceSheetRecord;
import app.valuationcontrol.webservice.enin.records.CompanyNameRecord;
import app.valuationcontrol.webservice.enin.records.EninCompanyRecord;
import app.valuationcontrol.webservice.enin.records.IncomeStatementRecord;
import app.valuationcontrol.webservice.helpers.CalculationData;
import app.valuationcontrol.webservice.helpers.FormulaEvaluator;
import app.valuationcontrol.webservice.helpers.exceptions.ResourceException;
import app.valuationcontrol.webservice.model.area.Area;
import app.valuationcontrol.webservice.model.events.Event;
import app.valuationcontrol.webservice.model.events.Events;
import app.valuationcontrol.webservice.model.graph.ModelGraph;
import app.valuationcontrol.webservice.model.segment.Segment;
import app.valuationcontrol.webservice.model.sensitivity.Sensitivity;
import app.valuationcontrol.webservice.model.subarea.SubArea;
import app.valuationcontrol.webservice.model.variable.Variable;
import app.valuationcontrol.webservice.model.variable.VariableData;
import app.valuationcontrol.webservice.model.variable.VariableRepository;
import app.valuationcontrol.webservice.model.variablevalue.VariableValue;
import app.valuationcontrol.webservice.openai.OpenAiServiceImplementation;
import app.valuationcontrol.webservice.presentation.PresentationManager;
import app.valuationcontrol.webservice.user.User;
import app.valuationcontrol.webservice.user.UserRepository;
import app.valuationcontrol.webservice.xlhandler.*;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import jakarta.validation.Valid;
import java.io.*;
import java.security.Principal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.IntConsumer;
import lombok.extern.log4j.Log4j2;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

@RestController
@Log4j2
@Transactional
public class ModelController {
  public static final String MODEL_ID = "modelId";
  public static final String MODEL_ID_DESCRIPTION = "The id of the model to be amended or deleted";
  private final ModelRepository modelRepository;
  private final VariableRepository variableRepository;
  private final XLHandleManager xlHandleManager;
  private final EntityService entityService;
  private final UserRepository userRepository;
  private final Events events;

  private final EninAPIService eninAPIService;

  private final OpenAiServiceImplementation openAiServiceImplementation;

  /**
   * Initializes the model controller and establish link to database Creator function that ensures
   * that the model is connected to a document
   */
  public ModelController(
      ModelRepository modelRepository,
      VariableRepository variableRepository,
      XLHandleManager xlHandleManager,
      EntityService entityService,
      UserRepository userRepository,
      Events events,
      EninAPIService eninAPIService,
      OpenAiServiceImplementation openAiServiceImplementation) {

    this.modelRepository = modelRepository;
    this.variableRepository = variableRepository;
    this.xlHandleManager = xlHandleManager;
    this.entityService = entityService;
    this.userRepository = userRepository;
    this.events = events;
    this.eninAPIService = eninAPIService;
    this.openAiServiceImplementation = openAiServiceImplementation;
  }

  /** Printing code for an entire model to use in FullModelTest */
  @Operation(summary = "This method is used to run tests", hidden = true)
  @GetMapping("/api/cypress/{modelId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  public ResponseEntity<List<String>> cypress(@PathVariable(value = MODEL_ID) Model model) {
    List<String> cypressCommands = new ArrayList<>();

    System.out.println("Long zoneId, subZoneId, variableId;");

    model
        .getSegments()
        .forEach(
            segment ->
                System.out.println(
                    "Long segment"
                        + segment.getId()
                        + " = addSegment(\""
                        + segment.getSegmentName()
                        + "\", \""
                        + segment.getSegmentCurrency()
                        + "\");"));

    model
        .getAreas()
        .forEach(
            area -> {
              System.out.println(
                  "zoneId = addZone(\""
                      + area.getAreaName()
                      + "\", \""
                      + area.getAreaDescription()
                      + "\");");
              area.getSubAreas()
                  .forEach(
                      subArea -> {
                        System.out.println(
                            "subZoneId = addSubZone(\""
                                + subArea.getSubAreaName()
                                + "\", \""
                                + subArea.getSubAreaDescription()
                                + "\", "
                                + "zoneId"
                                + ", "
                                + subArea.isModelledAtSegment()
                                + ");");
                        model.getVariables().stream()
                            .filter(variable -> variable.getVariableSubAreaId() == subArea.getId())
                            .forEach(
                                variable -> {
                                  System.out.println(
                                      "variableId = addVariableToAreaSubarea(\""
                                          + variable.getVariableName()
                                          + "\", \""
                                          + variable.getVariableFormula()
                                          + "\", \""
                                          + variable.getVariableType()
                                          + "\", \""
                                          + variable.getVariableFormat()
                                          + "\","
                                          + "zoneId"
                                          + ","
                                          + "subZoneId"
                                          + ");");
                                  variable
                                      .getVariableValues()
                                      .forEach(
                                          variableValue ->
                                              System.out.println(
                                                  "super.createVariableValue(new VariableValueData(-1, "
                                                      + (variableValue.getPeriod() != null
                                                          ? variableValue.getPeriod()
                                                          : "-1")
                                                      + ", "
                                                      + variableValue.getValue()
                                                      + "f, null,"
                                                      + variableValue.getScenarioNumber()
                                                      + ", "
                                                      + (variableValue.getAttachedSegment() != null
                                                          ? "segment"
                                                              + variableValue
                                                                  .getAttachedSegment()
                                                                  .getId()
                                                          : null)
                                                      + "), variableId);"));
                                });
                      });
            });

    return ResponseEntity.ok(cypressCommands);
  }

  @Operation(
      summary = "Create a new model from Enin company data",
      description = "Use this entrypoint to bootstrap a model from EninData",
      responses = {
        @ApiResponse(responseCode = "201", description = "the id of the created model"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PostMapping("/api/modelenin")
  public ResponseEntity<Long> createModelFromEnin(
      @Parameter(description = "a JSON object containing the company details") @RequestBody @Valid
          CompanyNameRecord companyNameRecord,
      @Parameter(description = "the start year for the model") @RequestParam Integer startYear,
      @Parameter(description = "A name for the created model")
          @RequestParam(defaultValue = "Valuation model")
          String modelName,
      @Parameter(description = "Boolean that indicates whether to import group figures")
          @RequestParam(defaultValue = "false")
          boolean doImportGroupFigures,
      @Parameter(description = "Indicates the unitDivider") @RequestParam(defaultValue = "1000")
          Integer unitDivider,
      Principal principal) {

    String currency =
        switch (unitDivider) {
          case 1000000:
            yield "NOKm";
          case 1000000000:
            yield "NOKb";
          default:
            yield "NOKk";
        };

    if (doImportGroupFigures && (!modelName.contains("group") || !modelName.contains("konsern")))
      modelName = modelName + " (Group)";

    ModelData modelData =
        new ModelData(
            -1L,
            modelName,
            startYear,
            3,
            10,
            companyNameRecord.company().name(),
            companyNameRecord.company().org_nr_schema() + companyNameRecord.company().org_nr(),
            currency,
            "template",
            false,
            false,
            null);
    Long modelId = addModel(modelData, 626L, false, principal).getBody();

    assert modelId != null;
    Model model = modelRepository.getReferenceById(modelId);
    updateModelWithEninData(model, unitDivider, doImportGroupFigures, principal);
    return ResponseEntity.ok(modelId);
  }

  private void updateModelWithEninData(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      Integer unitDivider,
      Boolean doImportGroupFigures,
      Principal principal) {

    EninCompanyRecord eninCompanyRecord =
        eninAPIService.getCompanyRecord(model.getCompanyNumber(), doImportGroupFigures);
    log.info(model.getCompanyNumber());

    EninMappingToTemplate.getMappingMap()
        .forEach(
            (key, value) -> {
              log.info("Checking value " + value + " for key " + key);
              Optional<Variable> variable =
                  model.getVariables().stream()
                      .filter(v -> v.getVariableName().equals(key))
                      .findFirst();
              if (variable.isPresent()) {
                log.info("Found " + variable.get().getVariableName());
                variable.get().getVariableValues().clear(); // Clearing variable balues
                for (int i = -model.getNbHistoricalPeriod(); i < 0; i++) {
                  Integer accounting_year = model.getStartYear() + i;
                  Integer index = i;

                  // Mapping P&L
                  Optional<IncomeStatementRecord> eisr =
                      Arrays.stream(eninCompanyRecord.incomeStatementRecords())
                          .filter(r -> r.accounting_year().equals(accounting_year))
                          .findFirst();
                  Optional<BalanceSheetRecord> ebsr =
                      Arrays.stream(eninCompanyRecord.balanceSheetRecords())
                          .filter(r -> r.accounting_year().equals(accounting_year))
                          .findFirst();
                  EvaluationContext context = new StandardEvaluationContext();
                  eisr.ifPresent(
                      incomeStatementRecord -> context.setVariable("eisr", incomeStatementRecord));
                  ebsr.ifPresent(
                      balanceSheetRecord -> context.setVariable("ebsr", balanceSheetRecord));
                  if (eisr.isPresent() && ebsr.isPresent()) {
                    Float eninValue =
                        new SpelExpressionParser()
                            .parseExpression(value)
                            .getValue(context, Float.class);

                    log.debug(eninValue);
                    if (eninValue != null) {
                      VariableValue vv = new VariableValue();
                      vv.setAttachedVariable(variable.get());
                      vv.setValue(eninValue / unitDivider);
                      vv.setPeriod(index);
                      vv.setScenarioNumber(0);
                      vv.setEditor(principal.getName());
                      vv.setSourceFile("Enin API");
                      variable.get().getVariableValues().add(vv);
                      Event<VariableValue> event =
                          Event.created(this, vv, principal, VariableValue.class, model);
                      events.publishCustomEvent(event);
                    }
                  }
                }
              }
            });

    events.processEvents(principal);
  }

  @Operation(
      summary = "Add a new model",
      description =
          "Use this entrypoint to add a new model or replicate an existing one using the template id paramater",
      responses = {
        @ApiResponse(responseCode = "201", description = "the id of the created model"),
            @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
            @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PostMapping("/api/model")
  @PreAuthorize("authentication.principal.hasAccessToTemplate(#templateId)")
  public ResponseEntity<Long> addModel(
      @Parameter(description = "a JSON object containing the model data") @RequestBody @Valid
          ModelData modelData,
      @Parameter(description = "the id of the template model")
          @RequestParam(required = false, defaultValue = "-1")
          Long templateId,
      @Parameter(
              description =
                  "A boolean indicating whether to copy existing values from template model")
          @RequestParam(required = false, defaultValue = "true")
          boolean copyVariableValues,
      Principal principal) {
    Model newModel = new Model(modelData);

    // Key param must be updated separately
    newModel.setKeyParam(new KeyParam());

    // No template
    if (templateId == -1L) {
      return entityService
          .safeCreate(Model.class, newModel)
          .map(
              model -> {
                Event<Model> modelEvent = Event.created(this, model, principal, Model.class, model);
                events.publishCustomEvent(modelEvent);
                return new ResponseEntity<>(model.getId(), CREATED);
              })
          .orElse(ResponseEntity.badRequest().build());
    }

    return modelRepository
        .findById(templateId)
        .map(
            templateModel -> {
              KeyParam keyParam = new KeyParam();
              keyParam.copyFromExisting(templateModel.getKeyParam(), newModel);
              newModel.setKeyParam(keyParam);
              newModel.setTemplateId(templateId);
              newModel.setSimpleModel(templateModel.getSimpleModel());

              final Model createdModel = entityService.create(Model.class, newModel);
              Event<Model> modelEvent =
                  Event.created(this, createdModel, principal, Model.class, createdModel);
              events.publishCustomEvent(modelEvent);

              /*replicating all areas and sub_areas*/
              copyAreasAndSubAreasSegmentsAndVariableValues(
                  createdModel, templateModel, copyVariableValues);

              // Creating a hashmap with old and new variable ids
              final Map<Long, Long> idMap =
                  createdModel.getVariables().stream()
                      .distinct()
                      .collect(toMap(Variable::getOriginalId, Variable::getId));

              copyGraphs(createdModel, templateModel, idMap);
              copySensitivities(createdModel, templateModel, idMap);

              keyParam.replaceIds(idMap);

              initiateFormulas(createdModel);

              return new ResponseEntity<>(
                  entityService.update(Model.class, newModel).getId(), CREATED);
            })
        .orElse(ResponseEntity.badRequest().build());
  }

  @Operation(
      summary = "Lock a model for changes",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PatchMapping("/api/model/{modelId}/lock")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'ADMIN')")
  public ResponseEntity<String> lockModel(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      Principal principal) {

    model.setLocked(true);
    Event<Model> event = Event.lightUpdated(this, model, principal, Model.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);
    return ResponseEntity.ok().build();
  }

  @Operation(
      summary = "Unlock a model for changes",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PatchMapping("/api/model/{modelId}/unlock")
  @PreAuthorize("authentication.principal.canUnlock(#model,'ADMIN')")
  public ResponseEntity<String> unlockModel(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      Principal principal) {

    model.setLocked(false);
    Event<Model> event = Event.lightUpdated(this, model, principal, Model.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);
    return ResponseEntity.ok().build();
  }

  @Operation(
      summary = "Update a model",
      description = "Use this entrypoint to update an existing model",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PutMapping("/api/model/{modelId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'ADMIN')")
  public ResponseEntity<String> updateModel(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @RequestBody @Valid ModelData modelData,
      Principal principal) {

    Model oldModel = new Model(model.asData());

    model.setSimpleModel(modelData.simpleModel());
    model.setCurrency(modelData.currency());
    model.setName(modelData.name());
    model.setCompany(modelData.company());
    model.setCompanyNumber(modelData.companyNumber());
    model.setLocked(modelData.locked());
    model.setIncludeYTD(modelData.includeYTD());

    Event<Model> event = Event.updated(this, oldModel, model, principal, Model.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    return ResponseEntity.ok().build();
  }

  /**
   * Copies sensitivities from the templateModel to the newModel
   *
   * @param newModel the new model
   * @param templateModel the template model
   * @param idMap a map containing information of variables that changed id
   */
  private void copySensitivities(Model newModel, Model templateModel, Map<Long, Long> idMap) {
    templateModel
        .getSensitivities()
        .forEach(
            sensitivity -> {
              Sensitivity newSensitivity = new Sensitivity(sensitivity);
              newSensitivity.setAttachedModel(newModel);
              newSensitivity.setId(-1L);

              // New ids
              if (sensitivity.getSensitivityVariable1Id() != null) {
                newSensitivity.setSensitivityVariable1Id(
                    idMap.get(sensitivity.getSensitivityVariable1Id()));
              }

              if (sensitivity.getSensitivityVariable2Id() != null) {
                newSensitivity.setSensitivityVariable2Id(
                    idMap.get(sensitivity.getSensitivityVariable2Id()));
              }

              if (sensitivity.getSensitivityMeasurementVariableId() != null) {
                newSensitivity.setSensitivityMeasurementVariableId(
                    idMap.get(sensitivity.getSensitivityMeasurementVariableId()));
              }

              entityService.create(Sensitivity.class, newSensitivity);
            });
  }

  private void copyGraphs(Model newModel, Model templateModel, Map<Long, Long> idMap) {
    templateModel
        .getGraphs()
        .forEach(
            modelGraph -> {
              ModelGraph newModelGraph = new ModelGraph(modelGraph);
              newModelGraph.setAttachedModel(newModel);

              if (modelGraph.getGraphVariable1Id() != null) {
                newModelGraph.setGraphVariable1Id(idMap.get(modelGraph.getGraphVariable1Id()));
              }
              if (modelGraph.getGraphVariable2Id() != null) {
                newModelGraph.setGraphVariable2Id(idMap.get(modelGraph.getGraphVariable2Id()));
              }
              if (modelGraph.getGraphVariable3Id() != null) {
                newModelGraph.setGraphVariable3Id(idMap.get(modelGraph.getGraphVariable3Id()));
              }

              entityService.create(ModelGraph.class, newModelGraph);
            });
  }

  private void copyAreasAndSubAreasSegmentsAndVariableValues(
      Model newModel, Model templateModel, boolean copyVariableValues) {

    /*Replicating segments*/
    Map<Long, Segment> oldIdToNewSegmentId = new HashMap<>();

    templateModel
        .getSegments()
        .forEach(
            segment -> {
              Segment segmentCopy = new Segment(segment.asData(), newModel);
              oldIdToNewSegmentId.put(
                  segment.getId(), entityService.create(Segment.class, segmentCopy));
            });

    for (Area area : templateModel.getAreas()) {
      // Creating a duplicate of the area
      Area areaCopy =
          new Area(newModel, area.getAreaName(), area.getAreaDescription(), area.getAreaOrder());
      newModel.getAreas().add(areaCopy);

      areaCopy = entityService.create(Area.class, areaCopy);

      copySubAreasAndVariableValues(
          area, areaCopy, templateModel, newModel, oldIdToNewSegmentId, copyVariableValues);
    }
  }

  private void copySubAreasAndVariableValues(
      Area originalArea,
      Area areaCopy,
      Model templateModel,
      Model newModel,
      Map<Long, Segment> oldIdToNewSegmentId,
      boolean copyVariableValues) {
    for (SubArea subArea : originalArea.getSubAreas()) {
      SubArea subAreaCopy = new SubArea(subArea);
      subAreaCopy.setAttachedArea(areaCopy);

      subAreaCopy = entityService.create(SubArea.class, subAreaCopy);

      // Replacing variables in the sub area of the model
      for (Variable variable : templateModel.getVariables()) {
        if (variable.getVariableSubArea() == subArea) {
          Variable variableCopy = new Variable(variable, subAreaCopy);
          variableCopy.setOriginalId(variable.getId());
          // Used to store the original ID from the template variable
          final Variable createdVariable = entityService.create(Variable.class, variableCopy);
          newModel.getVariables().add(createdVariable);

          AtomicInteger periodOffset = new AtomicInteger(0);
          /*Offsetting of variableValue*/
          if (!Objects.equals(newModel.getStartYear(), templateModel.getStartYear())) {
            periodOffset.set(templateModel.getStartYear() - newModel.getStartYear());
          }

          if (copyVariableValues) {
            copyVariableValues(
                newModel,
                oldIdToNewSegmentId,
                variable,
                variableCopy,
                createdVariable,
                periodOffset);
          }
        }
      }
    }
  }

  private void copyVariableValues(
      Model newModel,
      Map<Long, Segment> oldIdToNewSegmentId,
      Variable variable,
      Variable variableCopy,
      final Variable createdVariable,
      AtomicInteger periodOffset) {
    variable
        .getVariableValues()
        .forEach(
            variableValue -> {
              VariableValue copy = new VariableValue(variableValue.asData(), variableCopy);
              // Checking if newVariable.period (with offset) is within model period range
              Integer newPeriod =
                  copy.getPeriod() != null ? copy.getPeriod() + periodOffset.get() : null;

              if (newPeriod == null
                  || (newPeriod >= -newModel.getNbHistoricalPeriod()
                      && newPeriod < newModel.getNbProjectionPeriod())) {

                copy.setPeriod(newPeriod);

                if (variableValue.getAttachedSegment() != null) {
                  copy.setAttachedSegment(
                      oldIdToNewSegmentId.get(variableValue.getAttachedSegment().getId()));
                }
                final VariableValue createdVariableValue =
                    entityService.create(VariableValue.class, copy);
                createdVariable.getVariableValues().add(createdVariableValue);
              }
            });
  }

  /*
   * Resets the rows of the variables if there is a modification in the model (deletion)
   */
  public void initiateFormulas(Model model) {

    for (Variable variable : model.getVariables()) {
      // Re-evaluating variable if no Formula nor Dependencies
      if (!variable.getVariableType().equals("constant")) {
        if (variable.getEvaluatedVariableFormula() == null
            || (variable.getVariableDependencies().isEmpty()
                && !variable.getVariableType().equals("input"))) {
          variable.setEvaluatedVariableFormula(FormulaEvaluator.evaluateVariableFormula(variable));
          // variableRepository.save(variable);
        }
      } else {
        if (variable.getEvaluatedVariableFormula() == null) {
          variable.setEvaluatedVariableFormula(FormulaEvaluator.evaluateVariableFormula(variable));
          // variableRepository.save(variable);
        }
      }
    }
    variableRepository.saveAll(model.getVariables());
  }

  @Operation(
      summary = "Delete a model",
      description = "Use this entrypoint to delete an existing model",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @DeleteMapping("/api/model/{modelId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'ADMIN')")
  public ResponseEntity<Void> deleteModel(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      Principal principal) {

    final List<User> byModel = userRepository.findByModel(model);

    byModel.forEach(
        user -> {
          user.getModelRoles().remove(model);
          userRepository.saveAndFlush(user);
        });

    Event<Model> event = Event.deleted(this, model, principal, Model.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    entityService.safeDelete(Model.class, model);

    return ResponseEntity.ok().build();
  }

  @Operation(
      summary = "Update the key parameters of an existing model",
      description =
          "Use this entrypoint to update the key parameters of an existing model (main outputs and key values drivers)",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PutMapping("/api/model/{modelId}/keyparam")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> updateKeyParam(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(
              description = "A JSON Object specifying the key parameters and outputs of the model")
          @RequestBody
          @Valid
          KeyParam keyParam,
      Principal principal) {
    model.setKeyParam(keyParam);

    safeSetPeriod(
        keyParam.getKeyOutput1Id(), keyParam::setKeyOutput1Period, keyParam.getKeyOutput1Period());
    safeSetPeriod(
        keyParam.getKeyOutput2Id(), keyParam::setKeyOutput2Period, keyParam.getKeyOutput2Period());

    safeSetPeriod(
        keyParam.getKeyParam1Id(), keyParam::setKeyParam1Period, keyParam.getKeyParam1Period());
    safeSetPeriod(
        keyParam.getKeyParam2Id(), keyParam::setKeyParam2Period, keyParam.getKeyParam2Period());
    safeSetPeriod(
        keyParam.getKeyParam3Id(), keyParam::setKeyParam3Period, keyParam.getKeyParam3Period());
    safeSetPeriod(
        keyParam.getKeyParam4Id(), keyParam::setKeyParam4Period, keyParam.getKeyParam4Period());

    Event<Model> modelEvent = Event.lightUpdated(this, model, principal, Model.class, model);
    events.publishCustomEvent(modelEvent);
    events.processEvents(principal);

    return ResponseEntity.ok().build();
  }

  private void safeSetPeriod(Long id, IntConsumer setValue, Integer period) {
    variableRepository
        .findById(id)
        .ifPresent(
            variable -> {
              if (variable.isSingleOrConstantValue()) {
                setValue.accept(0);
              } else {
                try {
                  setValue.accept(Sensitivity.fromYear(period, variable.getAttachedModel()));
                } catch (IllegalArgumentException illegalArgumentException) {
                  setValue.accept(0);
                }
              }
            });
  }

  /**
   * @return an object that contains all the metadata of the model
   */
  @Operation(
      summary = "Get the model's metadata",
      description = "Use this entrypoint to fetch the model's metadata (no values)",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @GetMapping(value = "/api/model/{modelId}/metadata")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  public ResponseEntity<CalculationData> getMetadata(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(description = "the scenario number to fetch - (0-4) with default 0")
          @RequestParam(defaultValue = "0")
          Integer scenarioNumber) {

    final CalculationData content = new CalculationData(model, null, null, null, scenarioNumber);
    return ResponseEntity.ok().body(content);
  }

  /**
   * @return an object that contains all the processed data in the spreadsheet
   */
  @Operation(
      summary = "Get the model's metadata and calculations",
      description = "Use this entrypoint to fetch the model's metadata and values",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @GetMapping(value = "/api/model/{modelId}/calculation")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  public ResponseEntity<CalculationData> getCalculation(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(description = "the scenario number to fetch - (0-4) with default 0")
          @RequestParam(defaultValue = "0")
          Integer scenarioNumber) {

    final CalculationData content =
        this.xlHandleManager.getXLInstanceForModel(model).getContent(SCENARIO.from(scenarioNumber));
    return ResponseEntity.ok().body(content);
  }

  /**
   * @return an object that contains all the processed data in the spreadsheet
   */
  @Operation(
      summary = "Returns a simple excel file containing all variables",
      description = "Use this entrypoint to fetch an Excel file containing all variables",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(
            responseCode = "400",
            description = "Invalid request parameters or empty model with no variables"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @GetMapping(value = "/api/model/{modelId}/getExcel")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  ResponseEntity<byte[]> getModelAsExcel(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model) {

    if (model.getVariables().isEmpty()) {
      throw new ResourceException(
          HttpStatus.BAD_REQUEST, "Cannot export a model without any variable");
    } else {
      HttpHeaders headers = new HttpHeaders();
      headers.add("Content-Disposition", "attachment; filename=model.xlsx");
      headers.setContentType(
          new MediaType("application", "vnd.openxmlformats-officedocument.spreadsheetml.sheet"));
      try {
        return ResponseEntity.ok()
            .headers(headers)
            .body(this.xlHandleManager.getXLInstanceForModel(model).saveAs().readAllBytes());
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }
  }

  /**
   * @return an object that contains all the processed data in the spreadsheet
   */
  @Operation(
      summary = "Get the model's metadata and calculations",
      description = "Use this entrypoint to fetch the model's metadata and values",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @GetMapping(value = "/api/model/{modelId}/getPresentation")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  ResponseEntity<byte[]> getModelAsPPTX(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(
              description =
                  "A boolean indicating whether to include AI comments in the presentation")
          @RequestParam(required = false, defaultValue = "false")
          boolean includeAIComments) {

    HttpHeaders headers = new HttpHeaders();
    headers.add("Content-Disposition", "attachment; filename=model.pptx");
    headers.setContentType(
        new MediaType(
            "application", "vnd.openxmlformats-officedocument.presentationml.presentation"));
    try {
      ScenarioDataProvider scenarioDataProvider = this.xlHandleManager.getXLInstanceForModel(model);
      // Running sensitivities
      scenarioDataProvider.runSensitivities(model.getSensitivities(), SCENARIO.BASE);
      model
          .getSensitivities()
          .forEach(sensitivity -> sensitivity.setSensitivityLastRun(LocalDateTime.now()));

      // Getting data with sensitivities
      CalculationData calculationData = scenarioDataProvider.getContent(SCENARIO.BASE);

      PresentationManager presentationManager =
          new PresentationManager(model, calculationData, openAiServiceImplementation);

      presentationManager.createSummary();
      presentationManager.createCharts(includeAIComments);
      presentationManager.createSensitivities(includeAIComments);

      ScenarioComparison comparisons = Objects.requireNonNull(getComparison(model).getBody());
      if (!comparisons.comparisons().isEmpty()) {
        presentationManager.createComparison(comparisons, includeAIComments);
      }

      calculationData
          .getAreas()
          .forEach(
              a ->
                  a.subAreas()
                      .forEach(
                          sa ->
                              presentationManager.createTable(
                                  sa.id(),
                                  SCENARIO.BASE,
                                  a.areaName() + " - " + sa.subAreaName())));

      ByteArrayOutputStream outputStream1 = new ByteArrayOutputStream();
      presentationManager.getSlideShow().write(outputStream1);

      return ResponseEntity.ok()
          .headers(headers)
          .body(new ByteArrayInputStream(outputStream1.toByteArray()).readAllBytes());
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  @SuppressWarnings("SameReturnValue")
  @Operation(
      summary = "This method is used to test that the application is up and running, returns pong",
      hidden = true)
  @GetMapping("/api/ping")
  public String debug() {
    return "pong";
  }

  /**
   * @return an object that contains all the data in the spreadsheet
   */
  @Operation(
      summary = "Compare key variables in the model across the scenarios",
      description =
          "Compare the key variables across the scenarios used in the model (key parameters, graphs and sensitivities) and returns an ",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @GetMapping(value = "/api/model/{modelId}/comparison")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  public ResponseEntity<ScenarioComparison> getComparison(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model) {

    Set<Long> keyVariables = model.getKeyVariablesForModel();

    List<CalculationData> scenarioCalculationData = new ArrayList<>();
    final ScenarioDataProvider xlInstanceForModel =
        new XLInstance(
            model,
            new POICalcDocument(),
            null); // Creating a standalone excel file for scenarioComparison
    // this.xlHandleManager.getXLInstanceForModel(model); //

    for (int i = 0; i < 5; i++) {
      scenarioCalculationData.add(xlInstanceForModel.getContent(SCENARIO.from(i)));
    }

    final List<List<VariableData>> comparisonStructure = new ArrayList<>();

    for (Long variableId : keyVariables) {
      List<VariableData> data =
          scenarioCalculationData.stream()
              .flatMap(
                  calculationData ->
                      calculationData.getVariables().stream()
                          .filter(variableData -> Objects.equals(variableData.id(), variableId)))
              .toList();
      comparisonStructure.add(data);
    }

    return ResponseEntity.ok(new ScenarioComparison(comparisonStructure));
  }
}
