package app.valuationcontrol.webservice.model;

import app.valuationcontrol.webservice.helpers.CalculationData;
import app.valuationcontrol.webservice.helpers.exceptions.ResourceException;
import app.valuationcontrol.webservice.user.UserRepository;
import app.valuationcontrol.webservice.xlhandler.SCENARIO;
import app.valuationcontrol.webservice.xlhandler.XLHandleManager;
import jakarta.transaction.Transactional;
import java.security.Principal;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.messaging.handler.annotation.DestinationVariable;
import org.springframework.messaging.handler.annotation.MessageExceptionHandler;
import org.springframework.messaging.simp.annotation.SendToUser;
import org.springframework.messaging.simp.annotation.SubscribeMapping;
import org.springframework.stereotype.Controller;
import org.springframework.web.server.ResponseStatusException;

@Controller
@Log4j2
public class SubscribeController {

  private final XLHandleManager handleManager;
  private final ModelRepository modelRepository;
  private final UserRepository userRepository;

  @Autowired
  public SubscribeController(
      XLHandleManager handleManager,
      ModelRepository modelRepository,
      UserRepository userRepository) {
    this.handleManager = handleManager;
    this.modelRepository = modelRepository;
    this.userRepository = userRepository;
  }

  // See TopicSubscriptionInterceptor for a list of valid topics.

  @MessageExceptionHandler
  @SendToUser("/topic/errors")
  public String handleExceptions(ResponseStatusException responseStatusException) {
    log.info("SENDING EXCEPTION TO USER");
    return responseStatusException.getReason();
  }

  @SubscribeMapping("/topic/model/{modelId}/{scenario}")
  @Transactional
  public CalculationData subscribingToModel(
      @DestinationVariable("modelId") Long modelId,
      @DestinationVariable("scenario") int scenario,
      Principal principal)
      throws ResponseStatusException {

    try {
      checkAccess(principal, modelId);
      return modelRepository
          .findById(modelId)
          .map(
              model -> {
                log.debug("Preparing data to websocket");
                CalculationData returnData =
                    handleManager.getXLInstanceForModel(model).getContent(SCENARIO.from(scenario));
                log.debug("Sending data to websocket");
                return returnData;
              })
          .orElse(null);
    } catch (IllegalArgumentException e) {
      log.error(e);
      throw new ResourceException(HttpStatus.BAD_REQUEST, "Wrong scenarioNumber");
    }
  }

  @SubscribeMapping("/topic/changelog/{modelId}")
  @Transactional
  public void subscribingToChangeLog(
      @DestinationVariable("modelId") Long modelId, Principal principal) {
    assertAccessAndRun(modelId, principal, () -> {});
    log.debug(principal.getName() + " subscribed to changelog :" + modelId);
  }

  /**
   * check user has access to model or throw exception
   *
   * @param principal is the user to be checked
   * @param modelId is the id of the model to be checked
   */
  private void checkAccess(Principal principal, Long modelId) throws ResponseStatusException {
    Model model =
        modelRepository
            .findById(modelId)
            .orElseThrow(
                () ->
                    new ResponseStatusException(
                        HttpStatus.BAD_REQUEST, "Couldn't find the requested model"));

    userRepository.findByModel(model).stream()
        .filter(user -> user.getEmail().equalsIgnoreCase(principal.getName()))
        .findFirst()
        .orElseThrow(
            () ->
                new ResponseStatusException(
                    HttpStatus.UNAUTHORIZED, "You do not have access to this model"));
  }

  private void assertAccessAndRun(Long modelId, Principal principal, Runnable action) {
    checkAccess(principal, modelId);
    action.run();
  }
}
