package app.valuationcontrol.webservice.model.area;

import app.valuationcontrol.webservice.helpers.DataTransformer;
import app.valuationcontrol.webservice.helpers.EntityDTOConverter;
import app.valuationcontrol.webservice.helpers.ModelProvider;
import app.valuationcontrol.webservice.model.Model;
import app.valuationcontrol.webservice.model.subarea.SubArea;
import app.valuationcontrol.webservice.model.subarea.SubAreaData;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.GenerationType;
import jakarta.persistence.Id;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.OneToMany;
import jakarta.persistence.OrderBy;
import jakarta.persistence.Table;
import jakarta.persistence.UniqueConstraint;
import java.util.ArrayList;
import java.util.List;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

@Getter
@Entity
@NoArgsConstructor
@Table(uniqueConstraints = {@UniqueConstraint(columnNames = {"attached_model_id", "area_name"})})
public class Area implements DataTransformer<AreaData>, ModelProvider {
  @Id
  @Setter
  @GeneratedValue(strategy = GenerationType.IDENTITY)
  private long id;

  @Setter @ManyToOne private Model attachedModel;

  @Setter
  @Column(name = "area_name")
  private String areaName;

  @Setter
  @Column(name = "area_description")
  private String areaDescription;

  @Setter
  @Column(name = "area_order")
  private Integer areaOrder;

  @OneToMany(
      mappedBy = "attachedArea",
      fetch = FetchType.LAZY,
      cascade = CascadeType.ALL,
      orphanRemoval = true)
  @OrderBy(value = "subAreaOrder ASC")
  private List<SubArea> subAreas = new ArrayList<>();

  public Area(AreaData areaData, Model model) {
    this.updateFrom(areaData);
    this.attachedModel = model;
    this.setAreaOrder(model.getAreas().size() + 1);
  }

  /**
   * Creates a new @area
   *
   * @param attachedModel the model to attach
   * @param areaName is the areaName of the area
   * @param description is the areaDescription of the area
   */
  public Area(Model attachedModel, String areaName, String description, Integer order) {
    this.attachedModel = attachedModel;
    this.areaName = areaName;
    this.areaDescription = description;
    this.areaOrder = order;
  }

  public Area(Area existingArea) {
    this.attachedModel = existingArea.getAttachedModel();
    this.areaName = existingArea.getAreaName();
    this.areaDescription = existingArea.getAreaDescription();
    this.areaOrder = existingArea.getAreaOrder();
    this.subAreas = existingArea.getSubAreas();
  }

  public AreaData asData() {
    final List<SubAreaData> subAreaData = EntityDTOConverter.asData(this.subAreas);
    return new AreaData(this.id, this.areaName, this.areaDescription, this.areaOrder, subAreaData);
  }

  @Override
  public Long getModelId() {
    return this.attachedModel.getId();
  }

  public void updateFrom(AreaData areaData) {
    this.areaName = areaData.areaName();
    this.areaDescription = areaData.areaDescription();
  }
}
