package app.valuationcontrol.webservice.model.sensitivity;

import static app.valuationcontrol.webservice.helpers.ModelChecker.inSameModel;
import static org.springframework.http.HttpStatus.CREATED;

import app.valuationcontrol.webservice.EntityService;
import app.valuationcontrol.webservice.model.Model;
import app.valuationcontrol.webservice.model.events.Event;
import app.valuationcontrol.webservice.model.events.Events;
import app.valuationcontrol.webservice.xlhandler.SCENARIO;
import app.valuationcontrol.webservice.xlhandler.ScenarioDataProvider;
import app.valuationcontrol.webservice.xlhandler.XLHandleManager;
import jakarta.transaction.Transactional;
import jakarta.validation.Valid;
import java.security.Principal;
import java.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
@Transactional
public class SensitivityController {

  private final XLHandleManager xlHandleManager;
  private final EntityService entityService;
  private final Events events;

  /** Initializes the Sensitivity controller and establish link to database */
  @Autowired
  public SensitivityController(
      XLHandleManager xlHandleManager, EntityService entityService, Events events) {
    this.xlHandleManager = xlHandleManager;
    this.entityService = entityService;
    this.events = events;
  }

  @PostMapping("/api/model/{modelId}/sensitivity")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<Long> addSensitivity(
      @PathVariable(value = "modelId") Model model,
      @Valid @RequestBody SensitivityData sensitivityData,
      Principal principal) {

    Sensitivity sensitivity = new Sensitivity(sensitivityData, model);
    model.getSensitivities().add(sensitivity);

    return entityService
        .safeCreate(Sensitivity.class, sensitivity)
        .map(
            createdSensitivity -> {
              Event<Sensitivity> event =
                  Event.created(this, createdSensitivity, principal, Sensitivity.class, model);
              events.publishCustomEvent(event);
              events.processEvents(principal);

              return new ResponseEntity<>(createdSensitivity.getId(), CREATED);
            })
        .orElse(ResponseEntity.badRequest().build());
  }

  @PutMapping("/api/model/{modelId}/sensitivity/{sensitivityId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> updateSensitivity(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "sensitivityId") Sensitivity existingSensitivity,
      @Valid @RequestBody SensitivityData sensitivityData,
      Principal principal) {

    if (!inSameModel(model, existingSensitivity)) {
      return ResponseEntity.badRequest().build();
    }

    Sensitivity oldSensitivity = new Sensitivity(existingSensitivity);
    existingSensitivity.updateWith(sensitivityData);

    Event<Sensitivity> event =
        Event.updated(
            this, oldSensitivity, existingSensitivity, principal, Sensitivity.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    return ResponseEntity.ok(String.valueOf(existingSensitivity.getId()));
  }

  @DeleteMapping("/api/model/{modelId}/sensitivity/{sensitivityId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> deleteSensitivity(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "sensitivityId") Sensitivity sensitivity,
      Principal principal) {

    if (!inSameModel(model, sensitivity)) {
      return ResponseEntity.badRequest().build();
    }

    model.getSensitivities().remove(sensitivity);

    Event<Sensitivity> event =
        Event.deleted(this, sensitivity, principal, Sensitivity.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    return ResponseEntity.ok().build();
  }

  @PostMapping("/api/model/{modelId}/sensitivity/run")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  public ResponseEntity<String> runSensitivities(
      @PathVariable(value = "modelId") Model model,
      @RequestParam(value = "scenario_number") Integer scenarioNumber,
      Principal principal) {

    ScenarioDataProvider xlInstance = this.xlHandleManager.getXLInstanceForModel(model);

    final int scenarioToRun = scenarioNumber != null ? scenarioNumber : 0;

    xlInstance.runSensitivities(model.getSensitivities(), SCENARIO.from(scenarioToRun));

    model
        .getSensitivities()
        .forEach(sensitivity -> sensitivity.setSensitivityLastRun(LocalDateTime.now()));

    Event<Model> event = Event.lightUpdated(this, model, principal, Model.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    return ResponseEntity.ok("");
  }
}
