package app.valuationcontrol.webservice.openai;

import static app.valuationcontrol.webservice.openai.OpenAIHelperFunctions.doRequest;
import static app.valuationcontrol.webservice.openai.OpenAIHelperFunctions.prepareSingleVariable;

import app.valuationcontrol.webservice.helpers.CalculationData;
import app.valuationcontrol.webservice.model.Model;
import app.valuationcontrol.webservice.model.ModelController;
import app.valuationcontrol.webservice.model.variable.Variable;
import app.valuationcontrol.webservice.model.variable.VariableController;
import app.valuationcontrol.webservice.xlhandler.SCENARIO;
import app.valuationcontrol.webservice.xlhandler.XLHandleManager;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.theokanning.openai.completion.chat.*;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import java.util.*;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RestController;

@RestController
@Log4j2
public class OpenAiController {

  private final XLHandleManager xlHandleManager;
  private final OpenAiServiceImplementation openAiServiceImplementation;

  @Autowired
  public OpenAiController(
      XLHandleManager xlHandleManager, OpenAiServiceImplementation openAiServiceImplementation) {
    this.xlHandleManager = xlHandleManager;
    this.openAiServiceImplementation = openAiServiceImplementation;
  }

  @Operation(
      summary = "Review the model using OpenAI",
      description = "Use this entrypoint to review the model's variable using OPENAI",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @GetMapping(
      value = "/api/model/{modelId}/variable/{variableId}/aiprompt",
      produces = {MediaType.APPLICATION_JSON_VALUE})
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  public ResponseEntity<JsonNode> prompt(
      @Parameter(
              description = ModelController.MODEL_ID_DESCRIPTION,
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = ModelController.MODEL_ID)
          Model model,
      @Parameter(
              description = VariableController.VARIABLE_ID_DESCRIPTION,
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = VariableController.VARIABLE_ID)
          Variable variable,
      @Schema(type = "String") String prompt) {
    CalculationData calculationData =
        this.xlHandleManager.getXLInstanceForModel(model).getContent(SCENARIO.BASE);
    ObjectNode returnObject = prepareSingleVariable(calculationData, model, variable, true);

    ChatMessage systemMessage = new ChatMessage(ChatMessageRole.SYSTEM.value(), prompt);
    ChatMessage firstMsg =
        new ChatMessage(ChatMessageRole.USER.value(), returnObject.toPrettyString());

    List<ChatMessage> messages = new ArrayList<>();
    messages.add(systemMessage);
    messages.add(firstMsg);
    return ResponseEntity.ok(doRequest(openAiServiceImplementation, messages, null, null));
  }

  @Operation(
      summary = "Review the model using OpenAI",
      description = "Use this entrypoint to review the model's variable using OPENAI",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @GetMapping(
      value = "/api/model/{modelId}/variable/{variableId}/review",
      produces = {MediaType.APPLICATION_JSON_VALUE})
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  public ResponseEntity<JsonNode> reviewVariable(
      @Parameter(
              description = ModelController.MODEL_ID_DESCRIPTION,
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = ModelController.MODEL_ID)
          Model model,
      @Parameter(
              description = VariableController.VARIABLE_ID_DESCRIPTION,
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = VariableController.VARIABLE_ID)
          Variable variable) {
    CalculationData calculationData =
        this.xlHandleManager.getXLInstanceForModel(model).getContent(SCENARIO.BASE);
    ObjectNode returnObject = prepareSingleVariable(calculationData, model, variable, true);

    String question =
        "You are a financial analyst reviewing the soundness of a financial model provided in JSON. Comment on the consistency of the forecast using a mathematical approach. Forecasted values are in the field 'values' or 'value' if the variable is a constant";

    ChatMessage systemMessage = new ChatMessage(ChatMessageRole.SYSTEM.value(), question);
    ChatMessage firstMsg =
        new ChatMessage(ChatMessageRole.USER.value(), returnObject.toPrettyString());

    List<ChatMessage> messages = new ArrayList<>();
    messages.add(systemMessage);
    messages.add(firstMsg);
    return ResponseEntity.ok(
        doRequest(
            openAiServiceImplementation,
            messages,
            openAiServiceImplementation.getReviewFunction(),
            "get_review"));
  }
}
