package app.valuationcontrol.webservice.securityhelpers;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.security.Keys;
import java.util.Date;
import java.util.function.Function;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

@Component
public class JwtTokenUtil {

  private final Environment environment;

  @Autowired
  private JwtTokenUtil(Environment environment) {
    this.environment = environment;
  }

  private byte[] getSecret() {
    // @PropertySource("classpath:application.properties")
    // env is an Environment variable

    String secret = environment.getProperty("jwt.secret");

    if (secret == null) {
      throw new IllegalStateException("No secret configured in key jwt.secret");
    }

    return secret.getBytes();
  }

  // retrieve username from jwt token
  public String getUsernameFromToken(String token) {
    return getClaimFromToken(token, Claims::getSubject);
  }

  public <T> T getClaimFromToken(String token, Function<Claims, T> claimsResolver) {
    final Claims claims = getAllClaimsFromToken(token);
    if (claims != null) {
      return claimsResolver.apply(claims);
    } else {
      return null;
    }
  }

  // for retrieveing any information from token we will need the secret key
  private Claims getAllClaimsFromToken(String token) {
    Claims returnClaims;
    try {
      returnClaims =
          Jwts.parserBuilder().setSigningKey(getSecret()).build().parseClaimsJws(token).getBody();
    } catch (ExpiredJwtException E) {
      returnClaims = null;
    }
    return returnClaims;
  }

  public String generateToken(String username, Date expiration) {
    return Jwts.builder()
        .setSubject(username)
        .setExpiration(expiration)
        .signWith(Keys.hmacShaKeyFor(this.getSecret()))
        .compact();
  }
}
