package app.valuationcontrol.webservice.xlhandler;

import static app.valuationcontrol.webservice.xlhandler.CellValueHelper.cellValueOf;
import static app.valuationcontrol.webservice.xlhandler.CellValueHelper.generateAddress;
import static app.valuationcontrol.webservice.xlhandler.VariableEvaluator.AGGREGATION_SEGMENT;
import static app.valuationcontrol.webservice.xlhandler.VariableEvaluator.EQUALS;
import static app.valuationcontrol.webservice.xlhandler.VariableEvaluator.evaluateVariable;
import static java.lang.Math.max;
import static java.lang.String.valueOf;
import static java.text.MessageFormat.format;
import static java.util.Arrays.copyOfRange;

import app.valuationcontrol.webservice.helpers.CalculationData;
import app.valuationcontrol.webservice.helpers.CellError;
import app.valuationcontrol.webservice.helpers.DataPeriod;
import app.valuationcontrol.webservice.model.Model;
import app.valuationcontrol.webservice.model.sensitivity.Sensitivity;
import app.valuationcontrol.webservice.model.sensitivity.SensitivityResult;
import app.valuationcontrol.webservice.model.variable.Variable;
import app.valuationcontrol.webservice.model.variablevalue.VariableValue;
import java.io.InputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import lombok.Getter;
import lombok.extern.log4j.Log4j2;

@Log4j2
public class XLInstance implements ScenarioDataProvider {

  public static final String INPUT = "input";
  public static final String PERCENT = "percent";

  public static final int YTD_PERIOD = -500; // Implement the same value in frontend
  private final CalcDocument calcDocument;

  private final ModelChangeNotifier modelChangeNotifier;

  @Getter private Model attachedModel;

  /*Stored in memory as they are computed by the XL underlying file */
  private final Map<Long, List<CellError>> formulaErrors = new ConcurrentHashMap<>();
  private final Map<SCENARIO, Map<Long, List<CellError>>> variableCellErrorsByScenario =
      new ConcurrentHashMap<>();

  private final Map<SCENARIO, Object[][]> cacheValues = new ConcurrentHashMap<>();

  @Getter private List<SensitivityResult> sensitivityResults = new ArrayList<>();
  private final AtomicLong lastUsed = new AtomicLong();

  /** Connects the XLManager to a new Calc document */
  public XLInstance(
      Model attachedModel, CalcDocument calcDocument, ModelChangeNotifier modelChangeNotifier) {
    // Stores the attached model
    this.attachedModel = attachedModel;
    this.calcDocument = calcDocument;
    this.modelChangeNotifier = modelChangeNotifier;

    lastUsed.set(System.currentTimeMillis());

    reloadWholeBaseContentIfNoCache();
  }

  @Override
  public InputStream saveAs() {
    this.calcDocument.assertConnected(
        () -> {
          log.debug("saving document");
          cacheValues.clear();
          reloadWholeBaseContentIfNoCache();
        });

    return this.calcDocument.saveAs();
  }

  public void reloadAndUpdateClients() {
    log.debug("Entering reloadAndUpdateClients " + calcDocument);
    reloadWholeBaseContentIfNoCache();
    this.modelChangeNotifier
        .getLoadedScenario(this.attachedModel.getId())
        .forEach(
            scenarioNumber -> {
              log.debug("Reload and update clients : Content was reloaded for scenario Base ");
              this.sendUpdateToClients(SCENARIO.from(scenarioNumber));
            });
    log.debug("Exiting reloadAndUpdateClients" + calcDocument);
  }

  /**
   * Loads data from {@link VariableValue} and override manually the formula for a single variable
   * value Also applied for historical values
   */
  public void setSingleValue(VariableValue variableValue) {
    if (variableValue != null && variableValue.getValue() != null) {
      insertIntoCell(SCENARIO.from(variableValue.getScenarioNumber()), cellValueOf(variableValue));
    }
  }

  @Override
  public void runSensitivities(List<Sensitivity> sensitivities, SCENARIO scenario) {

    this.calcDocument.assertConnected(
        () -> {
          log.debug("CacheValues: Force Reloading base");
          cacheValues.clear();
          reloadWholeBaseContentIfNoCache();
        });

    SensitivityRunner sensitivityRunner =
        new SensitivityRunner(this.calcDocument, sensitivities, scenario, this.attachedModel);
    this.sensitivityResults = sensitivityRunner.runSensitivities();
    // Users are updated through events
  }

  public void updateCacheValues(SCENARIO scenario) {

    if (attachedModel.getVariables().isEmpty()) {
      return;
    }

    log.debug("Entering updateCacheValues" + calcDocument);
    try {
      lastUsed.set(System.currentTimeMillis());
      long millis = System.currentTimeMillis();

      if (!scenario.equals(SCENARIO.BASE) && cacheValues.get(SCENARIO.BASE) == null) {
        reloadWholeBaseContentIfNoCache();
        // updateCacheValues(SCENARIO.BASE);
      }

      this.calcDocument.assertConnected(
          () -> {
            log.debug("CacheValues: Force Reloading base");
            cacheValues.clear();
            reloadWholeBaseContentIfNoCache();
          });
      log.debug("CacheValues: Opening sheet in cacheValues " + scenario);

      GenericSheet scenarioGenericSheet = calcDocument.getSheet(scenario);
      setScenarioVariableValues(scenario, scenarioGenericSheet);
      int endRow = attachedModel.getNumberOfVariables() - 1;
      Object[][] xlObject =
          scenarioGenericSheet.getValues(attachedModel.indexOfLastColumn(), endRow);
      cacheValues.put(scenario, xlObject);

      log.debug(
          "CacheContent for scenario {} was defined in {}",
          scenario,
          (System.currentTimeMillis() - millis) + "ms");
      log.debug("Exiting updateCacheValues" + calcDocument);
    } catch (Exception e) {
      log.error("Couldn't get content " + e);
    }
  }

  /**
   * @return the content of the XL file for a certain scenario
   */
  @Override
  public synchronized CalculationData getContent(SCENARIO scenario) {
    log.debug("Entering getContent" + calcDocument);

    lastUsed.set(System.currentTimeMillis());
    long millis = System.currentTimeMillis();

    // Open the document and return the specified area
    try {

      if (attachedModel.getVariables().isEmpty()) {
        return new CalculationData(
            this.attachedModel,
            new ArrayList<>(),
            new HashMap<>(),
            new HashMap<>(),
            scenario.ordinal());
      }

      //
      if (cacheValues.get(scenario) == null) {
        log.debug("updating cacheValues in getContent");
        updateCacheValues(scenario);
      }

      Object[][] xlObject = cacheValues.get(scenario);
      log.debug("CacheValues is of size" + xlObject.length);

      for (Variable variable : attachedModel.getVariables()) {

        int row = variable.getRow();

        // If single or constant then only pick those values

        if (variable.isSingleOrConstantValue()) {
          ArrayList<Object> singleOrConstantValues = new ArrayList<>();
          singleOrConstantValues.add(xlObject[row][attachedModel.getConstantColumn()]);

          if (variable.isModelledAtSegment()) {
            for (int i = 1; i <= this.attachedModel.getSegments().size(); i++) {
              int column = variable.columnOfSegmentAndPeriod(i - 1, 0);
              singleOrConstantValues.add(xlObject[row][column]);
            }
          }

          variable.setSingleOrConstantValue(singleOrConstantValues);
        } else {
          ArrayList<Object[]> historicalValues = new ArrayList<>();
          ArrayList<Object[]> projectionValues = new ArrayList<>();

          if (this.attachedModel.isIncludeYTD()) {
            int column =
                variable.columnOfSegmentAndPeriod(-1, this.attachedModel.getNbProjectionPeriod());
            variable.getYearToDateValue().add(xlObject[row][column]);
            variable.getYearToGoValue().add(xlObject[row][column + 1]);
          }

          /*Adding variables from main segment */
          historicalValues.add(
              copyOfRange(
                  xlObject[row],
                  variable.columnOfSegmentAndPeriod(
                      -1, -this.attachedModel.getNbHistoricalPeriod()),
                  variable.columnOfSegmentAndPeriod(-1, 0)));

          projectionValues.add(
              copyOfRange(
                  xlObject[row],
                  variable.columnOfSegmentAndPeriod(-1, 0),
                  variable.columnOfSegmentAndPeriod(
                      -1, this.attachedModel.getNbProjectionPeriod())));

          if (variable.isModelledAtSegment()) {
            for (int i = 0; i < attachedModel.getSegments().size(); i++) {

              if (this.attachedModel.isIncludeYTD()) {
                int column =
                    variable.columnOfSegmentAndPeriod(
                        i, this.attachedModel.getNbProjectionPeriod());
                variable.getYearToDateValue().add(xlObject[row][column]);
                variable.getYearToGoValue().add(xlObject[row][column + 1]);
              }

              var historicalStart =
                  variable.columnOfSegmentAndPeriod(i, -this.attachedModel.getNbHistoricalPeriod());
              var historicalEnd = variable.columnOfSegmentAndPeriod(i, 0);

              historicalValues.add(copyOfRange(xlObject[row], historicalStart, historicalEnd));

              var projectionStart = variable.columnOfSegmentAndPeriod(i, 0);
              var projectionEnd =
                  variable.columnOfSegmentAndPeriod(i, this.attachedModel.getNbProjectionPeriod());

              projectionValues.add(copyOfRange(xlObject[row], projectionStart, projectionEnd));
            }
          }
          variable.setHistoricalValues(historicalValues);
          variable.setProjectionValues(projectionValues);
        }
        // Checking cell errors
        getCellErrors(variable, scenario);
      }

      CalculationData returnObject =
          new CalculationData(
              this.attachedModel,
              this.getSensitivityResults(),
              this.formulaErrors,
              this.variableCellErrorsByScenario.get(scenario),
              scenario.ordinal());
      log.debug(
          "Content for scenario {} was fetched in {}",
          scenario,
          (System.currentTimeMillis() - millis) + "ms");
      lastUsed.set(System.currentTimeMillis());
      log.debug("Exiting getContent" + calcDocument);
      return returnObject;

    } catch (Exception e) {
      log.error("Couldn't get content " + e);
    }
    return null;
  }

  private void setScenarioVariableValues(SCENARIO scenario, GenericSheet scenarioGenericSheet) {
    final List<GenericSheet.CellValue> cellValues =
        this.attachedModel.getVariables().stream()
            .flatMap(
                variable ->
                    variable.getVariableValuesForAScenario(scenario).stream()
                        .filter(v -> v.getValue() != null)
                        .filter(
                            v -> // Exclude YTD variableValues if model has no YTD
                            (v.getAttachedVariable().getAttachedModel().isIncludeYTD()
                                    || !Objects.requireNonNullElse(v.getPeriod(), 0)
                                        .equals(YTD_PERIOD)))
                        .map(CellValueHelper::cellValueOf))
            .toList();
    // calcDocument.assertConnected(this::reloadWholeContent);
    if (!cellValues.isEmpty()) {
      scenarioGenericSheet.setCellValues(cellValues.toArray(new GenericSheet.CellValue[0]));
    }
  }

  /**
   * Generates an array containing the metadata, the historical and projection values, including the
   * variableValues
   *
   * @param variable the variable to generate array for
   * @return an array containing all calculates values without variable values
   */
  public String[] generateVariableArray(Variable variable) {

    String[] dumpArrayRow = new String[variable.getAttachedModel().getArraySize()];

    /*Clearing the old validation*/
    formulaErrors.remove(variable.getId());
    /*Loading metadata */
    initVariableRow(dumpArrayRow, variable);
    boolean segmentExists = checkSegmentsExist(variable);

    if (variable.isSingleOrConstantValue()) {
      performSingleEntryReplication(variable, dumpArrayRow);
      String summaryFormula = EQUALS;
      if (segmentExists && !variable.getVariableFormat().equals("date")) {
        String divider =
            variable.getVariableFormat().equals(PERCENT)
                ? String.valueOf(Math.max(1, attachedModel.getSegments().size()))
                : "1";

        for (int j = 0; j < attachedModel.getSegments().size(); j++) {
          summaryFormula =
              format(
                  "{0}+{1}/{2}",
                  summaryFormula,
                  generateAddress(variable, variable.columnOfSegmentAndPeriod(j, 1)),
                  divider);
        }
        dumpArrayRow[variable.getPrimaryColumn()] = summaryFormula;
      }

    } else {
      boolean performHistoricalReplication = shouldReplicateHistorical(variable);
      /*Performing Historical Replication */
      if (performHistoricalReplication) {
        performHistoricalReplication(variable, dumpArrayRow, segmentExists);
      }
      /*Performing Forecast Replication */
      performForecastReplication(variable, dumpArrayRow, segmentExists);

      if (this.attachedModel.isIncludeYTD()) {
        performYTDReplication(variable, dumpArrayRow, segmentExists);
      }

      // Generating aggregation for historicals && projections
      if (segmentExists
          && !variable.getVariableFormat().equals("date")
          && !variable.getVariableType().equals("input")) {

        int start = -this.getAttachedModel().getNbHistoricalPeriod();
        int stop = this.getAttachedModel().getNbProjectionPeriod();

        for (int i = start; i < stop; i++) {
          String summaryFormula = EQUALS;
          if (variable.isPercentOrKPI()) { // If variable is a percent or KPI , then compute
            DataPeriod segmentConstantPeriod =
                new DataPeriod(this.getAttachedModel(), i, 0, variable.isSingleOrConstantValue());
            summaryFormula = evaluateVariable(variable, segmentConstantPeriod, formulaErrors);
          } else { // Else aggregate as sum of segments
            for (int j = 0; j < attachedModel.getSegments().size(); j++) {
              summaryFormula =
                  format(
                      "{0}+{1}",
                      summaryFormula,
                      generateAddress(variable, variable.columnOfSegmentAndPeriod(j, i)));
            }
          }
          // Updating historical summary
          dumpArrayRow[variable.getPrimaryColumn() + i] = summaryFormula;
        }
        // Checking for % calculations based on inputs at aggregated level
        if (variable.getVariableFormat().equals(PERCENT)) {
          // Only assesses for percent formatted variable, modelled at segment in aggregation
          // (segmentFactor=0)
          variable.getVariableDependencies().stream()
              .filter(variable1 -> variable1.getVariableType().equals(INPUT))
              .forEach(
                  variable2 -> {
                    CellError e =
                        new CellError(
                            0,
                            "Dependency to input",
                            "You should change the formula to avoid the use of input variables ("
                                + variable2.getVariableName()
                                + " ) in an aggregated segment value");
                    formulaErrors.computeIfAbsent(variable.getId(), l -> new ArrayList<>()).add(e);
                  });
        }
      }
    }

    return dumpArrayRow;
  }

  private void performSingleEntryReplication(Variable variable, String[] dumpArrayRow) {
    // For single and constant values
    if (variable.isSingleOrConstantValue()) {
      if (variable.isModelledAtSegment()
          && this.attachedModel.getSegments() != null
          && !this.attachedModel.getSegments().isEmpty()) {
        for (int i = 1; i <= this.attachedModel.getSegments().size(); i++) {
          DataPeriod segmentConstantPeriod =
              new DataPeriod(this.getAttachedModel(), 0, i, variable.isSingleOrConstantValue());
          final int cellColumn = variable.columnOfSegmentAndPeriod(i - 1, 0);
          dumpArrayRow[cellColumn] =
              evaluateVariable(variable, segmentConstantPeriod, formulaErrors);
        }
      } else {
        DataPeriod dataPeriod =
            new DataPeriod(this.getAttachedModel(), 0, 0, variable.isSingleOrConstantValue());
        dumpArrayRow[variable.columnOfSegmentAndPeriod(AGGREGATION_SEGMENT, 0)] =
            evaluateVariable(variable, dataPeriod, formulaErrors);
      }
    }
  }

  private void performYTDReplication(
      Variable variable, String[] dumpArrayRow, boolean segmentReplicationShouldBePerformed) {
    int YTDperiod = attachedModel.getNbProjectionPeriod();
    int YTGperiod = YTDperiod + 1;

    if (segmentReplicationShouldBePerformed) {
      for (int segmentIndex = 0;
          segmentIndex < attachedModel.getSegments().size();
          segmentIndex++) {

        String ytdFormula = null;
        String ytgFormula;
        if (variable.isPercentOrKPI()
            || variable.isTypeTotal()) { // If variable is a percent or KPI , then compute

          DataPeriod YTDdataPeriod =
              new DataPeriod(
                  this.getAttachedModel(),
                  YTDperiod,
                  segmentIndex + 1,
                  variable.isSingleOrConstantValue());

          DataPeriod YTGdataPeriod =
              new DataPeriod(
                  this.getAttachedModel(),
                  YTGperiod,
                  segmentIndex + 1,
                  variable.isSingleOrConstantValue());

          ytdFormula = evaluateVariable(variable, YTDdataPeriod, formulaErrors);
          ytgFormula = evaluateVariable(variable, YTGdataPeriod, formulaErrors);
        } else {
          ytgFormula =
              EQUALS
                  + format(
                      "{0}-{1}",
                      generateAddress(variable, variable.columnOfSegmentAndPeriod(segmentIndex, 0)),
                      generateAddress(
                          variable,
                          variable.columnOfSegmentAndPeriod(segmentIndex, YTGperiod - 1)));
        }
        if (ytdFormula != null)
          dumpArrayRow[variable.columnOfSegmentAndPeriod(segmentIndex, YTDperiod)] = ytdFormula;
        dumpArrayRow[variable.columnOfSegmentAndPeriod(segmentIndex, YTGperiod)] = ytgFormula;
      }
    } else {
      String ytdFormula = null;
      String ytgFormula;

      if (variable.isPercentOrKPI()
          || variable.isTypeTotal()) { // If variable is a percent or KPI , then compute
        DataPeriod YTDdataPeriod =
            new DataPeriod(
                this.getAttachedModel(), YTDperiod, 0, variable.isSingleOrConstantValue());
        DataPeriod YTGdataPeriod =
            new DataPeriod(
                this.getAttachedModel(), YTGperiod, 0, variable.isSingleOrConstantValue());
        ytdFormula = evaluateVariable(variable, YTDdataPeriod, formulaErrors);
        ytgFormula = evaluateVariable(variable, YTGdataPeriod, formulaErrors);
      } else {
        ytgFormula =
            EQUALS
                + format(
                    "{0}-{1}",
                    generateAddress(
                        variable, variable.columnOfSegmentAndPeriod(AGGREGATION_SEGMENT, 0)),
                    generateAddress(
                        variable,
                        variable.columnOfSegmentAndPeriod(AGGREGATION_SEGMENT, YTGperiod - 1)));
      }
      if (ytdFormula != null)
        dumpArrayRow[variable.columnOfSegmentAndPeriod(AGGREGATION_SEGMENT, YTDperiod)] =
            ytdFormula;
      dumpArrayRow[variable.columnOfSegmentAndPeriod(AGGREGATION_SEGMENT, YTGperiod)] = ytgFormula;
    }
  }

  private void performForecastReplication(
      Variable variable, String[] dumpArrayRow, boolean segmentReplicationShouldBePerformed) {

    for (int period = 0; period < this.getAttachedModel().getNbProjectionPeriod(); period++) {
      if (segmentReplicationShouldBePerformed) {
        for (int segmentIndex = 0;
            segmentIndex < attachedModel.getSegments().size();
            segmentIndex++) {

          DataPeriod dataPeriod =
              new DataPeriod(
                  this.getAttachedModel(),
                  period,
                  segmentIndex + 1,
                  variable.isSingleOrConstantValue());
          dumpArrayRow[variable.columnOfSegmentAndPeriod(segmentIndex, period)] =
              evaluateVariable(variable, dataPeriod, formulaErrors);
        }
      } else {
        DataPeriod dataPeriod =
            new DataPeriod(this.getAttachedModel(), period, 0, variable.isSingleOrConstantValue());
        dumpArrayRow[variable.columnOfSegmentAndPeriod(AGGREGATION_SEGMENT, period)] =
            evaluateVariable(variable, dataPeriod, formulaErrors);
      }
    }
  }

  private void performHistoricalReplication(
      Variable variable, String[] dumpArrayRow, boolean segmentReplicationShouldBePerformed) {

    for (int period = -this.getAttachedModel().getNbHistoricalPeriod(); period < 0; period++) {
      if (segmentReplicationShouldBePerformed) {
        for (int segmentIndex = 0;
            segmentIndex < attachedModel.getSegments().size();
            segmentIndex++) {
          DataPeriod dataPeriod =
              new DataPeriod(
                  this.getAttachedModel(),
                  period,
                  segmentIndex + 1,
                  variable.isSingleOrConstantValue());
          final int column = variable.columnOfSegmentAndPeriod(segmentIndex, period);
          dumpArrayRow[column] = evaluateVariable(variable, dataPeriod, formulaErrors);
        }
        // Updating historical summary

      } else {
        DataPeriod dataPeriod =
            new DataPeriod(this.getAttachedModel(), period, 0, variable.isSingleOrConstantValue());
        dumpArrayRow[variable.columnOfSegmentAndPeriod(-1, period)] =
            evaluateVariable(variable, dataPeriod, formulaErrors);
      }
    }
  }

  /**
   * Inits the first 9 columns of the variable row
   *
   * @param dumpArrayRow the array that contains all the values
   * @param variable the variable to be written down in the dumpArray
   */
  private void initVariableRow(String[] dumpArrayRow, Variable variable) {
    dumpArrayRow[0] = valueOf(variable.getId());
    dumpArrayRow[1] = variable.getVariableName();
    dumpArrayRow[2] = variable.getVariableFormula();
    dumpArrayRow[3] = variable.getVariableType();
    dumpArrayRow[4] = valueOf(variable.getVariableArea().getId());
    dumpArrayRow[5] = valueOf(variable.getVariableSubArea().getId());
    dumpArrayRow[6] = variable.getVariableFormat();
    dumpArrayRow[7] = valueOf(variable.getVariableOrder());
    dumpArrayRow[8] = valueOf(variable.getVariableDepth());
  }

  private boolean shouldReplicateHistorical(Variable myVariable) {
    return this.getAttachedModel().getNbHistoricalPeriod() > 0
        && (myVariable.isVariableApplyToHistoricals()
            || myVariable.getVariableType().contains("total")
            || myVariable.getVariableType().contains("kpi"));
  }

  private boolean checkSegmentsExist(Variable myVariable) {
    return myVariable.isModelledAtSegment() && !this.attachedModel.getSegments().isEmpty();
  }

  /** Removes all sheets apart from the "main" sheet (renamed in the constructor of this class) */
  public synchronized void cleanAndUpdateAdditionalSheets() {
    log.debug("Entering cleanAndUpdateAdditionalSheets" + calcDocument);
    calcDocument.closeAllButBase();
    modelChangeNotifier.getLoadedScenario(this.attachedModel.getId()).stream()
        .filter(scenarioNumber -> !scenarioNumber.equals(SCENARIO.BASE.ordinal()))
        .forEach(
            scenarioNumber -> {
              log.debug("Reloading closed sheet for scenario " + scenarioNumber);
              SCENARIO scenario = SCENARIO.from(scenarioNumber);
              log.debug("Refreshing scenario " + scenario);
              updateCacheValues(
                  scenario); // Update to all listening clients is performing for each request
            });
    log.debug("Exiting cleanAndUpdateAdditionalSheets" + calcDocument);
  }

  /**
   * Refreshes the scenarioSheetName to connected clients
   *
   * @param scenario is the SCENARIO to be updated
   */
  private void sendUpdateToClients(final SCENARIO scenario) {

    CalculationData calculationData = getContent(scenario);
    // modelChangeNotifier.getLoadedScenario(this.attachedModel.getId());
    modelChangeNotifier.updateUsersModel(calculationData);
  }

  private void insertIntoCell(SCENARIO scenario, GenericSheet.CellValue cellValue) {
    GenericSheet scenarioGenericSheet = calcDocument.getSheet(scenario);
    scenarioGenericSheet.setCellValues(cellValue);
  }

  /** Use this function to refresh a single variableValue (efficiency optimization) */
  public void refreshValue(Variable myVariable, VariableValue myVariableValue) {
    long millis = System.currentTimeMillis();
    calcDocument.preventScreenUpdating(true);

    this.calcDocument.assertConnected(
        () -> {
          log.debug("refreshValue: Force Reloading base");
          cacheValues.clear();
          reloadWholeBaseContentIfNoCache();
        });

    // evaluate the model logic and evaluate the variable logic for each variable
    // !!!!Removed the constant constraint
    if (Objects.equals(myVariableValue.getScenarioNumber(), SCENARIO.BASE.ordinal())) {
      reloadVariableAndUpdateCache(
          myVariable); // Closes additional sheets to reload content if changed value
    } else {
      // Override the values if a VariableValue exists
      cacheValues.remove(SCENARIO.from(myVariableValue.getScenarioNumber()));
      setSingleValue(myVariableValue);
    }
    calcDocument.preventScreenUpdating(false);

    log.debug("Model was refreshed in : " + (System.currentTimeMillis() - millis) + "ms");
  }

  public void reloadVariableAndUpdateCache(Variable variable) {
    log.debug("Entering reloadVariableAndUpdateCache " + calcDocument);

    this.calcDocument.assertConnected(
        () -> {
          log.debug("reloadVariableAndUpdateCache: Force Reloading base");
          cacheValues.clear();
          reloadWholeBaseContentIfNoCache();
        });

    int arraySize = variable.getAttachedModel().getArraySize();
    String[][] dumpArray = new String[1][arraySize];

    /*Initializing array other setDataArray crashes */
    for (int j = 0; j < arraySize; j++) {
      dumpArray[0][j] = "";
    }

    calcDocument.preventScreenUpdating(true);
    /*Generating logic and variableValues for main scenario*/
    dumpArray[0] = generateVariableArray(variable);
    GenericSheet baseGenericSheet = calcDocument.getSheet(SCENARIO.BASE);

    baseGenericSheet.setRowValues(variable.getRow(), dumpArray[0]);

    cacheValues.clear();
    this.updateCacheValues(SCENARIO.BASE);
    // Clean the sheets in background
    cleanAndUpdateAdditionalSheets();
    calcDocument.preventScreenUpdating(false);
    log.debug("Exiting reloadVariableAndUpdateCache " + calcDocument);
  }

  /**
   * Use this function to refresh the content Loads the metadata and the variable logic which is
   * replicated for each segment (including "main" segment)
   *
   * <p>return list of open sheets before reload
   */
  public void reloadWholeBaseContentIfNoCache() {
    log.debug("Entering reloadWholeBaseContentIfNoCache " + calcDocument);

    if (cacheValues.get(SCENARIO.BASE) == null) {
      int numberOfVariables = this.getAttachedModel().getNumberOfVariables();
      long millis = System.currentTimeMillis();

      if (numberOfVariables > 0) {
        int arraySize = this.getAttachedModel().getArraySize();
        String[][] dumpArray = new String[numberOfVariables][arraySize];
        /*Initializing array other setDataArray crashes */
        for (int i = 0; i < numberOfVariables; i++) {
          for (int j = 0; j < arraySize; j++) {
            dumpArray[i][j] = "";
          }
        }

        // For libreoffice only
        calcDocument.preventScreenUpdating(true);
        calcDocument.closeAllSheets();
        // Resetting the error_log prior to refresh
        formulaErrors.clear();

        long before = System.currentTimeMillis();

        // loading main model
        for (Variable myVariable : attachedModel.getVariables()) {
          dumpArray[myVariable.getRow()] = generateVariableArray(myVariable);
        }

        log.debug("Logic was loaded in : " + (System.currentTimeMillis() - before) + "ms");

        GenericSheet baseGenericSheet = calcDocument.getSheet(SCENARIO.BASE);
        baseGenericSheet.setValue(
            dumpArray, max(arraySize - 1, 0), max(attachedModel.getNumberOfVariables() - 1, 0));
        setScenarioVariableValues(SCENARIO.BASE, baseGenericSheet);

        updateCacheValues(SCENARIO.BASE);
        calcDocument.preventScreenUpdating(false);
        log.debug(
            "Model was refreshed and cached in : " + (System.currentTimeMillis() - millis) + "ms");
      } else {
        log.debug("No variables so model was not reloaded");
      }
    }
  }

  /**
   * Check if all cells for the variable contain no errors
   *
   * @param myVariable Add errors to variable.validationError list
   */
  private void getCellErrors(Variable myVariable, SCENARIO scenario) {

    GenericSheet genericSheet = calcDocument.getSheet(scenario);

    /*Clearing the old validation for this variable*/
    List<CellError> cellErrors = new ArrayList<>();
    variableCellErrorsByScenario
        .computeIfAbsent(scenario, i -> new ConcurrentHashMap<>())
        .put(myVariable.getId(), cellErrors);

    int historicalPeriods =
        myVariable.isSingleOrConstantValue() ? 0 : this.attachedModel.getNbHistoricalPeriod();
    int startIndex = myVariable.getPrimaryColumn() - historicalPeriods;

    for (int periodIndex = -historicalPeriods, i = startIndex;
        i < myVariable.getAttachedModel().getArraySize();
        i++, periodIndex++) {
      try {
        String error = genericSheet.getErrorInCell(i, myVariable.getRow());

        // log.debug("Checking cell {} - {} with errorCode {}", myVariable.getRow(), i, error);
        if (error != null) {
          // log.debug("Found error : {}", error);
          CellError e = new CellError(periodIndex, "CalculationError", error);
          cellErrors.add(e);
        }
      } catch (Exception e) {
        log.debug(e);
      }
    }
  }

  public void setAttachedModel(Model attachedModel) {
    this.attachedModel = attachedModel;
  }

  @Override
  public boolean isUnusedFor(Duration inactivityThreshold) {
    return (System.currentTimeMillis() - inactivityThreshold.toMillis()) > lastUsed.get();
  }

  @Override
  public void sleep() {
    calcDocument.close();
  }

  @Override
  public void clearCacheAndReloadAndUpdateClients() {
    cacheValues.clear();
    this.reloadAndUpdateClients();
  }
}
