package app.valuationcontrol.webservice.securityhelpers;

import jakarta.servlet.http.HttpServletRequest;
import java.util.List;
import lombok.extern.log4j.Log4j2;
import org.postgresql.util.PSQLException;
import org.postgresql.util.PSQLState;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;

@ControllerAdvice
@Log4j2
public class GlobalExceptionHandlers {
  @ExceptionHandler(MethodArgumentNotValidException.class)
  public ResponseEntity<List<org.springframework.validation.FieldError>>
      handleMethodArgumentNotValidException(
          HttpServletRequest request, MethodArgumentNotValidException ex) {

    log.error("Could not create method argument", ex);

    BindingResult result = ex.getBindingResult();

    List<org.springframework.validation.FieldError> fieldErrors = result.getFieldErrors();

    return ResponseEntity.badRequest().body(fieldErrors);
  }

  @ExceptionHandler(NullPointerException.class)
  public ResponseEntity<String> handleNullPointerException(
      HttpServletRequest request, NullPointerException exception) {
    return ResponseEntity.badRequest().body(exception.getMessage());
  }

  @ExceptionHandler(DataIntegrityViolationException.class)
  public ResponseEntity<Object> handleSqlException(DataIntegrityViolationException ex) {
    if (ex.getCause() != null
        && ex.getCause().getCause() != null
        && ex.getCause().getCause() instanceof PSQLException psqlException
        && PSQLState.UNIQUE_VIOLATION.getState().equals(psqlException.getSQLState())) {
      if (psqlException.getServerErrorMessage() != null
          && psqlException.getServerErrorMessage().getTable() != null) {
        final String table = psqlException.getServerErrorMessage().getTable();
        return ResponseEntity.badRequest()
            .body(
                "A "
                    + ("area".equals(table) ? "zone" : table)
                    + " with that same name already exists");
      }
    }
    return ResponseEntity.badRequest().body(ex.getCause());
  }
}
