package ch.admin.bit.jeap.deploymentlog.docgen.service;

import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import net.javacrumbs.shedlock.core.LockConfiguration;
import net.javacrumbs.shedlock.core.LockProvider;
import net.javacrumbs.shedlock.core.SimpleLock;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.Optional;

/**
 * Ensures that mutually-exclusive docgen jobs (generating pages for the same system) are executed one after each
 * other, even when the app is running on multiple instances.
 */
@Component
@Slf4j
class DocgenLocks {

    private static final String LOCK_NAME_PREFIX = "docgen-";
    private static final Duration LOCK_RETRY_WAIT_DURATION = Duration.ofSeconds(3);
    // Give docgen at most one minute to finish, then assume it has been interrupted without releasing the lock
    private static final Duration LOCK_AT_MOST_FOR = Duration.ofMinutes(1);
    private static final Duration LOCK_AT_LEAST_FOR = Duration.ZERO;
    // Wait at most this duration until giving up trying to acquire the lock
    private Duration tryAcquireTimeout = Duration.ofMinutes(3);

    private final LockProvider lockProvider;

    DocgenLocks(LockProvider lockProvider) {
        this.lockProvider = lockProvider;
    }

    /**
     * Runs the task if the lock can be acquired before {@link #tryAcquireTimeout} passed. Otherwise, the task is not
     * executed. This is acceptable for page generating tasks as the risk of not being able to acquire the lock for a
     * long time is very low, and the scheduled job generating missing deployment pages will pick up and execute the
     * task later.
     */
    void runIfLockAquiredBeforeTimeout(String systemName, Runnable task) {
        String lockName = LOCK_NAME_PREFIX + systemName;
        tryAcquireLockWithTimeout(lockName).ifPresentOrElse(
                lock -> runLockedTask(task,lockName, lock),
                () -> log.warn("Unable to aquire lock {}, not running docgen. Pages will be generated by scheduled job.", lockName));

    }

    private void runLockedTask(Runnable task, String lockName, SimpleLock lock) {
        try {
            log.info("Acquired lock {}, running task", lockName);
            task.run();
        } finally {
            log.info("Releasing lock {}", lockName);
            lock.unlock();
        }
    }

    @SneakyThrows
    private Optional<SimpleLock> tryAcquireLockWithTimeout(String lockName) {
        LocalDateTime startedAt = LocalDateTime.now();
        for (int i = 1; ; i++) {
            Optional<SimpleLock> lock = lockProvider.lock(newLockConfiguration(lockName));
            if (lock.isPresent() || waitedLongEnough(startedAt)) {
                return lock;
            }
            log.info("Docgen for this system name is busy - waiting to aquire lock {} (retry #{})...", lockName, i);
            Thread.sleep(LOCK_RETRY_WAIT_DURATION.toMillis());
        }
    }

    private boolean waitedLongEnough(LocalDateTime startedAt) {
        return LocalDateTime.now().isAfter(startedAt.plus(tryAcquireTimeout));
    }

    private LockConfiguration newLockConfiguration(String lockName) {
        return new LockConfiguration(
                Instant.now(), lockName.toLowerCase(), LOCK_AT_MOST_FOR, LOCK_AT_LEAST_FOR);
    }

    // For usage in tests
    void setTryAcquireTimeout(Duration tryAcquireTimeout) {
        this.tryAcquireTimeout = tryAcquireTimeout;
    }
}
