/*
 * Decompiled with CFR 0.152.
 */
package ch.bitagent.bitcoin.lib.tx;

import ch.bitagent.bitcoin.lib.ecc.Hex;
import ch.bitagent.bitcoin.lib.ecc.Int;
import ch.bitagent.bitcoin.lib.ecc.PrivateKey;
import ch.bitagent.bitcoin.lib.helper.Bytes;
import ch.bitagent.bitcoin.lib.helper.Helper;
import ch.bitagent.bitcoin.lib.helper.Varint;
import ch.bitagent.bitcoin.lib.network.Message;
import ch.bitagent.bitcoin.lib.script.OpCodeNames;
import ch.bitagent.bitcoin.lib.script.Script;
import ch.bitagent.bitcoin.lib.script.ScriptCmd;
import ch.bitagent.bitcoin.lib.tx.TxIn;
import ch.bitagent.bitcoin.lib.tx.TxOut;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public class Tx
implements Message {
    private static final Logger log = Logger.getLogger(Tx.class.getSimpleName());
    public static final String COMMAND = "tx";
    private final Int version;
    private final List<TxIn> txIns;
    private final List<TxOut> txOuts;
    private Int locktime;
    private final Boolean testnet;
    private final Boolean segwit;
    private byte[] _hashPrevouts = null;
    private byte[] _hashSequence = null;
    private byte[] _hashOutputs = null;

    @Override
    public byte[] getCommand() {
        return COMMAND.getBytes();
    }

    public Tx(Int version, List<TxIn> txIns, List<TxOut> txOuts, Int locktime, Boolean testnet, Boolean segwit) {
        this.version = version;
        this.txIns = txIns;
        this.txOuts = txOuts;
        this.locktime = locktime;
        this.testnet = Objects.requireNonNullElse(testnet, false);
        this.segwit = Objects.requireNonNullElse(segwit, false);
    }

    public String id() {
        return Bytes.byteArrayToHexString(this.hash());
    }

    public byte[] hash() {
        return Bytes.changeOrder(Helper.hash256(this.serializeLegacy()));
    }

    public static Tx parse(ByteArrayInputStream stream, Boolean testnet) {
        Bytes.read(stream, 4);
        Hex segwitMarker = Hex.parse(Bytes.read(stream, 1));
        stream.reset();
        if (segwitMarker.eq(Hex.parse("00"))) {
            return Tx.parseSegwit(stream, testnet);
        }
        return Tx.parseLegacy(stream, testnet);
    }

    private static Tx parseLegacy(ByteArrayInputStream stream, Boolean testnet) {
        Hex version = Hex.parse(Bytes.changeOrder(Bytes.read(stream, 4)));
        int numInputs = Varint.read(stream).intValue();
        ArrayList<TxIn> txIns = new ArrayList<TxIn>();
        for (int i = 0; i < numInputs; ++i) {
            txIns.add(TxIn.parse(stream));
        }
        int numOutputs = Varint.read(stream).intValue();
        ArrayList<TxOut> txOuts = new ArrayList<TxOut>();
        for (int i = 0; i < numOutputs; ++i) {
            txOuts.add(TxOut.parse(stream));
        }
        Hex locktime = Hex.parse(Bytes.changeOrder(Bytes.read(stream, 4)));
        return new Tx(version, txIns, txOuts, locktime, testnet, false);
    }

    private static Tx parseSegwit(ByteArrayInputStream stream, Boolean testnet) {
        Hex version = Hex.parse(Bytes.changeOrder(Bytes.read(stream, 4)));
        Hex marker = Hex.parse(Bytes.read(stream, 2));
        if (!marker.eq(Hex.parse("0001"))) {
            throw new IllegalArgumentException(String.format("Not a segwit transaction %s", marker));
        }
        int numInputs = Varint.read(stream).intValue();
        ArrayList<TxIn> inputs = new ArrayList<TxIn>();
        for (int i = 0; i < numInputs; ++i) {
            inputs.add(TxIn.parse(stream));
        }
        int numOutputs = Varint.read(stream).intValue();
        ArrayList<TxOut> outputs = new ArrayList<TxOut>();
        for (int i = 0; i < numOutputs; ++i) {
            outputs.add(TxOut.parse(stream));
        }
        for (TxIn txIn : inputs) {
            int numItems = Varint.read(stream).intValue();
            ArrayList<ScriptCmd> items = new ArrayList<ScriptCmd>();
            for (int i = 0; i < numItems; ++i) {
                int itemLen = Varint.read(stream).intValue();
                if (itemLen == 0) {
                    items.add(OpCodeNames.OP_0.toScriptCmd());
                    continue;
                }
                items.add(new ScriptCmd(Bytes.read(stream, itemLen)));
            }
            txIn.setWitness(new Script(items));
        }
        Hex locktime = Hex.parse(Bytes.changeOrder(Bytes.read(stream, 4)));
        return new Tx(version, inputs, outputs, locktime, testnet, true);
    }

    @Override
    public byte[] serialize() {
        if (Boolean.TRUE.equals(this.segwit)) {
            return this.serializeSegwit();
        }
        return this.serializeLegacy();
    }

    private byte[] serializeLegacy() {
        ByteArrayOutputStream result = new ByteArrayOutputStream();
        result.writeBytes(this.version.toBytesLittleEndian(4));
        result.writeBytes(Varint.encode(Int.parse(this.txIns.size())));
        for (TxIn txIn : this.txIns) {
            result.writeBytes(txIn.serialize());
        }
        result.writeBytes(Varint.encode(Int.parse(this.txOuts.size())));
        for (TxOut txOut : this.txOuts) {
            result.writeBytes(txOut.serialize());
        }
        result.writeBytes(this.locktime.toBytesLittleEndian(4));
        return result.toByteArray();
    }

    private byte[] serializeSegwit() {
        ByteArrayOutputStream result = new ByteArrayOutputStream();
        result.writeBytes(this.version.toBytesLittleEndian(4));
        result.writeBytes(new byte[]{0, 1});
        result.writeBytes(Varint.encode(Int.parse(this.txIns.size())));
        for (TxIn txIn : this.txIns) {
            result.writeBytes(txIn.serialize());
        }
        result.writeBytes(Varint.encode(Int.parse(this.txOuts.size())));
        for (TxOut txOut : this.txOuts) {
            result.writeBytes(txOut.serialize());
        }
        for (TxIn txIn : this.txIns) {
            result.writeBytes(Int.parse(txIn.getWitness().getCmds().size()).toBytesLittleEndian(1));
            for (ScriptCmd witness : txIn.getWitness().getCmds()) {
                if (witness.isOpCode()) {
                    result.writeBytes(witness.getOpCode().getCode().toBytesLittleEndian(1));
                    continue;
                }
                result.writeBytes(Varint.encode(Int.parse(witness.getElement().length)));
                result.writeBytes(witness.getElement());
            }
        }
        result.writeBytes(this.locktime.toBytesLittleEndian(4));
        return result.toByteArray();
    }

    public Int fee() {
        log.fine(String.format("txIns: %s, txOuts: %s", this.txIns.size(), this.txOuts.size()));
        long start = System.currentTimeMillis();
        Int inputSum = Int.parse(0);
        Int outputSum = Int.parse(0);
        int txin = 0;
        for (TxIn txIn : this.txIns) {
            log.info(String.format("txin %s/%s", ++txin, this.txIns.size()));
            inputSum = inputSum.add(txIn.value(this.testnet));
            if (txin % 100 != 0) continue;
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException interruptedException) {}
        }
        for (TxOut txOut : this.txOuts) {
            outputSum = outputSum.add(txOut.getAmount());
        }
        Int fee = inputSum.sub(outputSum);
        log.fine(String.format("time %sms", System.currentTimeMillis() - start));
        return fee;
    }

    public Int sigHash(int inputIndex, Script redeemScript) {
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        stream.writeBytes(this.version.toBytesLittleEndian(4));
        stream.writeBytes(Varint.encode(Int.parse(this.txIns.size())));
        for (int i = 0; i < this.txIns.size(); ++i) {
            TxIn txIn = this.txIns.get(i);
            Script scriptSig = i == inputIndex ? (redeemScript != null ? redeemScript : txIn.scriptPubkey(this.testnet)) : null;
            stream.writeBytes(new TxIn(txIn.getPrevTx(), txIn.getPrevIndex(), scriptSig, txIn.getSequence()).serialize());
        }
        stream.writeBytes(Varint.encode(Int.parse(this.txOuts.size())));
        for (TxOut txOut : this.txOuts) {
            stream.writeBytes(txOut.serialize());
        }
        stream.writeBytes(this.locktime.toBytesLittleEndian(4));
        stream.writeBytes(Helper.SIGHASH_ALL.toBytesLittleEndian(4));
        byte[] h256 = Helper.hash256(stream.toByteArray());
        return Hex.parse(h256);
    }

    private byte[] hashPrevouts() {
        if (this._hashPrevouts == null) {
            byte[] allPrevouts = new byte[]{};
            byte[] allSequence = new byte[]{};
            for (TxIn txIn : this.txIns) {
                allPrevouts = Bytes.add(new byte[][]{allPrevouts, txIn.getPrevTx().toBytesLittleEndian(), txIn.getPrevIndex().toBytesLittleEndian(4)});
                allSequence = Bytes.add(allSequence, txIn.getSequence().toBytesLittleEndian(4));
            }
            this._hashPrevouts = Helper.hash256(allPrevouts);
            this._hashSequence = Helper.hash256(allSequence);
        }
        return this._hashPrevouts;
    }

    private byte[] hashSequence() {
        if (this._hashSequence == null) {
            this.hashPrevouts();
        }
        return this._hashSequence;
    }

    private byte[] hashOutputs() {
        if (this._hashOutputs == null) {
            byte[] allOutputs = new byte[]{};
            for (TxOut txOut : this.txOuts) {
                allOutputs = Bytes.add(allOutputs, txOut.serialize());
            }
            this._hashOutputs = Helper.hash256(allOutputs);
        }
        return this._hashOutputs;
    }

    public Int sigHashBip143(int inputIndex, Script redeemScript, Script witnessScript) {
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        TxIn txIn = this.txIns.get(inputIndex);
        stream.writeBytes(this.version.toBytesLittleEndian(4));
        stream.writeBytes(this.hashPrevouts());
        stream.writeBytes(this.hashSequence());
        stream.writeBytes(txIn.getPrevTx().toBytesLittleEndian());
        stream.writeBytes(txIn.getPrevIndex().toBytesLittleEndian(4));
        byte[] scriptCode = witnessScript != null ? witnessScript.serialize() : (redeemScript != null ? Script.p2pkhScript(redeemScript.getCmds().get(1).getElement()).serialize() : Script.p2pkhScript(txIn.scriptPubkey(this.testnet).getCmds().get(1).getElement()).serialize());
        stream.writeBytes(scriptCode);
        stream.writeBytes(txIn.value(this.testnet).toBytesLittleEndian(8));
        stream.writeBytes(txIn.getSequence().toBytesLittleEndian(4));
        stream.writeBytes(this.hashOutputs());
        stream.writeBytes(this.locktime.toBytesLittleEndian(4));
        stream.writeBytes(Helper.SIGHASH_ALL.toBytesLittleEndian(4));
        return Hex.parse(Helper.hash256(stream.toByteArray()));
    }

    public boolean verifyInput(int inputIndex) {
        ScriptCmd cmd;
        TxIn txIn = this.txIns.get(inputIndex);
        Script scriptPubkey = txIn.scriptPubkey(this.testnet);
        Int z = null;
        Script witness = null;
        if (scriptPubkey.isP2shScriptPubkey()) {
            cmd = txIn.getScriptSig().getCmds().get(txIn.getScriptSig().getCmds().size() - 1);
            byte[] rawRedeem = Bytes.add(Int.parse(cmd.getElement().length).toBytesLittleEndian(1), cmd.getElement());
            Script redeemScript = Script.parse(new ByteArrayInputStream(rawRedeem));
            if (redeemScript.isP2wpkhScriptPubkey()) {
                z = this.sigHashBip143(inputIndex, redeemScript, null);
                witness = txIn.getWitness();
            } else if (redeemScript.isP2wshScriptPubkey()) {
                cmd = txIn.getWitness().getCmds().get(txIn.getWitness().getCmds().size() - 1);
                byte[] rawWitness = Bytes.add(Varint.encode(Int.parse(cmd.getElement().length)), cmd.getElement());
                Script witnessScript = Script.parse(new ByteArrayInputStream(rawWitness));
                z = this.sigHashBip143(inputIndex, null, witnessScript);
                witness = txIn.getWitness();
            } else {
                z = this.sigHash(inputIndex, redeemScript);
                witness = null;
            }
        } else if (scriptPubkey.isP2wpkhScriptPubkey()) {
            z = this.sigHashBip143(inputIndex, null, null);
            witness = txIn.getWitness();
        } else if (scriptPubkey.isP2wshScriptPubkey()) {
            cmd = txIn.getWitness().getCmds().get(txIn.getWitness().getCmds().size() - 1);
            byte[] rawWitness = Bytes.add(Varint.encode(Int.parse(cmd.getElement().length)), cmd.getElement());
            Script witnessScript = Script.parse(new ByteArrayInputStream(rawWitness));
            z = this.sigHashBip143(inputIndex, null, witnessScript);
            witness = txIn.getWitness();
        } else {
            z = this.sigHash(inputIndex, null);
            witness = null;
        }
        Script combined = txIn.getScriptSig().add(scriptPubkey);
        return combined.evaluate(z, witness);
    }

    public boolean verify() {
        if (this.fee().lt(Int.parse(0))) {
            return false;
        }
        for (int i = 0; i < this.txIns.size(); ++i) {
            log.info(String.format("txin %s/%s", i + 1, this.txIns.size()));
            if (this.verifyInput(i)) continue;
            log.warning(String.format("TxIn has no valid signature - %s", this.txIns.get(i)));
            return false;
        }
        return true;
    }

    public boolean signInput(int inputIndex, PrivateKey privateKey) {
        Int z = this.sigHash(inputIndex, null);
        byte[] der = privateKey.sign(z).der();
        byte[] sig = Bytes.add(der, Helper.SIGHASH_ALL.toBytes(1));
        byte[] sec = privateKey.getPoint().sec(null);
        Script scriptSig = new Script(List.of(new ScriptCmd(sig), new ScriptCmd(sec)));
        this.txIns.get(inputIndex).setScriptSig(scriptSig);
        return this.verifyInput(inputIndex);
    }

    public boolean isCoinbase() {
        if (this.txIns.size() != 1) {
            return false;
        }
        TxIn firstInput = this.txIns.get(0);
        if (firstInput.getPrevTx().ne(Hex.parse("0000000000000000000000000000000000000000000000000000000000000000"))) {
            return false;
        }
        return !firstInput.getPrevIndex().ne(Hex.parse("ffffffff"));
    }

    public Int coinbaseHeight() {
        if (!this.isCoinbase()) {
            return null;
        }
        ScriptCmd firstCmd = this.txIns.get(0).getScriptSig().getCmds().get(0);
        return Hex.parse(Bytes.changeOrder(firstCmd.getElement()));
    }

    public String toString() {
        return String.format("tx %s:%s:\n%s:\n%s:%s", this.id(), this.version, this.txIns.stream().map(TxIn::toString).collect(Collectors.joining("\n")), this.txOuts.stream().map(TxOut::toString).collect(Collectors.joining("\n")), this.locktime);
    }

    public List<TxOut> getTxOuts() {
        return this.txOuts;
    }

    public Boolean getSegwit() {
        return this.segwit;
    }

    public Int getLocktime() {
        return this.locktime;
    }

    public void setLocktime(Int locktime) {
        this.locktime = locktime;
    }

    public List<TxIn> getTxIns() {
        return this.txIns;
    }

    public Int getVersion() {
        return this.version;
    }

    public Boolean getTestnet() {
        return this.testnet;
    }
}

