/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azure;

import com.microsoft.azure.storage.AccessCondition;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.CloudBlob;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.azure.AzureBlobStorageTestAccount;
import org.apache.hadoop.fs.azure.AzureException;
import org.apache.hadoop.fs.azure.AzureNativeFileSystemStore;
import org.apache.hadoop.fs.azure.NativeAzureFileSystem;
import org.apache.hadoop.fs.azure.SelfRenewingLease;
import org.apache.hadoop.fs.azure.TestNativeAzureFSPageBlobLive;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;

public abstract class NativeAzureFileSystemBaseTest {
    protected FileSystem fs;
    private AzureBlobStorageTestAccount testAccount;
    private final long modifiedTimeErrorMargin = 5000L;
    public static final Log LOG = LogFactory.getLog(NativeAzureFileSystemBaseTest.class);
    int FILE_SIZE = 0x400001;
    int MAX_STRIDE = this.FILE_SIZE + 1;
    Path PATH = new Path("/available.dat");
    private long firstEndTime;
    private long secondStartTime;

    protected abstract AzureBlobStorageTestAccount createTestAccount() throws Exception;

    @Before
    public void setUp() throws Exception {
        this.testAccount = this.createTestAccount();
        if (this.testAccount != null) {
            this.fs = this.testAccount.getFileSystem();
        }
        Assume.assumeNotNull((Object[])new Object[]{this.testAccount});
    }

    @After
    public void tearDown() throws Exception {
        if (this.testAccount != null) {
            this.testAccount.cleanup();
            this.testAccount = null;
            this.fs = null;
        }
    }

    @Test
    public void testCheckingNonExistentOneLetterFile() throws Exception {
        Assert.assertFalse((boolean)this.fs.exists(new Path("/a")));
    }

    @Test
    public void testStoreRetrieveFile() throws Exception {
        Path testFile = new Path("unit-test-file");
        this.writeString(testFile, "Testing");
        Assert.assertTrue((boolean)this.fs.exists(testFile));
        FileStatus status = this.fs.getFileStatus(testFile);
        Assert.assertNotNull((Object)status);
        Assert.assertEquals((Object)new FsPermission(420), (Object)status.getPermission());
        Assert.assertEquals((Object)"Testing", (Object)this.readString(testFile));
        this.fs.delete(testFile, true);
    }

    @Test
    public void testStoreDeleteFolder() throws Exception {
        Path testFolder = new Path("storeDeleteFolder");
        Assert.assertFalse((boolean)this.fs.exists(testFolder));
        Assert.assertTrue((boolean)this.fs.mkdirs(testFolder));
        Assert.assertTrue((boolean)this.fs.exists(testFolder));
        FileStatus status = this.fs.getFileStatus(testFolder);
        Assert.assertNotNull((Object)status);
        Assert.assertTrue((boolean)status.isDirectory());
        Assert.assertEquals((Object)new FsPermission(493), (Object)status.getPermission());
        Path innerFile = new Path(testFolder, "innerFile");
        Assert.assertTrue((boolean)this.fs.createNewFile(innerFile));
        Assert.assertTrue((boolean)this.fs.exists(innerFile));
        Assert.assertTrue((boolean)this.fs.delete(testFolder, true));
        Assert.assertFalse((boolean)this.fs.exists(innerFile));
        Assert.assertFalse((boolean)this.fs.exists(testFolder));
    }

    @Test
    public void testFileOwnership() throws Exception {
        Path testFile = new Path("ownershipTestFile");
        this.writeString(testFile, "Testing");
        this.testOwnership(testFile);
    }

    @Test
    public void testFolderOwnership() throws Exception {
        Path testFolder = new Path("ownershipTestFolder");
        this.fs.mkdirs(testFolder);
        this.testOwnership(testFolder);
    }

    private void testOwnership(Path pathUnderTest) throws IOException {
        FileStatus ret = this.fs.getFileStatus(pathUnderTest);
        UserGroupInformation currentUser = UserGroupInformation.getCurrentUser();
        Assert.assertTrue((boolean)ret.getOwner().equals(currentUser.getShortUserName()));
        this.fs.delete(pathUnderTest, true);
    }

    private static FsPermission ignoreStickyBit(FsPermission original) {
        return new FsPermission(original.getUserAction(), original.getGroupAction(), original.getOtherAction());
    }

    private static void assertEqualsIgnoreStickyBit(FsPermission expected, FsPermission actual) {
        Assert.assertEquals((Object)NativeAzureFileSystemBaseTest.ignoreStickyBit(expected), (Object)NativeAzureFileSystemBaseTest.ignoreStickyBit(actual));
    }

    @Test
    public void testFilePermissions() throws Exception {
        Path testFile = new Path("permissionTestFile");
        FsPermission permission = FsPermission.createImmutable((short)644);
        this.createEmptyFile(testFile, permission);
        FileStatus ret = this.fs.getFileStatus(testFile);
        NativeAzureFileSystemBaseTest.assertEqualsIgnoreStickyBit(permission, ret.getPermission());
        this.fs.delete(testFile, true);
    }

    @Test
    public void testFolderPermissions() throws Exception {
        Path testFolder = new Path("permissionTestFolder");
        FsPermission permission = FsPermission.createImmutable((short)644);
        this.fs.mkdirs(testFolder, permission);
        FileStatus ret = this.fs.getFileStatus(testFolder);
        NativeAzureFileSystemBaseTest.assertEqualsIgnoreStickyBit(permission, ret.getPermission());
        this.fs.delete(testFolder, true);
    }

    void testDeepFileCreationBase(String testFilePath, String firstDirPath, String middleDirPath, short permissionShort, short umaskedPermissionShort) throws Exception {
        Path testFile = new Path(testFilePath);
        Path firstDir = new Path(firstDirPath);
        Path middleDir = new Path(middleDirPath);
        FsPermission permission = FsPermission.createImmutable((short)permissionShort);
        FsPermission umaskedPermission = FsPermission.createImmutable((short)umaskedPermissionShort);
        this.createEmptyFile(testFile, permission);
        FsPermission rootPerm = this.fs.getFileStatus(firstDir.getParent()).getPermission();
        FsPermission inheritPerm = FsPermission.createImmutable((short)((short)(rootPerm.toShort() | 0xC0)));
        Assert.assertTrue((boolean)this.fs.exists(testFile));
        Assert.assertTrue((boolean)this.fs.exists(firstDir));
        Assert.assertTrue((boolean)this.fs.exists(middleDir));
        FileStatus directoryStatus = this.fs.getFileStatus(middleDir);
        Assert.assertTrue((boolean)directoryStatus.isDirectory());
        NativeAzureFileSystemBaseTest.assertEqualsIgnoreStickyBit(inheritPerm, directoryStatus.getPermission());
        FileStatus fileStatus = this.fs.getFileStatus(testFile);
        Assert.assertFalse((boolean)fileStatus.isDirectory());
        NativeAzureFileSystemBaseTest.assertEqualsIgnoreStickyBit(umaskedPermission, fileStatus.getPermission());
        Assert.assertTrue((boolean)this.fs.delete(firstDir, true));
        Assert.assertFalse((boolean)this.fs.exists(testFile));
    }

    @Test
    public void testDeepFileCreation() throws Exception {
        this.testDeepFileCreationBase("deep/file/creation/test", "deep", "deep/file/creation", (short)420, (short)420);
        this.testDeepFileCreationBase("deep/file/creation/test", "deep", "deep/file/creation", (short)511, (short)493);
        this.testDeepFileCreationBase("/deep/file/creation/test", "/deep", "/deep/file/creation", (short)420, (short)420);
        this.testDeepFileCreationBase("/deep/file/creation/test", "/deep", "/deep/file/creation", (short)448, (short)448);
        this.testDeepFileCreationBase("/deep/file", "/deep", "/deep", (short)420, (short)420);
        this.testDeepFileCreationBase("deep/file", "deep", "deep", (short)420, (short)420);
    }

    @Test
    public void testRename() throws Exception {
        for (RenameVariation variation : RenameVariation.values()) {
            Path originalFile;
            System.out.printf("Rename variation: %s\n", new Object[]{variation});
            switch (variation) {
                case NormalFileName: {
                    originalFile = new Path("fileToRename");
                    break;
                }
                case SourceInAFolder: {
                    originalFile = new Path("file/to/rename");
                    break;
                }
                case SourceWithSpace: {
                    originalFile = new Path("file to rename");
                    break;
                }
                case SourceWithPlusAndPercent: {
                    originalFile = new Path("file+to%rename");
                    break;
                }
                default: {
                    throw new Exception("Unknown variation");
                }
            }
            Path destinationFile = new Path("file/resting/destination");
            Assert.assertTrue((boolean)this.fs.createNewFile(originalFile));
            Assert.assertTrue((boolean)this.fs.exists(originalFile));
            Assert.assertFalse((boolean)this.fs.rename(originalFile, destinationFile));
            Assert.assertTrue((boolean)this.fs.mkdirs(destinationFile.getParent()));
            boolean result = this.fs.rename(originalFile, destinationFile);
            Assert.assertTrue((boolean)result);
            Assert.assertTrue((boolean)this.fs.exists(destinationFile));
            Assert.assertFalse((boolean)this.fs.exists(originalFile));
            this.fs.delete(destinationFile.getParent(), true);
        }
    }

    @Test
    public void testRenameImplicitFolder() throws Exception {
        Path testFile = new Path("deep/file/rename/test");
        FsPermission permission = FsPermission.createImmutable((short)644);
        this.createEmptyFile(testFile, permission);
        boolean renameResult = this.fs.rename(new Path("deep/file"), new Path("deep/renamed"));
        Assert.assertTrue((boolean)renameResult);
        Assert.assertFalse((boolean)this.fs.exists(testFile));
        FileStatus newStatus = this.fs.getFileStatus(new Path("deep/renamed/rename/test"));
        Assert.assertNotNull((Object)newStatus);
        NativeAzureFileSystemBaseTest.assertEqualsIgnoreStickyBit(permission, newStatus.getPermission());
        Assert.assertTrue((boolean)this.fs.delete(new Path("deep"), true));
    }

    @Test
    public void testRenameFolder() throws Exception {
        for (RenameFolderVariation variation : RenameFolderVariation.values()) {
            Path originalFolder = new Path("folderToRename");
            if (variation != RenameFolderVariation.CreateJustInnerFile) {
                Assert.assertTrue((boolean)this.fs.mkdirs(originalFolder));
            }
            Path innerFile = new Path(originalFolder, "innerFile");
            Path innerFile2 = new Path(originalFolder, "innerFile2");
            if (variation != RenameFolderVariation.CreateJustFolder) {
                Assert.assertTrue((boolean)this.fs.createNewFile(innerFile));
                Assert.assertTrue((boolean)this.fs.createNewFile(innerFile2));
            }
            Path destination = new Path("renamedFolder");
            Assert.assertTrue((boolean)this.fs.rename(originalFolder, destination));
            Assert.assertTrue((boolean)this.fs.exists(destination));
            if (variation != RenameFolderVariation.CreateJustFolder) {
                Assert.assertTrue((boolean)this.fs.exists(new Path(destination, innerFile.getName())));
                Assert.assertTrue((boolean)this.fs.exists(new Path(destination, innerFile2.getName())));
            }
            Assert.assertFalse((boolean)this.fs.exists(originalFolder));
            Assert.assertFalse((boolean)this.fs.exists(innerFile));
            Assert.assertFalse((boolean)this.fs.exists(innerFile2));
            this.fs.delete(destination, true);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCopyFromLocalFileSystem() throws Exception {
        Path localFilePath = new Path(System.getProperty("test.build.data", "azure_test"));
        FileSystem localFs = FileSystem.get((Configuration)new Configuration());
        localFs.delete(localFilePath, true);
        try {
            this.writeString(localFs, localFilePath, "Testing");
            Path dstPath = new Path("copiedFromLocal");
            Assert.assertTrue((boolean)FileUtil.copy((FileSystem)localFs, (Path)localFilePath, (FileSystem)this.fs, (Path)dstPath, (boolean)false, (Configuration)this.fs.getConf()));
            Assert.assertTrue((boolean)this.fs.exists(dstPath));
            Assert.assertEquals((Object)"Testing", (Object)this.readString(this.fs, dstPath));
            this.fs.delete(dstPath, true);
        }
        finally {
            localFs.delete(localFilePath, true);
        }
    }

    @Test
    public void testListDirectory() throws Exception {
        Path rootFolder = new Path("testingList");
        Assert.assertTrue((boolean)this.fs.mkdirs(rootFolder));
        FileStatus[] listed = this.fs.listStatus(rootFolder);
        Assert.assertEquals((long)0L, (long)listed.length);
        Path innerFolder = new Path(rootFolder, "inner");
        Assert.assertTrue((boolean)this.fs.mkdirs(innerFolder));
        listed = this.fs.listStatus(rootFolder);
        Assert.assertEquals((long)1L, (long)listed.length);
        Assert.assertTrue((boolean)listed[0].isDirectory());
        Path innerFile = new Path(innerFolder, "innerFile");
        this.writeString(innerFile, "testing");
        listed = this.fs.listStatus(rootFolder);
        Assert.assertEquals((long)1L, (long)listed.length);
        Assert.assertTrue((boolean)listed[0].isDirectory());
        listed = this.fs.listStatus(innerFolder);
        Assert.assertEquals((long)1L, (long)listed.length);
        Assert.assertFalse((boolean)listed[0].isDirectory());
        Assert.assertTrue((boolean)this.fs.delete(rootFolder, true));
    }

    @Test
    public void testStatistics() throws Exception {
        FileSystem.clearStatistics();
        FileSystem.Statistics stats = FileSystem.getStatistics((String)"wasb", NativeAzureFileSystem.class);
        Assert.assertEquals((long)0L, (long)stats.getBytesRead());
        Assert.assertEquals((long)0L, (long)stats.getBytesWritten());
        Path newFile = new Path("testStats");
        this.writeString(newFile, "12345678");
        Assert.assertEquals((long)8L, (long)stats.getBytesWritten());
        Assert.assertEquals((long)0L, (long)stats.getBytesRead());
        String readBack = this.readString(newFile);
        Assert.assertEquals((Object)"12345678", (Object)readBack);
        Assert.assertEquals((long)8L, (long)stats.getBytesRead());
        Assert.assertEquals((long)8L, (long)stats.getBytesWritten());
        Assert.assertTrue((boolean)this.fs.delete(newFile, true));
        Assert.assertEquals((long)8L, (long)stats.getBytesRead());
        Assert.assertEquals((long)8L, (long)stats.getBytesWritten());
    }

    @Test
    public void testUriEncoding() throws Exception {
        this.fs.create(new Path("p/t%5Fe")).close();
        FileStatus[] listing = this.fs.listStatus(new Path("p"));
        Assert.assertEquals((long)1L, (long)listing.length);
        Assert.assertEquals((Object)"t%5Fe", (Object)listing[0].getPath().getName());
        Assert.assertTrue((boolean)this.fs.rename(new Path("p"), new Path("q")));
        Assert.assertTrue((boolean)this.fs.delete(new Path("q"), true));
    }

    @Test
    public void testUriEncodingMoreComplexCharacters() throws Exception {
        String fileName = "!#$'()*;=[]%";
        String directoryName = "*;=[]%!#$'()";
        this.fs.create(new Path(directoryName, fileName)).close();
        FileStatus[] listing = this.fs.listStatus(new Path(directoryName));
        Assert.assertEquals((long)1L, (long)listing.length);
        Assert.assertEquals((Object)fileName, (Object)listing[0].getPath().getName());
        FileStatus status = this.fs.getFileStatus(new Path(directoryName, fileName));
        Assert.assertEquals((Object)fileName, (Object)status.getPath().getName());
        FSDataInputStream stream = this.fs.open(new Path(directoryName, fileName));
        Assert.assertNotNull((Object)stream);
        stream.close();
        Assert.assertTrue((boolean)this.fs.delete(new Path(directoryName, fileName), true));
        Assert.assertTrue((boolean)this.fs.delete(new Path(directoryName), true));
    }

    @Test
    public void testChineseCharacters() throws Exception {
        String chinese = "\u963f\u4db5";
        String fileName = "filename" + chinese;
        String directoryName = chinese;
        this.fs.create(new Path(directoryName, fileName)).close();
        FileStatus[] listing = this.fs.listStatus(new Path(directoryName));
        Assert.assertEquals((long)1L, (long)listing.length);
        Assert.assertEquals((Object)fileName, (Object)listing[0].getPath().getName());
        FileStatus status = this.fs.getFileStatus(new Path(directoryName, fileName));
        Assert.assertEquals((Object)fileName, (Object)status.getPath().getName());
        FSDataInputStream stream = this.fs.open(new Path(directoryName, fileName));
        Assert.assertNotNull((Object)stream);
        stream.close();
        Assert.assertTrue((boolean)this.fs.delete(new Path(directoryName, fileName), true));
        Assert.assertTrue((boolean)this.fs.delete(new Path(directoryName), true));
    }

    @Test
    public void testChineseCharactersFolderRename() throws Exception {
        String chinese = "\u963f\u4db5";
        String fileName = "filename" + chinese;
        String srcDirectoryName = chinese;
        String targetDirectoryName = "target" + chinese;
        this.fs.create(new Path(srcDirectoryName, fileName)).close();
        this.fs.rename(new Path(srcDirectoryName), new Path(targetDirectoryName));
        FileStatus[] listing = this.fs.listStatus(new Path(targetDirectoryName));
        Assert.assertEquals((long)1L, (long)listing.length);
        Assert.assertEquals((Object)fileName, (Object)listing[0].getPath().getName());
        FileStatus status = this.fs.getFileStatus(new Path(targetDirectoryName, fileName));
        Assert.assertEquals((Object)fileName, (Object)status.getPath().getName());
        Assert.assertTrue((boolean)this.fs.delete(new Path(targetDirectoryName, fileName), true));
        Assert.assertTrue((boolean)this.fs.delete(new Path(targetDirectoryName), true));
    }

    @Test
    public void testReadingDirectoryAsFile() throws Exception {
        Path dir = new Path("/x");
        Assert.assertTrue((boolean)this.fs.mkdirs(dir));
        try {
            this.fs.open(dir).close();
            Assert.assertTrue((String)"Should've thrown", (boolean)false);
        }
        catch (FileNotFoundException ex) {
            Assert.assertEquals((Object)"/x is a directory not a file.", (Object)ex.getMessage());
        }
    }

    @Test
    public void testCreatingFileOverDirectory() throws Exception {
        Path dir = new Path("/x");
        Assert.assertTrue((boolean)this.fs.mkdirs(dir));
        try {
            this.fs.create(dir).close();
            Assert.assertTrue((String)"Should've thrown", (boolean)false);
        }
        catch (IOException ex) {
            Assert.assertEquals((Object)"Cannot create file /x; already exists as a directory.", (Object)ex.getMessage());
        }
    }

    @Test
    public void testSetPermissionOnFile() throws Exception {
        Path newFile = new Path("testPermission");
        FSDataOutputStream output = this.fs.create(newFile);
        output.write(13);
        output.close();
        FsPermission newPermission = new FsPermission(448);
        this.fs.setPermission(newFile, newPermission);
        FileStatus newStatus = this.fs.getFileStatus(newFile);
        Assert.assertNotNull((Object)newStatus);
        Assert.assertEquals((Object)newPermission, (Object)newStatus.getPermission());
        Assert.assertEquals((Object)"supergroup", (Object)newStatus.getGroup());
        Assert.assertEquals((Object)UserGroupInformation.getCurrentUser().getShortUserName(), (Object)newStatus.getOwner());
        if (!(this instanceof TestNativeAzureFSPageBlobLive)) {
            Assert.assertEquals((long)1L, (long)newStatus.getLen());
        }
    }

    @Test
    public void testSetPermissionOnFolder() throws Exception {
        Path newFolder = new Path("testPermission");
        Assert.assertTrue((boolean)this.fs.mkdirs(newFolder));
        FsPermission newPermission = new FsPermission(384);
        this.fs.setPermission(newFolder, newPermission);
        FileStatus newStatus = this.fs.getFileStatus(newFolder);
        Assert.assertNotNull((Object)newStatus);
        Assert.assertEquals((Object)newPermission, (Object)newStatus.getPermission());
        Assert.assertTrue((boolean)newStatus.isDirectory());
    }

    @Test
    public void testSetOwnerOnFile() throws Exception {
        Path newFile = new Path("testOwner");
        FSDataOutputStream output = this.fs.create(newFile);
        output.write(13);
        output.close();
        this.fs.setOwner(newFile, "newUser", null);
        FileStatus newStatus = this.fs.getFileStatus(newFile);
        Assert.assertNotNull((Object)newStatus);
        Assert.assertEquals((Object)"newUser", (Object)newStatus.getOwner());
        Assert.assertEquals((Object)"supergroup", (Object)newStatus.getGroup());
        if (!(this instanceof TestNativeAzureFSPageBlobLive)) {
            Assert.assertEquals((long)1L, (long)newStatus.getLen());
        }
        this.fs.setOwner(newFile, null, "newGroup");
        newStatus = this.fs.getFileStatus(newFile);
        Assert.assertNotNull((Object)newStatus);
        Assert.assertEquals((Object)"newUser", (Object)newStatus.getOwner());
        Assert.assertEquals((Object)"newGroup", (Object)newStatus.getGroup());
    }

    @Test
    public void testSetOwnerOnFolder() throws Exception {
        Path newFolder = new Path("testOwner");
        Assert.assertTrue((boolean)this.fs.mkdirs(newFolder));
        this.fs.setOwner(newFolder, "newUser", null);
        FileStatus newStatus = this.fs.getFileStatus(newFolder);
        Assert.assertNotNull((Object)newStatus);
        Assert.assertEquals((Object)"newUser", (Object)newStatus.getOwner());
        Assert.assertTrue((boolean)newStatus.isDirectory());
    }

    @Test
    public void testModifiedTimeForFile() throws Exception {
        Path testFile = new Path("testFile");
        this.fs.create(testFile).close();
        this.testModifiedTime(testFile);
    }

    @Test
    public void testModifiedTimeForFolder() throws Exception {
        Path testFolder = new Path("testFolder");
        Assert.assertTrue((boolean)this.fs.mkdirs(testFolder));
        this.testModifiedTime(testFolder);
    }

    @Test
    public void testFolderLastModifiedTime() throws Exception {
        Path parentFolder = new Path("testFolder");
        Path innerFile = new Path(parentFolder, "innerfile");
        Assert.assertTrue((boolean)this.fs.mkdirs(parentFolder));
        long lastModifiedTime = this.fs.getFileStatus(parentFolder).getModificationTime();
        Thread.sleep(5001L);
        Assert.assertTrue((boolean)this.fs.createNewFile(innerFile));
        Assert.assertFalse((boolean)this.testModifiedTime(parentFolder, lastModifiedTime));
        this.testModifiedTime(parentFolder);
        lastModifiedTime = this.fs.getFileStatus(parentFolder).getModificationTime();
        Path destFolder = new Path("testDestFolder");
        Assert.assertTrue((boolean)this.fs.mkdirs(destFolder));
        long destLastModifiedTime = this.fs.getFileStatus(destFolder).getModificationTime();
        Thread.sleep(5001L);
        Path destFile = new Path(destFolder, "innerfile");
        Assert.assertTrue((boolean)this.fs.rename(innerFile, destFile));
        Assert.assertFalse((boolean)this.testModifiedTime(parentFolder, lastModifiedTime));
        Assert.assertFalse((boolean)this.testModifiedTime(destFolder, destLastModifiedTime));
        this.testModifiedTime(parentFolder);
        this.testModifiedTime(destFolder);
        destLastModifiedTime = this.fs.getFileStatus(destFolder).getModificationTime();
        Thread.sleep(5001L);
        this.fs.delete(destFile, false);
        Assert.assertFalse((boolean)this.testModifiedTime(destFolder, destLastModifiedTime));
        this.testModifiedTime(destFolder);
    }

    @Test
    public void testListSlash() throws Exception {
        Path testFolder = new Path("/testFolder");
        Path testFile = new Path(testFolder, "testFile");
        Assert.assertTrue((boolean)this.fs.mkdirs(testFolder));
        Assert.assertTrue((boolean)this.fs.createNewFile(testFile));
        FileStatus status = this.fs.getFileStatus(new Path("/testFolder"));
        Assert.assertTrue((boolean)status.isDirectory());
        status = this.fs.getFileStatus(new Path("/testFolder/"));
        Assert.assertTrue((boolean)status.isDirectory());
        status = this.fs.getFileStatus(new Path("/testFolder/."));
        Assert.assertTrue((boolean)status.isDirectory());
    }

    @Test
    public void testCannotCreatePageBlobByDefault() throws Exception {
        Configuration conf = new Configuration();
        String[] rawPageBlobDirs = conf.getStrings("fs.azure.page.blob.dir");
        Assert.assertTrue((rawPageBlobDirs == null ? 1 : 0) != 0);
    }

    @Test
    public void testRedoRenameFolder() throws IOException {
        String srcKey = "folderToRename";
        Path originalFolder = new Path(srcKey);
        Assert.assertTrue((boolean)this.fs.mkdirs(originalFolder));
        Path innerFile = new Path(originalFolder, "innerFile");
        Assert.assertTrue((boolean)this.fs.createNewFile(innerFile));
        Path innerFile2 = new Path(originalFolder, "innerFile2");
        Assert.assertTrue((boolean)this.fs.createNewFile(innerFile2));
        String dstKey = "renamedFolder";
        Path home = this.fs.getHomeDirectory();
        String relativeHomeDir = this.getRelativePath(home.toString());
        NativeAzureFileSystem.FolderRenamePending pending = new NativeAzureFileSystem.FolderRenamePending(relativeHomeDir + "/" + srcKey, relativeHomeDir + "/" + dstKey, null, (NativeAzureFileSystem)this.fs);
        String renameDescription = pending.makeRenamePendingFileContents();
        Assert.assertTrue((boolean)this.fs.delete(innerFile, false));
        Path destination = new Path(dstKey);
        Path innerDest = new Path(destination, "innerFile");
        Assert.assertTrue((boolean)this.fs.createNewFile(innerDest));
        String renamePendingStr = "folderToRename-RenamePending.json";
        Path renamePendingFile = new Path("folderToRename-RenamePending.json");
        FSDataOutputStream out = this.fs.create(renamePendingFile, true);
        Assert.assertTrue((out != null ? 1 : 0) != 0);
        this.writeString(out, renameDescription);
        Assert.assertFalse((boolean)this.fs.exists(originalFolder));
        Assert.assertTrue((boolean)this.fs.exists(destination));
        Assert.assertTrue((boolean)this.fs.exists(new Path(destination, innerFile.getName())));
        Assert.assertTrue((boolean)this.fs.exists(new Path(destination, innerFile2.getName())));
        Assert.assertFalse((boolean)this.fs.exists(originalFolder));
        Assert.assertFalse((boolean)this.fs.exists(innerFile));
        Assert.assertFalse((boolean)this.fs.exists(innerFile2));
        Assert.assertFalse((boolean)this.fs.exists(renamePendingFile));
        FileStatus[] listed = this.fs.listStatus(destination);
        Assert.assertEquals((long)2L, (long)listed.length);
        Path root = this.fs.getHomeDirectory();
        listed = this.fs.listStatus(root);
        Assert.assertEquals((long)1L, (long)listed.length);
        Assert.assertTrue((boolean)listed[0].isDirectory());
    }

    @Test
    public void testRedoRenameFolderInFolderListing() throws IOException {
        String parent = "parent";
        Path parentFolder = new Path(parent);
        Assert.assertTrue((boolean)this.fs.mkdirs(parentFolder));
        Path inner = new Path(parentFolder, "innerFolder");
        Assert.assertTrue((boolean)this.fs.mkdirs(inner));
        Path inner2 = new Path(parentFolder, "innerFolder2");
        Assert.assertTrue((boolean)this.fs.mkdirs(inner2));
        Path innerFile = new Path(inner2, "file");
        Assert.assertTrue((boolean)this.fs.createNewFile(innerFile));
        Path inner2renamed = new Path(parentFolder, "innerFolder2Renamed");
        Path home = this.fs.getHomeDirectory();
        String relativeHomeDir = this.getRelativePath(home.toString());
        NativeAzureFileSystem.FolderRenamePending pending = new NativeAzureFileSystem.FolderRenamePending(relativeHomeDir + "/" + inner2, relativeHomeDir + "/" + inner2renamed, null, (NativeAzureFileSystem)this.fs);
        String renamePendingStr = inner2 + "-RenamePending.json";
        Path renamePendingFile = new Path(renamePendingStr);
        FSDataOutputStream out = this.fs.create(renamePendingFile, true);
        Assert.assertTrue((out != null ? 1 : 0) != 0);
        this.writeString(out, pending.makeRenamePendingFileContents());
        FileStatus[] listed = this.fs.listStatus(parentFolder);
        Assert.assertEquals((long)2L, (long)listed.length);
        Assert.assertTrue((boolean)listed[0].isDirectory());
        Assert.assertTrue((boolean)listed[1].isDirectory());
        Assert.assertFalse((boolean)this.fs.exists(inner2));
        Assert.assertTrue((boolean)this.fs.exists(inner2renamed));
        Assert.assertTrue((boolean)this.fs.exists(new Path(inner2renamed, "file")));
    }

    @Test
    public void testRenameRedoFolderAlreadyDone() throws IOException {
        String orig = "originalFolder";
        String dest = "renamedFolder";
        Path destPath = new Path(dest);
        Assert.assertTrue((boolean)this.fs.mkdirs(destPath));
        Path home = this.fs.getHomeDirectory();
        String relativeHomeDir = this.getRelativePath(home.toString());
        NativeAzureFileSystem.FolderRenamePending pending = new NativeAzureFileSystem.FolderRenamePending(relativeHomeDir + "/" + orig, relativeHomeDir + "/" + dest, null, (NativeAzureFileSystem)this.fs);
        String renamePendingStr = orig + "-RenamePending.json";
        Path renamePendingFile = new Path(renamePendingStr);
        FSDataOutputStream out = this.fs.create(renamePendingFile, true);
        Assert.assertTrue((out != null ? 1 : 0) != 0);
        this.writeString(out, pending.makeRenamePendingFileContents());
        try {
            pending.redo();
        }
        catch (Exception e) {
            Assert.fail();
        }
        FileStatus[] listed = this.fs.listStatus(new Path("/"));
        Assert.assertEquals((long)1L, (long)listed.length);
        Assert.assertTrue((boolean)listed[0].isDirectory());
    }

    @Test
    public void testRedoFolderRenameAll() throws IllegalArgumentException, IOException {
        FileFolder original = new FileFolder("folderToRename");
        original.add("innerFile").add("innerFile2");
        FileFolder partialSrc = original.copy();
        FileFolder partialDst = original.copy();
        partialDst.setName("renamedFolder");
        partialSrc.setPresent(0, false);
        partialDst.setPresent(1, false);
        this.testRenameRedoFolderSituation(original, partialSrc, partialDst);
        original = new FileFolder("folderToRename");
        original.add("file1").add("file2").add("file3");
        partialSrc = original.copy();
        partialDst = original.copy();
        partialDst.setName("renamedFolder");
        partialSrc.setPresent(1, false);
        partialDst.setPresent(2, false);
        this.testRenameRedoFolderSituation(original, partialSrc, partialDst);
        int SIZE = 5;
        Assert.assertTrue((boolean)true);
        FileFolder original2 = new FileFolder("folderToRename");
        for (int i = 0; i < 5; ++i) {
            original2.add("file" + Integer.toString(i));
        }
        FileFolder partialSrc2 = original2.copy();
        FileFolder partialDst2 = original2.copy();
        partialDst2.setName("renamedFolder");
        for (int i = 0; i < 5; ++i) {
            partialSrc2.setPresent(i, i >= 2);
            partialDst2.setPresent(i, i <= 2);
        }
        this.testRenameRedoFolderSituation(original2, partialSrc2, partialDst2);
        original = new FileFolder("folderToRename");
        FileFolder nested = new FileFolder("nestedFolder");
        nested.add("a").add("b").add("c");
        original.add(nested).add("p").add("q");
        partialSrc2 = original.copy();
        partialDst2 = original.copy();
        partialDst2.setName("renamedFolder");
        partialSrc2.getMember(0).setPresent(0, false);
        partialDst2.getMember(0).setPresent(1, false);
        partialDst2.getMember(0).setPresent(2, false);
        this.testRenameRedoFolderSituation(original, partialSrc2, partialDst2);
    }

    private void testRenameRedoFolderSituation(FileFolder fullSrc, FileFolder partialSrc, FileFolder partialDst) throws IllegalArgumentException, IOException {
        fullSrc.create();
        fullSrc.makeRenamePending(partialDst);
        partialSrc.prune();
        partialDst.create();
        Assert.assertFalse((boolean)fullSrc.exists());
        partialDst.verifyExists();
        fullSrc.verifyGone();
        this.fs.delete(new Path(partialDst.getName()), true);
    }

    private String getRelativePath(String path) {
        int slashCount = 0;
        for (int i = 0; i < path.length(); ++i) {
            if (path.charAt(i) != '/' || ++slashCount != 3) continue;
            return path.substring(i + 1, path.length());
        }
        throw new RuntimeException("Incorrect path prefix -- expected wasb://.../...");
    }

    @Test
    public void testCloseFileSystemTwice() throws Exception {
        this.fs.close();
        this.fs.close();
    }

    @Test
    public void testAvailable() throws IOException {
        FSDataOutputStream out = this.fs.create(this.PATH);
        byte[] data = new byte[this.FILE_SIZE];
        Arrays.fill(data, (byte)5);
        out.write(data, 0, this.FILE_SIZE);
        out.close();
        this.verifyAvailable(1);
        this.verifyAvailable(100);
        this.verifyAvailable(5000);
        this.verifyAvailable(this.FILE_SIZE);
        this.verifyAvailable(this.MAX_STRIDE);
        this.fs.delete(this.PATH, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyAvailable(int readStride) throws IOException {
        try (FSDataInputStream in = this.fs.open(this.PATH);){
            int available;
            byte[] inputBuffer = new byte[this.MAX_STRIDE];
            int position = 0;
            int bytesRead = 0;
            while (bytesRead != this.FILE_SIZE) {
                available = in.available();
                if ((bytesRead += in.read(inputBuffer, position, readStride)) >= this.FILE_SIZE || available >= 1) continue;
                Assert.fail((String)String.format("expected available > 0 but got: position = %d, bytesRead = %d, in.available() = %d", position, bytesRead, available));
            }
            available = in.available();
            Assert.assertTrue((available == 0 ? 1 : 0) != 0);
        }
    }

    @Test
    public void testGetFileSizeFromListing() throws IOException {
        Path path = new Path("file.dat");
        int PAGE_SIZE = 512;
        int FILE_SIZE = 513;
        FSDataOutputStream out = this.fs.create(path);
        byte[] data = new byte[513];
        Arrays.fill(data, (byte)5);
        out.write(data, 0, 513);
        out.close();
        FileStatus[] status = this.fs.listStatus(path);
        Assert.assertEquals((long)1L, (long)status.length);
        Assert.assertEquals((long)513L, (long)status[0].getLen());
    }

    private boolean testModifiedTime(Path testPath, long time) throws Exception {
        FileStatus fileStatus = this.fs.getFileStatus(testPath);
        long errorMargin = 5000L;
        long lastModified = fileStatus.getModificationTime();
        return lastModified > time - 5000L && lastModified < time + 5000L;
    }

    @Test
    public void testCreateNonRecursive() throws Exception {
        Path testFolder = new Path("/testFolder");
        Path testFile = new Path(testFolder, "testFile");
        try {
            this.fs.createNonRecursive(testFile, true, 1024, (short)1, 1024L, null);
            Assert.assertTrue((String)"Should've thrown", (boolean)false);
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        this.fs.mkdirs(testFolder);
        this.fs.createNonRecursive(testFile, true, 1024, (short)1, 1024L, null).close();
        Assert.assertTrue((boolean)this.fs.exists(testFile));
    }

    public void testFileEndingInDot() throws Exception {
        Path testFolder = new Path("/testFolder.");
        Path testFile = new Path(testFolder, "testFile.");
        Assert.assertTrue((boolean)this.fs.mkdirs(testFolder));
        Assert.assertTrue((boolean)this.fs.createNewFile(testFile));
        Assert.assertTrue((boolean)this.fs.exists(testFile));
        FileStatus[] listed = this.fs.listStatus(testFolder);
        Assert.assertEquals((long)1L, (long)listed.length);
        Assert.assertEquals((Object)"testFile.", (Object)listed[0].getPath().getName());
    }

    private void testModifiedTime(Path testPath) throws Exception {
        Calendar utc = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
        long currentUtcTime = utc.getTime().getTime();
        FileStatus fileStatus = this.fs.getFileStatus(testPath);
        long errorMargin = 10000L;
        Assert.assertTrue((String)("Modification time " + new Date(fileStatus.getModificationTime()) + " is not close to now: " + utc.getTime()), (fileStatus.getModificationTime() > currentUtcTime - 10000L && fileStatus.getModificationTime() < currentUtcTime + 10000L ? 1 : 0) != 0);
    }

    private void createEmptyFile(Path testFile, FsPermission permission) throws IOException {
        FSDataOutputStream outputStream = this.fs.create(testFile, permission, true, 4096, (short)1, 1024L, null);
        outputStream.close();
    }

    private String readString(Path testFile) throws IOException {
        return this.readString(this.fs, testFile);
    }

    private String readString(FileSystem fs, Path testFile) throws IOException {
        FSDataInputStream inputStream = fs.open(testFile);
        String ret = this.readString(inputStream);
        inputStream.close();
        return ret;
    }

    private String readString(FSDataInputStream inputStream) throws IOException {
        BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)inputStream));
        int BUFFER_SIZE = 1024;
        char[] buffer = new char[1024];
        int count = reader.read(buffer, 0, 1024);
        if (count > 1024) {
            throw new IOException("Exceeded buffer size");
        }
        inputStream.close();
        return new String(buffer, 0, count);
    }

    private void writeString(Path path, String value) throws IOException {
        this.writeString(this.fs, path, value);
    }

    private void writeString(FileSystem fs, Path path, String value) throws IOException {
        FSDataOutputStream outputStream = fs.create(path, true);
        this.writeString(outputStream, value);
    }

    private void writeString(FSDataOutputStream outputStream, String value) throws IOException {
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)outputStream));
        writer.write(value);
        writer.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSelfRenewingLease() throws IllegalArgumentException, IOException, InterruptedException, StorageException {
        String FILE_KEY = "file";
        this.fs.create(new Path("file"));
        NativeAzureFileSystem nfs = (NativeAzureFileSystem)this.fs;
        String fullKey = nfs.pathToKey(nfs.makeAbsolute(new Path("file")));
        AzureNativeFileSystemStore store = nfs.getStore();
        SelfRenewingLease lease = store.acquireLease(fullKey);
        Assert.assertTrue((lease.getLeaseID() != null ? 1 : 0) != 0);
        Thread.sleep(42000L);
        lease.free();
        CloudBlob blob = lease.getCloudBlob();
        String differentLeaseID = null;
        try {
            differentLeaseID = blob.acquireLease(Integer.valueOf(15), null);
            Assert.assertTrue((differentLeaseID != null ? 1 : 0) != 0);
        }
        catch (Exception e) {
            try {
                e.printStackTrace();
                Assert.fail((String)"Caught exception trying to directly re-acquire lease from Azure");
                Assert.assertTrue((differentLeaseID != null ? 1 : 0) != 0);
            }
            catch (Throwable throwable) {
                Assert.assertTrue((differentLeaseID != null ? 1 : 0) != 0);
                AccessCondition accessCondition = AccessCondition.generateEmptyCondition();
                accessCondition.setLeaseID(differentLeaseID);
                blob.releaseLease(accessCondition);
                throw throwable;
            }
            AccessCondition accessCondition = AccessCondition.generateEmptyCondition();
            accessCondition.setLeaseID(differentLeaseID);
            blob.releaseLease(accessCondition);
        }
        AccessCondition accessCondition = AccessCondition.generateEmptyCondition();
        accessCondition.setLeaseID(differentLeaseID);
        blob.releaseLease(accessCondition);
    }

    @Test
    public void testSelfRenewingLeaseFileDelete() throws IllegalArgumentException, IOException, InterruptedException, StorageException {
        String FILE_KEY = "file";
        Path path = new Path("file");
        this.fs.create(path);
        NativeAzureFileSystem nfs = (NativeAzureFileSystem)this.fs;
        String fullKey = nfs.pathToKey(nfs.makeAbsolute(path));
        SelfRenewingLease lease = nfs.getStore().acquireLease(fullKey);
        Assert.assertTrue((lease.getLeaseID() != null ? 1 : 0) != 0);
        Thread.sleep(42000L);
        nfs.getStore().delete(fullKey, lease);
        Assert.assertTrue((!this.fs.exists(path) ? 1 : 0) != 0);
        Assert.assertTrue((boolean)lease.isFreed());
    }

    @Test
    public void testLeaseAsDistributedLock() throws IllegalArgumentException, IOException {
        String LEASE_LOCK_FILE_KEY = "file";
        this.fs.create(new Path("file"));
        NativeAzureFileSystem nfs = (NativeAzureFileSystem)this.fs;
        String fullKey = nfs.pathToKey(nfs.makeAbsolute(new Path("file")));
        Thread first = new Thread(new LeaseLockAction("first-thread", fullKey));
        first.start();
        Thread second = new Thread(new LeaseLockAction("second-thread", fullKey));
        second.start();
        try {
            first.join();
            second.join();
            Assert.assertTrue((this.firstEndTime < this.secondStartTime ? 1 : 0) != 0);
        }
        catch (InterruptedException e) {
            Assert.fail((String)"Unable to wait for threads to finish");
            Thread.currentThread().interrupt();
        }
    }

    private class LeaseLockAction
    implements Runnable {
        private String name;
        private String key;

        LeaseLockAction(String name, String key) {
            this.name = name;
            this.key = key;
        }

        @Override
        public void run() {
            LOG.info((Object)("starting thread " + this.name));
            SelfRenewingLease lease = null;
            NativeAzureFileSystem nfs = (NativeAzureFileSystem)NativeAzureFileSystemBaseTest.this.fs;
            if (this.name.equals("first-thread")) {
                try {
                    lease = nfs.getStore().acquireLease(this.key);
                    LOG.info((Object)(this.name + " acquired lease " + lease.getLeaseID()));
                }
                catch (AzureException e) {
                    Assert.assertTrue((String)"Unanticipated exception", (boolean)false);
                }
                Assert.assertTrue((lease != null ? 1 : 0) != 0);
                try {
                    Thread.sleep(42000L);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                try {
                    NativeAzureFileSystemBaseTest.this.firstEndTime = System.currentTimeMillis();
                    lease.free();
                    LOG.info((Object)(this.name + " freed lease " + lease.getLeaseID()));
                }
                catch (StorageException e) {
                    Assert.fail((String)"Unanticipated exception");
                }
            } else if (this.name.equals("second-thread")) {
                try {
                    Thread.sleep(2000L);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                try {
                    LOG.info((Object)(this.name + " before getting lease"));
                    lease = nfs.getStore().acquireLease(this.key);
                    NativeAzureFileSystemBaseTest.this.secondStartTime = System.currentTimeMillis();
                    LOG.info((Object)(this.name + " acquired lease " + lease.getLeaseID()));
                }
                catch (AzureException e) {
                    Assert.assertTrue((String)"Unanticipated exception", (boolean)false);
                }
                Assert.assertTrue((lease != null ? 1 : 0) != 0);
                try {
                    lease.free();
                    LOG.info((Object)(this.name + " freed lease " + lease.getLeaseID()));
                }
                catch (StorageException e) {
                    Assert.assertTrue((String)"Unanticipated exception", (boolean)false);
                }
            } else {
                Assert.assertTrue((String)"Unknown thread name", (boolean)false);
            }
            LOG.info((Object)(this.name + " is exiting."));
        }
    }

    private class FileFolder {
        private String name;
        private boolean present;
        ArrayList<FileFolder> members;

        public FileFolder(String name) {
            this.name = name;
            this.present = true;
            this.members = new ArrayList();
        }

        public FileFolder getMember(int i) {
            return this.members.get(i);
        }

        public void verifyGone() throws IllegalArgumentException, IOException {
            Assert.assertFalse((boolean)NativeAzureFileSystemBaseTest.this.fs.exists(new Path(this.name)));
            Assert.assertTrue((boolean)this.isFolder());
            this.verifyGone(new Path(this.name), this.members);
        }

        private void verifyGone(Path prefix, ArrayList<FileFolder> members2) throws IOException {
            for (FileFolder f : members2) {
                f.verifyGone(prefix);
            }
        }

        private void verifyGone(Path prefix) throws IOException {
            Assert.assertFalse((boolean)NativeAzureFileSystemBaseTest.this.fs.exists(new Path(prefix, this.name)));
            if (this.isLeaf()) {
                return;
            }
            for (FileFolder f : this.members) {
                f.verifyGone(new Path(prefix, this.name));
            }
        }

        public void verifyExists() throws IllegalArgumentException, IOException {
            Assert.assertTrue((boolean)NativeAzureFileSystemBaseTest.this.fs.exists(new Path(this.name)));
            Assert.assertTrue((boolean)this.isFolder());
            this.verifyExists(new Path(this.name), this.members);
        }

        private void verifyExists(Path prefix, ArrayList<FileFolder> members2) throws IOException {
            for (FileFolder f : members2) {
                f.verifyExists(prefix);
            }
        }

        private void verifyExists(Path prefix) throws IOException {
            Assert.assertTrue((boolean)NativeAzureFileSystemBaseTest.this.fs.exists(new Path(prefix, this.name)));
            if (this.isLeaf()) {
                return;
            }
            for (FileFolder f : this.members) {
                f.verifyExists(new Path(prefix, this.name));
            }
        }

        public boolean exists() throws IOException {
            return NativeAzureFileSystemBaseTest.this.fs.exists(new Path(this.name));
        }

        public void makeRenamePending(FileFolder dst) throws IOException {
            Path home = NativeAzureFileSystemBaseTest.this.fs.getHomeDirectory();
            String relativeHomeDir = NativeAzureFileSystemBaseTest.this.getRelativePath(home.toString());
            NativeAzureFileSystem.FolderRenamePending pending = new NativeAzureFileSystem.FolderRenamePending(relativeHomeDir + "/" + this.getName(), relativeHomeDir + "/" + dst.getName(), null, (NativeAzureFileSystem)NativeAzureFileSystemBaseTest.this.fs);
            String renameDescription = pending.makeRenamePendingFileContents();
            String renamePendingStr = this.getName() + "-RenamePending.json";
            Path renamePendingFile = new Path(renamePendingStr);
            FSDataOutputStream out = NativeAzureFileSystemBaseTest.this.fs.create(renamePendingFile, true);
            Assert.assertTrue((out != null ? 1 : 0) != 0);
            NativeAzureFileSystemBaseTest.this.writeString(out, renameDescription);
        }

        public void setPresent(int i, boolean b) {
            this.members.get(i).setPresent(b);
        }

        private FileFolder() {
            this.present = true;
        }

        public void setPresent(boolean value) {
            this.present = value;
        }

        public FileFolder makeLeaf(String name) {
            FileFolder f = new FileFolder();
            f.setName(name);
            return f;
        }

        void setName(String name) {
            this.name = name;
        }

        public String getName() {
            return this.name;
        }

        public boolean isLeaf() {
            return this.members == null;
        }

        public boolean isFolder() {
            return this.members != null;
        }

        FileFolder add(FileFolder folder) {
            this.members.add(folder);
            return this;
        }

        FileFolder add(String file) {
            FileFolder leaf = this.makeLeaf(file);
            this.members.add(leaf);
            return this;
        }

        public FileFolder copy() {
            if (this.isLeaf()) {
                return this.makeLeaf(this.name);
            }
            FileFolder f = new FileFolder(this.name);
            for (FileFolder member : this.members) {
                f.add(member.copy());
            }
            return f;
        }

        public void create() throws IllegalArgumentException, IOException {
            this.create(null);
        }

        private void create(Path prefix) throws IllegalArgumentException, IOException {
            if (this.isFolder()) {
                if (this.present) {
                    Assert.assertTrue((boolean)NativeAzureFileSystemBaseTest.this.fs.mkdirs(this.makePath(prefix, this.name)));
                }
                this.create(this.makePath(prefix, this.name), this.members);
            } else if (this.isLeaf()) {
                if (this.present) {
                    Assert.assertTrue((boolean)NativeAzureFileSystemBaseTest.this.fs.createNewFile(this.makePath(prefix, this.name)));
                }
            } else {
                Assert.assertTrue((String)"The object must be a (leaf) file or a folder.", (boolean)false);
            }
        }

        private void create(Path prefix, ArrayList<FileFolder> members2) throws IllegalArgumentException, IOException {
            for (FileFolder f : members2) {
                f.create(prefix);
            }
        }

        private Path makePath(Path prefix, String name) {
            if (prefix == null) {
                return new Path(name);
            }
            return new Path(prefix, name);
        }

        public void prune() throws IOException {
            this.prune(null);
        }

        private void prune(Path prefix) throws IOException {
            Path path = null;
            path = prefix == null ? new Path(this.name) : new Path(prefix, this.name);
            if (this.isLeaf() && !this.present) {
                Assert.assertTrue((boolean)NativeAzureFileSystemBaseTest.this.fs.delete(path, false));
            } else if (this.isFolder() && !this.present) {
                Assert.assertTrue((boolean)NativeAzureFileSystemBaseTest.this.fs.delete(path, true));
            } else if (this.isFolder()) {
                for (FileFolder f : this.members) {
                    f.prune(path);
                }
            }
        }
    }

    private static enum RenameFolderVariation {
        CreateFolderAndInnerFile,
        CreateJustInnerFile,
        CreateJustFolder;

    }

    private static enum RenameVariation {
        NormalFileName,
        SourceInAFolder,
        SourceWithSpace,
        SourceWithPlusAndPercent;

    }
}

