/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode.erasurecode;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletionService;
import java.util.concurrent.Future;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.server.datanode.CachingStrategy;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.erasurecode.StripedBlockReader;
import org.apache.hadoop.hdfs.server.datanode.erasurecode.StripedReconstructionInfo;
import org.apache.hadoop.hdfs.server.datanode.erasurecode.StripedReconstructor;
import org.apache.hadoop.hdfs.util.StripedBlockUtil;
import org.apache.hadoop.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.util.DataChecksum;
import org.slf4j.Logger;

@InterfaceAudience.Private
class StripedReader {
    private static final Logger LOG = DataNode.LOG;
    private final int stripedReadTimeoutInMills;
    private final int stripedReadBufferSize;
    private StripedReconstructor reconstructor;
    private final DataNode datanode;
    private final Configuration conf;
    private final int dataBlkNum;
    private final int parityBlkNum;
    private DataChecksum checksum;
    private int bufferSize;
    private int[] successList;
    private final int minRequiredSources;
    private final int xmits;
    private ByteBuffer[] zeroStripeBuffers;
    private short[] zeroStripeIndices;
    private final byte[] liveIndices;
    private final DatanodeInfo[] sources;
    private final List<StripedBlockReader> readers;
    private final Map<Future<StripedBlockUtil.BlockReadStats>, Integer> futures = new HashMap<Future<StripedBlockUtil.BlockReadStats>, Integer>();
    private final CompletionService<StripedBlockUtil.BlockReadStats> readService;

    StripedReader(StripedReconstructor reconstructor, DataNode datanode, Configuration conf, StripedReconstructionInfo stripedReconInfo) {
        this.stripedReadTimeoutInMills = conf.getInt("dfs.datanode.ec.reconstruction.stripedread.timeout.millis", 5000);
        this.stripedReadBufferSize = conf.getInt("dfs.datanode.ec.reconstruction.stripedread.buffer.size", 65536);
        this.reconstructor = reconstructor;
        this.datanode = datanode;
        this.conf = conf;
        this.dataBlkNum = stripedReconInfo.getEcPolicy().getNumDataUnits();
        this.parityBlkNum = stripedReconInfo.getEcPolicy().getNumParityUnits();
        int cellsNum = (int)((stripedReconInfo.getBlockGroup().getNumBytes() - 1L) / (long)stripedReconInfo.getEcPolicy().getCellSize() + 1L);
        this.minRequiredSources = Math.min(cellsNum, this.dataBlkNum);
        if (this.minRequiredSources < this.dataBlkNum) {
            int zeroStripNum = this.dataBlkNum - this.minRequiredSources;
            this.zeroStripeBuffers = new ByteBuffer[zeroStripNum];
            this.zeroStripeIndices = new short[zeroStripNum];
        }
        this.xmits = Math.max(this.minRequiredSources, stripedReconInfo.getTargets() != null ? stripedReconInfo.getTargets().length : 0);
        this.liveIndices = stripedReconInfo.getLiveIndices();
        assert (this.liveIndices != null);
        this.sources = stripedReconInfo.getSources();
        assert (this.sources != null);
        this.readers = new ArrayList<StripedBlockReader>(this.sources.length);
        this.readService = reconstructor.createReadService();
        Preconditions.checkArgument((this.liveIndices.length >= this.minRequiredSources ? 1 : 0) != 0, (Object)"No enough live striped blocks.");
        Preconditions.checkArgument((this.liveIndices.length == this.sources.length ? 1 : 0) != 0, (Object)"liveBlockIndices and source datanodes should match");
    }

    void init() throws IOException {
        this.initReaders();
        this.initBufferSize();
        this.initZeroStrip();
    }

    private void initReaders() throws IOException {
        this.successList = new int[this.minRequiredSources];
        int nSuccess = 0;
        for (int i = 0; i < this.sources.length && nSuccess < this.minRequiredSources; ++i) {
            StripedBlockReader reader = this.createReader(i, 0L);
            this.readers.add(reader);
            if (reader.getBlockReader() == null) continue;
            this.initOrVerifyChecksum(reader);
            this.successList[nSuccess++] = i;
        }
        if (nSuccess < this.minRequiredSources) {
            String error = "Can't find minimum sources required by reconstruction, block id: " + this.reconstructor.getBlockGroup().getBlockId();
            throw new IOException(error);
        }
    }

    StripedBlockReader createReader(int idxInSources, long offsetInBlock) {
        return new StripedBlockReader(this, this.datanode, this.conf, this.liveIndices[idxInSources], this.reconstructor.getBlock(this.liveIndices[idxInSources]), this.sources[idxInSources], offsetInBlock);
    }

    private void initBufferSize() {
        int readBufferSize = this.stripedReadBufferSize;
        int bytesPerChecksum = this.checksum.getBytesPerChecksum();
        this.bufferSize = readBufferSize < bytesPerChecksum ? bytesPerChecksum : readBufferSize - readBufferSize % bytesPerChecksum;
    }

    private void initOrVerifyChecksum(StripedBlockReader reader) {
        if (this.checksum == null) {
            this.checksum = reader.getBlockReader().getDataChecksum();
        } else assert (reader.getBlockReader().getDataChecksum().equals((Object)this.checksum));
    }

    protected ByteBuffer allocateReadBuffer() {
        return this.reconstructor.allocateBuffer(this.getBufferSize());
    }

    private void initZeroStrip() {
        if (this.zeroStripeBuffers != null) {
            for (int i = 0; i < this.zeroStripeBuffers.length; ++i) {
                this.zeroStripeBuffers[i] = this.reconstructor.allocateBuffer(this.bufferSize);
            }
        }
        BitSet bitset = this.reconstructor.getLiveBitSet();
        int k = 0;
        for (int i = 0; i < this.dataBlkNum + this.parityBlkNum; ++i) {
            if (bitset.get(i) || this.reconstructor.getBlockLen(i) > 0L) continue;
            this.zeroStripeIndices[k++] = (short)i;
        }
    }

    private int getReadLength(int index, int reconstructLength) {
        long blockLen = this.reconstructor.getBlockLen(index);
        long remaining = blockLen - this.reconstructor.getPositionInBlock();
        return (int)Math.min(remaining, (long)reconstructLength);
    }

    ByteBuffer[] getInputBuffers(int toReconstructLen) {
        int i;
        ByteBuffer[] inputs = new ByteBuffer[this.dataBlkNum + this.parityBlkNum];
        for (i = 0; i < this.successList.length; ++i) {
            int index = this.successList[i];
            StripedBlockReader reader = this.getReader(index);
            ByteBuffer buffer = reader.getReadBuffer();
            this.paddingBufferToLen(buffer, toReconstructLen);
            inputs[reader.getIndex()] = (ByteBuffer)buffer.flip();
        }
        if (this.successList.length < this.dataBlkNum) {
            for (i = 0; i < this.zeroStripeBuffers.length; ++i) {
                ByteBuffer buffer = this.zeroStripeBuffers[i];
                this.paddingBufferToLen(buffer, toReconstructLen);
                short index = this.zeroStripeIndices[i];
                inputs[index] = (ByteBuffer)buffer.flip();
            }
        }
        return inputs;
    }

    private void paddingBufferToLen(ByteBuffer buffer, int len) {
        if (len > buffer.limit()) {
            buffer.limit(len);
        }
        int toPadding = len - buffer.position();
        for (int i = 0; i < toPadding; ++i) {
            buffer.put((byte)0);
        }
    }

    void readMinimumSources(int reconstructLength) throws IOException {
        DFSUtilClient.CorruptedBlocks corruptedBlocks = new DFSUtilClient.CorruptedBlocks();
        try {
            this.successList = this.doReadMinimumSources(reconstructLength, corruptedBlocks);
        }
        finally {
            this.datanode.reportCorruptedBlocks(corruptedBlocks);
        }
    }

    int[] doReadMinimumSources(int reconstructLength, DFSUtilClient.CorruptedBlocks corruptedBlocks) throws IOException {
        Preconditions.checkArgument((reconstructLength >= 0 && reconstructLength <= this.bufferSize ? 1 : 0) != 0);
        int nSuccess = 0;
        int[] newSuccess = new int[this.minRequiredSources];
        BitSet usedFlag = new BitSet(this.sources.length);
        for (int i = 0; i < this.minRequiredSources; ++i) {
            StripedBlockReader reader = this.readers.get(this.successList[i]);
            int toRead = this.getReadLength(this.liveIndices[this.successList[i]], reconstructLength);
            if (toRead > 0) {
                Callable<StripedBlockUtil.BlockReadStats> readCallable = reader.readFromBlock(toRead, corruptedBlocks);
                Future<StripedBlockUtil.BlockReadStats> f = this.readService.submit(readCallable);
                this.futures.put(f, this.successList[i]);
            } else {
                reader.getReadBuffer().position(0);
                newSuccess[nSuccess++] = this.successList[i];
            }
            usedFlag.set(this.successList[i]);
        }
        while (!this.futures.isEmpty()) {
            try {
                StripedBlockUtil.StripingChunkReadResult result = StripedBlockUtil.getNextCompletedStripedRead(this.readService, this.futures, (long)this.stripedReadTimeoutInMills);
                int resultIndex = -1;
                if (result.state == 1) {
                    resultIndex = result.index;
                } else if (result.state == 2) {
                    StripedBlockReader failedReader = this.readers.get(result.index);
                    failedReader.closeBlockReader();
                    resultIndex = this.scheduleNewRead(usedFlag, reconstructLength, corruptedBlocks);
                } else if (result.state == 4) {
                    resultIndex = this.scheduleNewRead(usedFlag, reconstructLength, corruptedBlocks);
                }
                if (resultIndex < 0) continue;
                newSuccess[nSuccess++] = resultIndex;
                if (nSuccess < this.minRequiredSources) continue;
                StripedReader.cancelReads(this.futures.keySet());
                this.futures.clear();
                break;
            }
            catch (InterruptedException e) {
                LOG.info("Read data interrupted.", (Throwable)e);
                StripedReader.cancelReads(this.futures.keySet());
                this.futures.clear();
                break;
            }
        }
        if (nSuccess < this.minRequiredSources) {
            String error = "Can't read data from minimum number of sources required by reconstruction, block id: " + this.reconstructor.getBlockGroup().getBlockId();
            throw new IOException(error);
        }
        return newSuccess;
    }

    private int scheduleNewRead(BitSet used, int reconstructLength, DFSUtilClient.CorruptedBlocks corruptedBlocks) {
        StripedBlockReader reader = null;
        int m = this.readers.size();
        int toRead = 0;
        while (reader == null && m < this.sources.length) {
            reader = this.createReader(m, this.reconstructor.getPositionInBlock());
            this.readers.add(reader);
            toRead = this.getReadLength(this.liveIndices[m], reconstructLength);
            if (toRead > 0) {
                if (reader.getBlockReader() != null) continue;
                reader = null;
                ++m;
                continue;
            }
            used.set(m);
            return m;
        }
        for (int i = 0; reader == null && i < this.readers.size(); ++i) {
            if (used.get(i)) continue;
            StripedBlockReader stripedReader = this.readers.get(i);
            toRead = this.getReadLength(this.liveIndices[i], reconstructLength);
            if (toRead > 0) {
                stripedReader.closeBlockReader();
                stripedReader.resetBlockReader(this.reconstructor.getPositionInBlock());
                if (stripedReader.getBlockReader() == null) continue;
                stripedReader.getReadBuffer().position(0);
                m = i;
                reader = stripedReader;
                continue;
            }
            used.set(i);
            stripedReader.getReadBuffer().position(0);
            return i;
        }
        if (reader != null) {
            Callable<StripedBlockUtil.BlockReadStats> readCallable = reader.readFromBlock(toRead, corruptedBlocks);
            Future<StripedBlockUtil.BlockReadStats> f = this.readService.submit(readCallable);
            this.futures.put(f, m);
            used.set(m);
        }
        return -1;
    }

    private static void cancelReads(Collection<Future<StripedBlockUtil.BlockReadStats>> futures) {
        for (Future<StripedBlockUtil.BlockReadStats> future : futures) {
            future.cancel(true);
        }
    }

    void close() {
        if (this.zeroStripeBuffers != null) {
            for (ByteBuffer zeroStripeBuffer : this.zeroStripeBuffers) {
                this.reconstructor.freeBuffer(zeroStripeBuffer);
            }
        }
        this.zeroStripeBuffers = null;
        for (StripedBlockReader reader : this.readers) {
            this.reconstructor.freeBuffer(reader.getReadBuffer());
            reader.freeReadBuffer();
            reader.closeBlockReader();
        }
    }

    StripedReconstructor getReconstructor() {
        return this.reconstructor;
    }

    StripedBlockReader getReader(int i) {
        return this.readers.get(i);
    }

    int getBufferSize() {
        return this.bufferSize;
    }

    DataChecksum getChecksum() {
        return this.checksum;
    }

    void clearBuffers() {
        if (this.zeroStripeBuffers != null) {
            for (ByteBuffer zeroStripeBuffer : this.zeroStripeBuffers) {
                zeroStripeBuffer.clear();
            }
        }
        for (StripedBlockReader reader : this.readers) {
            if (reader.getReadBuffer() == null) continue;
            reader.getReadBuffer().clear();
        }
    }

    InetSocketAddress getSocketAddress4Transfer(DatanodeInfo dnInfo) {
        return this.reconstructor.getSocketAddress4Transfer(dnInfo);
    }

    CachingStrategy getCachingStrategy() {
        return this.reconstructor.getCachingStrategy();
    }

    int getXmits() {
        return this.xmits;
    }
}

