/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.resourcemanager;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.yarn.api.records.ApplicationId;
import org.apache.hadoop.yarn.api.records.ContainerState;
import org.apache.hadoop.yarn.api.records.ContainerStatus;
import org.apache.hadoop.yarn.api.records.NodeId;
import org.apache.hadoop.yarn.api.records.NodeState;
import org.apache.hadoop.yarn.event.Event;
import org.apache.hadoop.yarn.server.api.records.NodeStatus;
import org.apache.hadoop.yarn.server.resourcemanager.RMContext;
import org.apache.hadoop.yarn.server.resourcemanager.rmapp.RMApp;
import org.apache.hadoop.yarn.server.resourcemanager.rmapp.RMAppState;
import org.apache.hadoop.yarn.server.resourcemanager.rmnode.RMNode;
import org.apache.hadoop.yarn.server.resourcemanager.rmnode.RMNodeEvent;
import org.apache.hadoop.yarn.server.resourcemanager.rmnode.RMNodeEventType;
import org.apache.hadoop.yarn.util.MonotonicClock;

public class DecommissioningNodesWatcher {
    private static final Log LOG = LogFactory.getLog(DecommissioningNodesWatcher.class);
    private final RMContext rmContext;
    private HashMap<NodeId, DecommissioningNodeContext> decomNodes = new HashMap();
    private Timer pollTimer;
    private MonotonicClock mclock;

    public DecommissioningNodesWatcher(RMContext rmContext) {
        this.rmContext = rmContext;
        this.pollTimer = new Timer(true);
        this.mclock = new MonotonicClock();
    }

    public void init(Configuration conf) {
        int v = conf.getInt("yarn.resourcemanager.decommissioning-nodes-watcher.poll-interval-secs", 20);
        this.pollTimer.schedule((TimerTask)new PollTimerTask(this.rmContext), 0L, 1000L * (long)v);
    }

    public synchronized void update(RMNode rmNode, NodeStatus remoteNodeStatus) {
        DecommissioningNodeContext context = this.decomNodes.get(rmNode.getNodeID());
        long now = this.mclock.getTime();
        if (rmNode.getState() == NodeState.DECOMMISSIONED) {
            if (context == null) {
                return;
            }
            context.nodeState = rmNode.getState();
            if (context.decommissionedTime == 0L) {
                context.decommissionedTime = now;
            } else if (now - context.decommissionedTime > 60000L) {
                this.decomNodes.remove(rmNode.getNodeID());
            }
        } else if (rmNode.getState() == NodeState.DECOMMISSIONING) {
            if (context == null) {
                context = new DecommissioningNodeContext(rmNode.getNodeID(), rmNode.getDecommissioningTimeout());
                this.decomNodes.put(rmNode.getNodeID(), context);
                context.nodeState = rmNode.getState();
                context.decommissionedTime = 0L;
            }
            context.updateTimeout(rmNode.getDecommissioningTimeout());
            context.lastUpdateTime = now;
            if (remoteNodeStatus.getKeepAliveApplications() != null) {
                context.appIds.addAll(remoteNodeStatus.getKeepAliveApplications());
            }
            int numActiveContainers = 0;
            for (ContainerStatus cs : remoteNodeStatus.getContainersStatuses()) {
                ContainerState newState = cs.getState();
                if (newState == ContainerState.RUNNING || newState == ContainerState.NEW) {
                    ++numActiveContainers;
                }
                context.numActiveContainers = numActiveContainers;
                ApplicationId aid = cs.getContainerId().getApplicationAttemptId().getApplicationId();
                if (context.appIds.contains(aid)) continue;
                context.appIds.add(aid);
            }
            context.numActiveContainers = numActiveContainers;
            if (context.numActiveContainers == 0 && context.lastContainerFinishTime == 0L) {
                context.lastContainerFinishTime = now;
            }
        } else if (context != null) {
            this.decomNodes.remove(rmNode.getNodeID());
        }
    }

    public synchronized void remove(NodeId nodeId) {
        DecommissioningNodeContext context = this.decomNodes.get(nodeId);
        if (context != null) {
            LOG.info((Object)("remove " + nodeId + " in " + context.nodeState));
            this.decomNodes.remove(nodeId);
        }
    }

    public void stop() {
        this.pollTimer.cancel();
        this.pollTimer = null;
    }

    public boolean checkReadyToBeDecommissioned(NodeId nodeId) {
        DecommissioningNodeStatus s = this.checkDecommissioningStatus(nodeId);
        return s == DecommissioningNodeStatus.READY || s == DecommissioningNodeStatus.TIMEOUT;
    }

    public DecommissioningNodeStatus checkDecommissioningStatus(NodeId nodeId) {
        DecommissioningNodeContext context = this.decomNodes.get(nodeId);
        if (context == null) {
            return DecommissioningNodeStatus.NONE;
        }
        if (context.nodeState == NodeState.DECOMMISSIONED) {
            return DecommissioningNodeStatus.DECOMMISSIONED;
        }
        long waitTime = this.mclock.getTime() - context.decommissioningStartTime;
        if (context.numActiveContainers > 0) {
            return context.timeoutMs < 0L || waitTime < context.timeoutMs ? DecommissioningNodeStatus.WAIT_CONTAINER : DecommissioningNodeStatus.TIMEOUT;
        }
        this.removeCompletedApps(context);
        if (context.appIds.size() == 0) {
            return DecommissioningNodeStatus.READY;
        }
        return context.timeoutMs < 0L || waitTime < context.timeoutMs ? DecommissioningNodeStatus.WAIT_APP : DecommissioningNodeStatus.TIMEOUT;
    }

    private RMNode getRmNode(NodeId nodeId) {
        RMNode rmNode = (RMNode)this.rmContext.getRMNodes().get(nodeId);
        if (rmNode == null) {
            rmNode = (RMNode)this.rmContext.getInactiveRMNodes().get(nodeId);
        }
        return rmNode;
    }

    private void removeCompletedApps(DecommissioningNodeContext context) {
        Iterator it = context.appIds.iterator();
        while (it.hasNext()) {
            ApplicationId appId = (ApplicationId)it.next();
            RMApp rmApp = (RMApp)this.rmContext.getRMApps().get(appId);
            if (rmApp == null) {
                LOG.debug((Object)("Consider non-existing app " + appId + " as completed"));
                it.remove();
                continue;
            }
            if (rmApp.getState() != RMAppState.FINISHED && rmApp.getState() != RMAppState.FAILED && rmApp.getState() != RMAppState.KILLED) continue;
            LOG.debug((Object)("Remove " + (Object)((Object)rmApp.getState()) + " app " + appId));
            it.remove();
        }
    }

    private int getTimeoutInSec(DecommissioningNodeContext context) {
        if (context.nodeState == NodeState.DECOMMISSIONED) {
            return 0;
        }
        if (context.nodeState != NodeState.DECOMMISSIONING) {
            return -1;
        }
        if (context.appIds.size() == 0 && context.numActiveContainers == 0) {
            return 0;
        }
        if (context.timeoutMs < 0L) {
            return -1;
        }
        long now = this.mclock.getTime();
        long timeout = context.decommissioningStartTime + context.timeoutMs - now;
        return Math.max(0, (int)(timeout / 1000L));
    }

    private void logDecommissioningNodesStatus() {
        if (!LOG.isDebugEnabled() || this.decomNodes.size() == 0) {
            return;
        }
        long now = this.mclock.getTime();
        for (DecommissioningNodeContext d : this.decomNodes.values()) {
            StringBuilder sb = new StringBuilder();
            DecommissioningNodeStatus s = this.checkDecommissioningStatus(d.nodeId);
            sb.append(String.format("%n  %-34s %4ds fresh:%3ds containers:%2d %14s", new Object[]{d.nodeId.getHost(), (now - d.decommissioningStartTime) / 1000L, (now - d.lastUpdateTime) / 1000L, d.numActiveContainers, s}));
            if (s == DecommissioningNodeStatus.WAIT_APP || s == DecommissioningNodeStatus.WAIT_CONTAINER) {
                sb.append(String.format(" timeout:%4ds", this.getTimeoutInSec(d)));
            }
            for (ApplicationId aid : d.appIds) {
                sb.append("\n    " + aid);
                RMApp rmApp = (RMApp)this.rmContext.getRMApps().get(aid);
                if (rmApp == null) continue;
                sb.append(String.format(" %s %9s %5.2f%% %5ds", new Object[]{rmApp.getState(), rmApp.getApplicationType() == null ? "" : rmApp.getApplicationType(), 100.0 * (double)rmApp.getProgress(), (this.mclock.getTime() - rmApp.getStartTime()) / 1000L}));
            }
            LOG.debug((Object)("Decommissioning node: " + sb.toString()));
        }
    }

    class PollTimerTask
    extends TimerTask {
        private final RMContext rmContext;

        public PollTimerTask(RMContext rmContext) {
            this.rmContext = rmContext;
        }

        @Override
        public void run() {
            DecommissioningNodesWatcher.this.logDecommissioningNodesStatus();
            long now = DecommissioningNodesWatcher.this.mclock.getTime();
            HashSet<NodeId> staleNodes = new HashSet<NodeId>();
            Iterator it = DecommissioningNodesWatcher.this.decomNodes.entrySet().iterator();
            while (it.hasNext()) {
                RMNode rmNode;
                Map.Entry e = it.next();
                DecommissioningNodeContext d = (DecommissioningNodeContext)e.getValue();
                if (now - d.lastUpdateTime < 5000L) continue;
                if (d.nodeState != NodeState.DECOMMISSIONING) {
                    LOG.debug((Object)("remove " + d.nodeState + " " + d.nodeId));
                    it.remove();
                    continue;
                }
                if (now - d.lastUpdateTime > 60000L && (rmNode = DecommissioningNodesWatcher.this.getRmNode(d.nodeId)) != null && rmNode.getState() == NodeState.DECOMMISSIONED) {
                    LOG.debug((Object)("remove " + rmNode.getState() + " " + d.nodeId));
                    it.remove();
                    continue;
                }
                if (d.timeoutMs < 0L || d.decommissioningStartTime + d.timeoutMs >= now) continue;
                staleNodes.add(d.nodeId);
                LOG.debug((Object)("Identified stale and timeout node " + d.nodeId));
            }
            for (NodeId nodeId : staleNodes) {
                RMNode rmNode = (RMNode)this.rmContext.getRMNodes().get(nodeId);
                if (rmNode == null || rmNode.getState() != NodeState.DECOMMISSIONING) {
                    DecommissioningNodesWatcher.this.remove(nodeId);
                    continue;
                }
                if (rmNode.getState() != NodeState.DECOMMISSIONING || !DecommissioningNodesWatcher.this.checkReadyToBeDecommissioned(rmNode.getNodeID())) continue;
                LOG.info((Object)("DECOMMISSIONING " + nodeId + " timeout"));
                this.rmContext.getDispatcher().getEventHandler().handle((Event)new RMNodeEvent(nodeId, RMNodeEventType.DECOMMISSION));
            }
        }
    }

    public static enum DecommissioningNodeStatus {
        NONE,
        WAIT_CONTAINER,
        WAIT_APP,
        TIMEOUT,
        READY,
        DECOMMISSIONED;

    }

    class DecommissioningNodeContext {
        private final NodeId nodeId;
        private NodeState nodeState;
        private final long decommissioningStartTime;
        private long lastContainerFinishTime;
        private int numActiveContainers;
        private Set<ApplicationId> appIds;
        private long decommissionedTime;
        private long timeoutMs;
        private long lastUpdateTime;

        public DecommissioningNodeContext(NodeId nodeId, int timeoutSec) {
            this.nodeId = nodeId;
            this.appIds = new HashSet<ApplicationId>();
            this.decommissioningStartTime = DecommissioningNodesWatcher.this.mclock.getTime();
            this.timeoutMs = 1000L * (long)timeoutSec;
        }

        void updateTimeout(int timeoutSec) {
            this.timeoutMs = 1000L * (long)timeoutSec;
        }
    }
}

