/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.diskbalancer;

import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import java.io.IOException;
import java.net.URI;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.DiskBalancer;
import org.apache.hadoop.hdfs.server.datanode.DiskBalancerWorkItem;
import org.apache.hadoop.hdfs.server.datanode.DiskBalancerWorkStatus;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.diskbalancer.DiskBalancerException;
import org.apache.hadoop.hdfs.server.diskbalancer.DiskBalancerResultVerifier;
import org.apache.hadoop.hdfs.server.diskbalancer.connectors.ClusterConnector;
import org.apache.hadoop.hdfs.server.diskbalancer.connectors.ConnectorFactory;
import org.apache.hadoop.hdfs.server.diskbalancer.datamodel.DiskBalancerCluster;
import org.apache.hadoop.hdfs.server.diskbalancer.datamodel.DiskBalancerDataNode;
import org.apache.hadoop.hdfs.server.diskbalancer.datamodel.DiskBalancerVolumeSet;
import org.apache.hadoop.hdfs.server.diskbalancer.planner.GreedyPlanner;
import org.apache.hadoop.hdfs.server.diskbalancer.planner.MoveStep;
import org.apache.hadoop.hdfs.server.diskbalancer.planner.NodePlan;
import org.apache.hadoop.hdfs.server.diskbalancer.planner.Step;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.util.Time;
import org.hamcrest.Matcher;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestDiskBalancerWithMockMover {
    static final Logger LOG = LoggerFactory.getLogger(TestDiskBalancerWithMockMover.class);
    @Rule
    public ExpectedException thrown = ExpectedException.none();
    private static final String PLAN_FILE = "/system/current.plan.json";
    private MiniDFSCluster cluster;
    private String sourceName;
    private String destName;
    private String sourceUUID;
    private String destUUID;
    private String nodeID;
    private DataNode dataNode;

    @Test
    public void testDiskBalancerDisabled() throws Exception {
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setBoolean("dfs.disk.balancer.enabled", false);
        this.restartDataNode();
        TestMover blockMover = new TestMover(this.cluster.getDataNodes().get(0).getFSDataset());
        DiskBalancer balancer = new DiskBalancerBuilder((Configuration)conf).setMover(blockMover).build();
        this.thrown.expect(DiskBalancerException.class);
        this.thrown.expect((Matcher)new DiskBalancerResultVerifier(DiskBalancerException.Result.DISK_BALANCER_NOT_ENABLED));
        balancer.queryWorkStatus();
    }

    @Test
    public void testDiskBalancerEnabled() throws DiskBalancerException {
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setBoolean("dfs.disk.balancer.enabled", true);
        TestMover blockMover = new TestMover(this.cluster.getDataNodes().get(0).getFSDataset());
        DiskBalancer balancer = new DiskBalancerBuilder((Configuration)conf).setMover(blockMover).build();
        DiskBalancerWorkStatus status = balancer.queryWorkStatus();
        Assert.assertEquals((Object)DiskBalancerWorkStatus.Result.NO_PLAN, (Object)status.getResult());
    }

    private void executeSubmitPlan(NodePlan plan, DiskBalancer balancer, int version) throws IOException {
        String planJson = plan.toJson();
        String planID = DigestUtils.shaHex((String)planJson);
        balancer.submitPlan(planID, (long)version, PLAN_FILE, planJson, false);
    }

    private void executeSubmitPlan(NodePlan plan, DiskBalancer balancer) throws IOException {
        this.executeSubmitPlan(plan, balancer, 1);
    }

    @Test
    public void testResubmitDiskBalancerPlan() throws Exception {
        MockMoverHelper mockMoverHelper = new MockMoverHelper().invoke();
        NodePlan plan = mockMoverHelper.getPlan();
        DiskBalancer balancer = mockMoverHelper.getBalancer();
        mockMoverHelper.getBlockMover().setSleep();
        this.executeSubmitPlan(plan, balancer);
        this.thrown.expect(DiskBalancerException.class);
        this.thrown.expect((Matcher)new DiskBalancerResultVerifier(DiskBalancerException.Result.PLAN_ALREADY_IN_PROGRESS));
        this.executeSubmitPlan(plan, balancer);
        mockMoverHelper.getBlockMover().clearSleep();
    }

    @Test
    public void testSubmitDiskBalancerPlan() throws Exception {
        MockMoverHelper mockMoverHelper = new MockMoverHelper().invoke();
        NodePlan plan = mockMoverHelper.getPlan();
        final DiskBalancer balancer = mockMoverHelper.getBalancer();
        this.executeSubmitPlan(plan, balancer);
        GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

            public Boolean get() {
                try {
                    return balancer.queryWorkStatus().getResult() == DiskBalancerWorkStatus.Result.PLAN_DONE;
                }
                catch (IOException ex) {
                    return false;
                }
            }
        }, (int)1000, (int)100000);
        Assert.assertTrue((mockMoverHelper.getBlockMover().getRunCount() == 1 ? 1 : 0) != 0);
    }

    @Test
    public void testSubmitWithOlderPlan() throws Exception {
        long millisecondInAnHour = 3600000L;
        MockMoverHelper mockMoverHelper = new MockMoverHelper().invoke();
        NodePlan plan = mockMoverHelper.getPlan();
        DiskBalancer balancer = mockMoverHelper.getBalancer();
        plan.setTimeStamp(Time.now() - 0x6DDD000L);
        this.thrown.expect(DiskBalancerException.class);
        this.thrown.expect((Matcher)new DiskBalancerResultVerifier(DiskBalancerException.Result.OLD_PLAN_SUBMITTED));
        this.executeSubmitPlan(plan, balancer);
    }

    @Test
    public void testSubmitWithOldInvalidVersion() throws Exception {
        MockMoverHelper mockMoverHelper = new MockMoverHelper().invoke();
        NodePlan plan = mockMoverHelper.getPlan();
        DiskBalancer balancer = mockMoverHelper.getBalancer();
        this.thrown.expect(DiskBalancerException.class);
        this.thrown.expect((Matcher)new DiskBalancerResultVerifier(DiskBalancerException.Result.INVALID_PLAN_VERSION));
        this.executeSubmitPlan(plan, balancer, 0);
    }

    @Test
    public void testSubmitWithNullPlan() throws Exception {
        MockMoverHelper mockMoverHelper = new MockMoverHelper().invoke();
        NodePlan plan = mockMoverHelper.getPlan();
        DiskBalancer balancer = mockMoverHelper.getBalancer();
        String planJson = plan.toJson();
        String planID = DigestUtils.shaHex((String)planJson);
        this.thrown.expect(DiskBalancerException.class);
        this.thrown.expect((Matcher)new DiskBalancerResultVerifier(DiskBalancerException.Result.INVALID_PLAN));
        balancer.submitPlan(planID, 1L, "no-plan-file.json", null, false);
    }

    @Test
    public void testSubmitWithInvalidHash() throws Exception {
        MockMoverHelper mockMoverHelper = new MockMoverHelper().invoke();
        NodePlan plan = mockMoverHelper.getPlan();
        DiskBalancer balancer = mockMoverHelper.getBalancer();
        String planJson = plan.toJson();
        String planID = DigestUtils.shaHex((String)planJson);
        char repChar = planID.charAt(0);
        repChar = (char)(repChar + '\u0001');
        this.thrown.expect(DiskBalancerException.class);
        this.thrown.expect((Matcher)new DiskBalancerResultVerifier(DiskBalancerException.Result.INVALID_PLAN_HASH));
        balancer.submitPlan(planID.replace(planID.charAt(0), repChar), 1L, PLAN_FILE, planJson, false);
    }

    @Test
    public void testCancelDiskBalancerPlan() throws Exception {
        MockMoverHelper mockMoverHelper = new MockMoverHelper().invoke();
        NodePlan plan = mockMoverHelper.getPlan();
        DiskBalancer balancer = mockMoverHelper.getBalancer();
        mockMoverHelper.getBlockMover().setSleep();
        this.executeSubmitPlan(plan, balancer);
        String planJson = plan.toJson();
        String planID = DigestUtils.shaHex((String)planJson);
        balancer.cancelPlan(planID);
        DiskBalancerWorkStatus status = balancer.queryWorkStatus();
        Assert.assertEquals((Object)DiskBalancerWorkStatus.Result.PLAN_CANCELLED, (Object)status.getResult());
        this.executeSubmitPlan(plan, balancer);
        char first = planID.charAt(0);
        first = (char)(first + '\u0001');
        this.thrown.expect(DiskBalancerException.class);
        this.thrown.expect((Matcher)new DiskBalancerResultVerifier(DiskBalancerException.Result.NO_SUCH_PLAN));
        balancer.cancelPlan(planID.replace(planID.charAt(0), first));
        balancer.cancelPlan(planID);
        mockMoverHelper.getBlockMover().clearSleep();
        status = balancer.queryWorkStatus();
        Assert.assertEquals((Object)DiskBalancerWorkStatus.Result.PLAN_CANCELLED, (Object)status.getResult());
    }

    @Test
    public void testCustomBandwidth() throws Exception {
        MockMoverHelper mockMoverHelper = new MockMoverHelper().invoke();
        NodePlan plan = mockMoverHelper.getPlan();
        DiskBalancer balancer = mockMoverHelper.getBalancer();
        for (Step step : plan.getVolumeSetPlans()) {
            MoveStep tempStep = (MoveStep)step;
            tempStep.setBandwidth(100L);
        }
        this.executeSubmitPlan(plan, balancer);
        DiskBalancerWorkStatus status = balancer.queryWorkStatus();
        Assert.assertNotNull((Object)status);
        DiskBalancerWorkStatus.DiskBalancerWorkEntry entry = (DiskBalancerWorkStatus.DiskBalancerWorkEntry)balancer.queryWorkStatus().getCurrentState().get(0);
        Assert.assertEquals((long)100L, (long)entry.getWorkItem().getBandwidth());
    }

    @Before
    public void setUp() throws Exception {
        HdfsConfiguration conf = new HdfsConfiguration();
        int numStoragesPerDn = 2;
        this.cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(3).storagesPerDatanode(2).build();
        this.cluster.waitActive();
        this.dataNode = this.cluster.getDataNodes().get(0);
        FsDatasetSpi.FsVolumeReferences references = this.dataNode.getFSDataset().getFsVolumeReferences();
        this.nodeID = this.dataNode.getDatanodeUuid();
        this.sourceName = references.get(0).getBaseURI().getPath();
        this.destName = references.get(1).getBaseURI().getPath();
        this.sourceUUID = references.get(0).getStorageID();
        this.destUUID = references.get(1).getStorageID();
        references.close();
    }

    @After
    public void tearDown() throws Exception {
        if (this.cluster != null) {
            this.cluster.shutdown();
        }
    }

    private void restartDataNode() throws IOException {
        if (this.cluster != null) {
            this.cluster.restartDataNode(0);
        }
    }

    private static class PlanBuilder {
        private String sourcePath;
        private String destPath;
        private String sourceUUID;
        private String destUUID;
        private DiskBalancerCluster balancerCluster;
        private String nodeID;

        public PlanBuilder(DiskBalancerCluster balancerCluster, String nodeID) {
            this.balancerCluster = balancerCluster;
            this.nodeID = nodeID;
        }

        public PlanBuilder setPathMap(String sourcePath, String destPath) {
            this.sourcePath = sourcePath;
            this.destPath = destPath;
            return this;
        }

        public PlanBuilder setUUIDMap(String sourceUUID, String destUUID) {
            this.sourceUUID = sourceUUID;
            this.destUUID = destUUID;
            return this;
        }

        public NodePlan build() throws Exception {
            boolean dnIndex = false;
            Preconditions.checkNotNull((Object)this.balancerCluster);
            Preconditions.checkState((this.nodeID.length() > 0 ? 1 : 0) != 0);
            DiskBalancerDataNode node = (DiskBalancerDataNode)this.balancerCluster.getNodes().get(0);
            node.setDataNodeUUID(this.nodeID);
            GreedyPlanner planner = new GreedyPlanner(10.0, node);
            NodePlan plan = new NodePlan(node.getDataNodeName(), node.getDataNodePort());
            planner.balanceVolumeSet(node, (DiskBalancerVolumeSet)node.getVolumeSets().get("DISK"), plan);
            this.setVolumeNames(plan);
            return plan;
        }

        private void setVolumeNames(NodePlan plan) {
            for (MoveStep nextStep : plan.getVolumeSetPlans()) {
                nextStep.getSourceVolume().setPath(this.sourcePath);
                nextStep.getSourceVolume().setUuid(this.sourceUUID);
                nextStep.getDestinationVolume().setPath(this.destPath);
                nextStep.getDestinationVolume().setUuid(this.destUUID);
            }
        }
    }

    private static class DiskBalancerClusterBuilder {
        private String jsonFilePath;
        private Configuration conf;

        private DiskBalancerClusterBuilder() {
        }

        public DiskBalancerClusterBuilder setConf(Configuration conf) {
            this.conf = conf;
            return this;
        }

        public DiskBalancerClusterBuilder setClusterSource(String jsonFilePath) throws Exception {
            this.jsonFilePath = jsonFilePath;
            return this;
        }

        public DiskBalancerCluster build() throws Exception {
            URI clusterJson = this.getClass().getResource(this.jsonFilePath).toURI();
            ClusterConnector jsonConnector = ConnectorFactory.getCluster((URI)clusterJson, (Configuration)this.conf);
            DiskBalancerCluster diskBalancerCluster = new DiskBalancerCluster(jsonConnector);
            diskBalancerCluster.readClusterInfo();
            diskBalancerCluster.setNodesToProcess(diskBalancerCluster.getNodes());
            return diskBalancerCluster;
        }
    }

    private static class DiskBalancerBuilder {
        private TestMover blockMover;
        private Configuration conf;
        private String nodeID;

        public DiskBalancerBuilder(Configuration conf) {
            this.conf = conf;
        }

        public DiskBalancerBuilder setNodeID(String nodeID) {
            this.nodeID = nodeID;
            return this;
        }

        public DiskBalancerBuilder setConf(Configuration conf) {
            this.conf = conf;
            return this;
        }

        public DiskBalancerBuilder setMover(TestMover mover) {
            this.blockMover = mover;
            return this;
        }

        public DiskBalancerBuilder setRunnable() {
            this.blockMover.setRunnable();
            return this;
        }

        public DiskBalancer build() {
            Preconditions.checkNotNull((Object)this.blockMover);
            return new DiskBalancer(this.nodeID, this.conf, (DiskBalancer.BlockMover)this.blockMover);
        }
    }

    private class MockMoverHelper {
        private DiskBalancer balancer;
        private NodePlan plan;
        private TestMover blockMover;

        private MockMoverHelper() {
        }

        public DiskBalancer getBalancer() {
            return this.balancer;
        }

        public NodePlan getPlan() {
            return this.plan;
        }

        public TestMover getBlockMover() {
            return this.blockMover;
        }

        public MockMoverHelper invoke() throws Exception {
            HdfsConfiguration conf = new HdfsConfiguration();
            conf.setBoolean("dfs.disk.balancer.enabled", true);
            TestDiskBalancerWithMockMover.this.restartDataNode();
            this.blockMover = new TestMover(TestDiskBalancerWithMockMover.this.dataNode.getFSDataset());
            this.blockMover.setRunnable();
            this.balancer = new DiskBalancerBuilder((Configuration)conf).setMover(this.blockMover).setNodeID(TestDiskBalancerWithMockMover.this.nodeID).build();
            DiskBalancerCluster diskBalancerCluster = new DiskBalancerClusterBuilder().setClusterSource("/diskBalancer/data-cluster-3node-3disk.json").build();
            this.plan = new PlanBuilder(diskBalancerCluster, TestDiskBalancerWithMockMover.this.nodeID).setPathMap(TestDiskBalancerWithMockMover.this.sourceName, TestDiskBalancerWithMockMover.this.destName).setUUIDMap(TestDiskBalancerWithMockMover.this.sourceUUID, TestDiskBalancerWithMockMover.this.destUUID).build();
            return this;
        }
    }

    public static class TestMover
    implements DiskBalancer.BlockMover {
        private AtomicBoolean shouldRun;
        private FsDatasetSpi dataset;
        private int runCount;
        private volatile boolean sleepInCopyBlocks;
        private long delay;

        public TestMover(FsDatasetSpi dataset) {
            this.dataset = dataset;
            this.shouldRun = new AtomicBoolean(false);
        }

        public void setSleep() {
            this.sleepInCopyBlocks = true;
        }

        public void clearSleep() {
            this.sleepInCopyBlocks = false;
        }

        public void setDelay(long milliseconds) {
            this.delay = milliseconds;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void copyBlocks(DiskBalancer.VolumePair pair, DiskBalancerWorkItem item) {
            try {
                while (this.sleepInCopyBlocks) {
                    if (!this.shouldRun()) {
                        return;
                    }
                    Thread.sleep(10L);
                }
                if (this.delay > 0L) {
                    Thread.sleep(this.delay);
                }
                TestMover testMover = this;
                synchronized (testMover) {
                    if (this.shouldRun()) {
                        ++this.runCount;
                    }
                }
            }
            catch (InterruptedException ex) {
                LOG.error(ex.toString());
            }
        }

        public void setRunnable() {
            this.shouldRun.set(true);
        }

        public void setExitFlag() {
            this.shouldRun.set(false);
        }

        public boolean shouldRun() {
            return this.shouldRun.get();
        }

        public FsDatasetSpi getDataset() {
            return this.dataset;
        }

        public long getStartTime() {
            return 0L;
        }

        public long getElapsedSeconds() {
            return 0L;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public int getRunCount() {
            TestMover testMover = this;
            synchronized (testMover) {
                LOG.info("Run count : " + this.runCount);
                return this.runCount;
            }
        }
    }
}

