/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util.compaction;

import ch.cern.hbase.thirdparty.com.google.common.annotations.VisibleForTesting;
import ch.cern.hbase.thirdparty.com.google.common.base.Joiner;
import ch.cern.hbase.thirdparty.com.google.common.base.Splitter;
import ch.cern.hbase.thirdparty.com.google.common.collect.Iterables;
import ch.cern.hbase.thirdparty.com.google.common.collect.Lists;
import ch.cern.hbase.thirdparty.com.google.common.collect.Sets;
import ch.cern.hbase.thirdparty.org.apache.commons.cli.CommandLine;
import ch.cern.hbase.thirdparty.org.apache.commons.cli.DefaultParser;
import ch.cern.hbase.thirdparty.org.apache.commons.cli.HelpFormatter;
import ch.cern.hbase.thirdparty.org.apache.commons.cli.Option;
import ch.cern.hbase.thirdparty.org.apache.commons.cli.Options;
import ch.cern.hbase.thirdparty.org.apache.commons.cli.ParseException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.NotServingRegionException;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.CompactionState;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.compaction.ClusterCompactionQueues;
import org.apache.hadoop.hbase.util.compaction.MajorCompactionRequest;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"Tools"})
public class MajorCompactor {
    private static final Logger LOG = LoggerFactory.getLogger(MajorCompactor.class);
    private static final Set<MajorCompactionRequest> ERRORS = ConcurrentHashMap.newKeySet();
    private final ClusterCompactionQueues clusterCompactionQueues;
    private final long timestamp;
    private final Set<String> storesToCompact;
    private final ExecutorService executor;
    private final long sleepForMs;
    private final Connection connection;
    private final TableName tableName;

    public MajorCompactor(Configuration conf, TableName tableName, Set<String> storesToCompact, int concurrency, long timestamp, long sleepForMs) throws IOException {
        this.connection = ConnectionFactory.createConnection((Configuration)conf);
        this.tableName = tableName;
        this.timestamp = timestamp;
        this.storesToCompact = storesToCompact;
        this.executor = Executors.newFixedThreadPool(concurrency);
        this.clusterCompactionQueues = new ClusterCompactionQueues(concurrency);
        this.sleepForMs = sleepForMs;
    }

    public void compactAllRegions() throws Exception {
        ArrayList futures = Lists.newArrayList();
        while (this.clusterCompactionQueues.hasWorkItems() || !this.futuresComplete(futures)) {
            while (this.clusterCompactionQueues.atCapacity()) {
                LOG.debug("Waiting for servers to complete Compactions");
                Thread.sleep(this.sleepForMs);
            }
            Optional<ServerName> serverToProcess = this.clusterCompactionQueues.getLargestQueueFromServersNotCompacting();
            if (serverToProcess.isPresent() && this.clusterCompactionQueues.hasWorkItems()) {
                ServerName serverName = serverToProcess.get();
                MajorCompactionRequest request = this.clusterCompactionQueues.reserveForCompaction(serverName);
                ServerName currentServer = this.connection.getRegionLocator(this.tableName).getRegionLocation(request.getRegion().getStartKey()).getServerName();
                if (!currentServer.equals((Object)serverName)) {
                    LOG.info("Server changed for region: " + request.getRegion().getEncodedName() + " from: " + serverName + " to: " + currentServer + " re-queuing request");
                    this.clusterCompactionQueues.addToCompactionQueue(currentServer, request);
                    this.clusterCompactionQueues.releaseCompaction(serverName);
                    continue;
                }
                LOG.info("Firing off compaction request for server: " + serverName + ", " + request + " total queue size left: " + this.clusterCompactionQueues.getCompactionRequestsLeftToFinish());
                futures.add(this.executor.submit(new Compact(serverName, request)));
                continue;
            }
            Thread.sleep(this.sleepForMs);
        }
        LOG.info("All compactions have completed");
    }

    private boolean futuresComplete(List<Future<?>> futures) {
        futures.removeIf(Future::isDone);
        return futures.isEmpty();
    }

    public void shutdown() throws Exception {
        this.executor.shutdown();
        this.executor.awaitTermination(Long.MAX_VALUE, TimeUnit.MILLISECONDS);
        if (!ERRORS.isEmpty()) {
            StringBuilder builder = new StringBuilder().append("Major compaction failed, there were: ").append(ERRORS.size()).append(" regions / stores that failed compacting\n").append("Failed compaction requests\n").append("--------------------------\n").append(Joiner.on((String)"\n").join(ERRORS));
            LOG.error(builder.toString());
        }
        if (this.connection != null) {
            this.connection.close();
        }
        LOG.info("All regions major compacted successfully");
    }

    @VisibleForTesting
    void initializeWorkQueues() throws IOException {
        if (this.storesToCompact.isEmpty()) {
            this.connection.getTable(this.tableName).getDescriptor().getColumnFamilyNames().forEach(a -> this.storesToCompact.add(Bytes.toString((byte[])a)));
            LOG.info("No family specified, will execute for all families");
        }
        LOG.info("Initializing compaction queues for table:  " + this.tableName + " with cf: " + this.storesToCompact);
        List regionLocations = this.connection.getRegionLocator(this.tableName).getAllRegionLocations();
        for (HRegionLocation location : regionLocations) {
            Optional<MajorCompactionRequest> request = MajorCompactionRequest.newRequest(this.connection.getConfiguration(), location.getRegion(), this.storesToCompact, this.timestamp);
            request.ifPresent(majorCompactionRequest -> this.clusterCompactionQueues.addToCompactionQueue(location.getServerName(), (MajorCompactionRequest)majorCompactionRequest));
        }
    }

    private boolean isCompacting(MajorCompactionRequest request) throws Exception {
        CompactionState compactionState = this.connection.getAdmin().getCompactionStateForRegion(request.getRegion().getEncodedNameAsBytes());
        return compactionState.equals((Object)CompactionState.MAJOR) || compactionState.equals((Object)CompactionState.MAJOR_AND_MINOR);
    }

    private void addNewRegions() {
        try {
            List locations = this.connection.getRegionLocator(this.tableName).getAllRegionLocations();
            for (HRegionLocation location : locations) {
                if (location.getRegion().getRegionId() <= this.timestamp) continue;
                Optional<MajorCompactionRequest> compactionRequest = MajorCompactionRequest.newRequest(this.connection.getConfiguration(), location.getRegion(), this.storesToCompact, this.timestamp);
                compactionRequest.ifPresent(request -> this.clusterCompactionQueues.addToCompactionQueue(location.getServerName(), (MajorCompactionRequest)request));
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void main(String[] args) throws Exception {
        Options options = new Options();
        options.addOption(Option.builder((String)"table").required().desc("table name").hasArg().build());
        options.addOption(Option.builder((String)"cf").optionalArg(true).desc("column families: comma separated eg: a,b,c").hasArg().build());
        options.addOption(Option.builder((String)"servers").required().desc("Concurrent servers compacting").hasArg().build());
        options.addOption(Option.builder((String)"minModTime").desc("Compact if store files have modification time < minModTime").hasArg().build());
        options.addOption(Option.builder((String)"zk").optionalArg(true).desc("zk quorum").hasArg().build());
        options.addOption(Option.builder((String)"rootDir").optionalArg(true).desc("hbase.rootDir").hasArg().build());
        options.addOption(Option.builder((String)"sleep").desc("Time to sleepForMs (ms) for checking compaction status per region and available work queues: default 30s").hasArg().build());
        options.addOption(Option.builder((String)"retries").desc("Max # of retries for a compaction request, defaults to 3").hasArg().build());
        options.addOption(Option.builder((String)"dryRun").desc("Dry run, will just output a list of regions that require compaction based on parameters passed").hasArg(false).build());
        DefaultParser cmdLineParser = new DefaultParser();
        CommandLine commandLine = null;
        try {
            commandLine = cmdLineParser.parse(options, args);
        }
        catch (ParseException parseException) {
            System.out.println("ERROR: Unable to parse command-line arguments " + Arrays.toString(args) + " due to: " + (Object)((Object)parseException));
            MajorCompactor.printUsage(options);
            return;
        }
        if (commandLine == null) {
            System.out.println("ERROR: Failed parse, empty commandLine; " + Arrays.toString(args));
            MajorCompactor.printUsage(options);
            return;
        }
        String tableName = commandLine.getOptionValue("table");
        String cf = commandLine.getOptionValue("cf", null);
        HashSet families = Sets.newHashSet();
        if (cf != null) {
            Iterables.addAll((Collection)families, (Iterable)Splitter.on((String)",").split((CharSequence)cf));
        }
        Configuration configuration = HBaseConfiguration.create();
        int concurrency = Integer.parseInt(commandLine.getOptionValue("servers"));
        long minModTime = Long.parseLong(commandLine.getOptionValue("minModTime", String.valueOf(System.currentTimeMillis())));
        String quorum = commandLine.getOptionValue("zk", configuration.get("hbase.zookeeper.quorum"));
        String rootDir = commandLine.getOptionValue("rootDir", configuration.get("hbase.rootdir"));
        long sleep = Long.parseLong(commandLine.getOptionValue("sleep", Long.toString(30000L)));
        configuration.set("hbase.rootdir", rootDir);
        configuration.set("hbase.zookeeper.quorum", quorum);
        MajorCompactor compactor = new MajorCompactor(configuration, TableName.valueOf((String)tableName), families, concurrency, minModTime, sleep);
        compactor.initializeWorkQueues();
        if (!commandLine.hasOption("dryRun")) {
            compactor.compactAllRegions();
        }
        compactor.shutdown();
    }

    private static void printUsage(Options options) {
        String header = "\nUsage instructions\n\n";
        String footer = "\n";
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp(MajorCompactor.class.getSimpleName(), header, options, footer, true);
    }

    class Compact
    implements Runnable {
        private final ServerName serverName;
        private final MajorCompactionRequest request;

        Compact(ServerName serverName, MajorCompactionRequest request) {
            this.serverName = serverName;
            this.request = request;
        }

        @Override
        public void run() {
            try {
                this.compactAndWait(this.request);
            }
            catch (NotServingRegionException e) {
                LOG.warn("Region is invalid, requesting updated regions", (Throwable)e);
                MajorCompactor.this.addNewRegions();
            }
            catch (Exception e) {
                LOG.warn("Error compacting:", (Throwable)e);
            }
            finally {
                MajorCompactor.this.clusterCompactionQueues.releaseCompaction(this.serverName);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void compactAndWait(MajorCompactionRequest request) throws Exception {
            Admin admin = MajorCompactor.this.connection.getAdmin();
            try {
                Set<String> stores;
                if (!MajorCompactor.this.isCompacting(request) && !(stores = request.getStoresRequiringCompaction(MajorCompactor.this.storesToCompact)).isEmpty()) {
                    request.setStores(stores);
                    for (String store : request.getStores()) {
                        admin.majorCompactRegion(request.getRegion().getEncodedNameAsBytes(), Bytes.toBytes((String)store));
                    }
                }
                while (MajorCompactor.this.isCompacting(request)) {
                    Thread.sleep(MajorCompactor.this.sleepForMs);
                    LOG.debug("Waiting for compaction to complete for region: " + request.getRegion().getEncodedName());
                }
            }
            finally {
                int waitForArchive = MajorCompactor.this.connection.getConfiguration().getInt("hbase.hfile.compaction.discharger.interval", 120000);
                Thread.sleep(waitForArchive);
                Set<String> storesRequiringCompaction = request.getStoresRequiringCompaction(MajorCompactor.this.storesToCompact);
                if (!storesRequiringCompaction.isEmpty()) {
                    boolean regionHasNotMoved = MajorCompactor.this.connection.getRegionLocator(MajorCompactor.this.tableName).getRegionLocation(request.getRegion().getStartKey()).getServerName().equals((Object)this.serverName);
                    if (regionHasNotMoved) {
                        LOG.error("Not all store files were compacted, this may be due to the regionserver not being aware of all store files.  Will not reattempt compacting, " + request);
                        ERRORS.add(request);
                    } else {
                        request.setStores(storesRequiringCompaction);
                        MajorCompactor.this.clusterCompactionQueues.addToCompactionQueue(this.serverName, request);
                        LOG.info("Compaction failed for the following stores: " + storesRequiringCompaction + " region: " + request.getRegion().getEncodedName());
                    }
                } else {
                    LOG.info("Compaction complete for region: " + request.getRegion().getEncodedName() + " -> cf(s): " + request.getStores());
                }
            }
        }
    }
}

