package ch.vd.shared.iam.web.common;

import ch.vd.shared.iam.core.filter.auth.IamAuthority;
import ch.vd.shared.iam.web.filter.auth.IamRequestWrapper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Set;

public class RequestHelper {

    public static final Logger LOGGER = LoggerFactory.getLogger("iam.http.request");
    public static final Logger HEADER_LOGGER = LoggerFactory.getLogger("iam.http.headers");

    public static void logRequestInfos(IamRequestWrapper request, HttpServletResponse response, String prefix, String username) {
        Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        Collection<IamAuthority> auths = null;
        if (auth != null) {
            auths = (Collection) auth.getAuthorities();
        }
        logRequestInfos(request, response, prefix, username, auths);
    }

    @SuppressWarnings("unchecked")
    public static void logRequestInfos(IamRequestWrapper request, HttpServletResponse response, String prefix, String username, Collection<IamAuthority> auths) {
        if (LOGGER.isDebugEnabled()) {
            String contextPath = request.getContextPath();
            String requestUri = request.getRequestURI();
            String url = requestUri.substring(contextPath.length());

            // Logging de la requete
            StringBuilder msg = new StringBuilder();
            msg.append(prefix).append(" '").append(normalize(url, 30)).append("'");

            // User
            if (username != null) {
                msg.append(" by ").append(normalize(username, 10));
            } else {
                msg.append(" not logged");
            }

            // Method
            msg.append(" ").append(normalize(request.getMethod(), 4));

            // Authorities
            if (auths != null) {
                for (IamAuthority ga : auths) {
                    msg.append(" [").append(ga.getAuthority()).append("]");
                }
            }

            // ContentType
            if (response != null) {
                if (response.getContentType() != null) {
                    msg.append(" ContentType:'").append(response.getContentType()).append("'");
                }
            }

            // URI en entier
            msg.append(" Url:'").append(url).append("'");

            // Params
            if (request.getParameterMap().size() > 0) {
                msg.append(" Params:");
                for (String name : (Set<String>) request.getParameterMap().keySet()) {
                    String param = request.getParameter(name);
                    if (param != null) {
                        param = StringUtils.replace(param, "\n", "");
                        param = StringUtils.replace(param, "\r", "");
                    }
                    msg.append(" (" + name + ">'" + param + "')");
                }
            }
            LOGGER.debug("" + msg);
        }
    }

    public static String normalize(String in, int len) {
        return normalize(in, len, false);
    }

    public static String normalize(String in, int len, boolean atBegin) {
        String out;
        if (in.length() > len) {
            out = in.substring(0, len);
        } else if (atBegin) {
            StringBuilder str = new StringBuilder(in);
            for (int i = in.length(); i < len; i++) {
                str.insert(0, " ");
            }
            out = str.toString();
        } else { // At end
            out = String.format("%1$-" + len + "s", in);
        }
        return out;
    }

    @SuppressWarnings("unchecked")
    public static void dumpRequestForDebug(HttpServletRequest request) {

        if (HEADER_LOGGER.isDebugEnabled()) {
            HEADER_LOGGER.debug("Request for: " + request.getRequestURL());

            // Attributes
            {
                Enumeration<String> enu = request.getHeaderNames();
                while (enu.hasMoreElements()) {
                    String name = enu.nextElement();
                    Object value = request.getHeader(name);
                    HEADER_LOGGER.debug("Header: " + name + " => " + value);
                }
            }
            // Parameters
            {
                for (Object key : request.getParameterMap().keySet()) {
                    Object value = request.getParameterMap().get(key);
                    if (value != null) {
                        String toDump;
                        if (value instanceof String[]) {
                            StringBuilder str = new StringBuilder();
                            for (String s : (String[]) value) {
                                if (str.length() > 0) {
                                    str.append(",");
                                }
                                str.append(s);
                            }
                            toDump = str.toString();
                        } else {
                            toDump = value.toString();
                        }
                        HEADER_LOGGER.debug("Parameter: " + key + " => " + toDump);
                    }
                }
            }
            // Attributes
            {
                Enumeration<String> enu = request.getAttributeNames();
                while (enu.hasMoreElements()) {
                    String name = enu.nextElement();
                    Object value = request.getAttribute(name);
                    HEADER_LOGGER.debug("Attribute: " + name + " => " + value);
                }
            }
            HEADER_LOGGER.debug("End Request " + request.getRequestURL());
        }
    }
}
