package ch.vd.shared.iam.web.filter.auth;

import ch.vd.shared.iam.core.filter.auth.AbstractIamDetails;
import ch.vd.shared.iam.core.filter.auth.IamAuthentication;
import ch.vd.shared.iam.core.filter.auth.IamAuthority;
import ch.vd.shared.iam.web.utils.IamWebUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.filter.GenericFilterBean;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.*;

import static ch.vd.shared.iam.web.utils.IamWebUtils.convertToAuthority;

public abstract class AbstractSharedIamAuthenticationFilter extends GenericFilterBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractSharedIamAuthenticationFilter.class);
    public static final String IAM_AUTH_SAVED_PATH = "iam-auth-saved-path";

    private AdditionalInformationProvider additionalProvider;

    static void relativeRedirect(IamRequestWrapper wrapper, HttpServletResponse response, String url) throws IOException {
        response.sendRedirect(IamWebUtils.calculateRedirect(wrapper.getRequest(), url));
    }

    protected void authenticate(AuthenticateDTO authenticateDTO) {
        Date lastLoginTime;
        Integer authLevel = authenticateDTO.getAuthLevelStr();
        lastLoginTime = IamWebUtils.parseIsoTime(authenticateDTO.getLastLoginTimeStr());

        // Tout va bien, on authentifie
        SecurityContextHolder.clearContext();
        SecurityContext ctx = SecurityContextHolder.createEmptyContext();
        SecurityContextHolder.setContext(ctx);

        List<String> iamRoles = IamWebUtils.parseRoles(authenticateDTO.getApplication(), authenticateDTO.getRolesStr());
        Set<IamAuthority> allRoles = new HashSet<>(convertToAuthority(iamRoles));

        // Récupération d'éventuelles données propres à chaque application (droits IFO-SEC, données de la DB, etc...)
        AbstractIamDetails iamDetails = null;
        if (additionalProvider != null) {
            iamDetails = additionalProvider.createIamDetails(new AuthenticateDtoProvider(authenticateDTO, allRoles));
            if (iamDetails != null) {
                allRoles.addAll(iamDetails.getAdditionalRoles());
            }
        }

        // Authentication
        IamAuthentication auth = new IamAuthentication(
                authenticateDTO.getApplication(), authenticateDTO.getUsedPrincipal(), authenticateDTO.getUsername(), authenticateDTO.getFirstName(),
                authenticateDTO.getLastName(), authenticateDTO.getEmail(), iamRoles, authLevel, lastLoginTime, allRoles
        );
        auth.setDetails(iamDetails);
        ctx.setAuthentication(auth);
    }

    protected String getUsernameForLog(IamRequestWrapper request) {
        String reqUsername = getUsedPrincipal(request);
        Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        String authUsername;
        if (auth != null) {
            authUsername = auth.getName();
        } else {
            authUsername = null;
        }

        if (reqUsername != null) {
            return reqUsername;
        } else if (authUsername != null) {
            return authUsername;
        }
        return "<unknown>";
    }

    protected String getUsedPrincipal(IamRequestWrapper request) {
        return request.getUserName();
    }

    public void setAdditionalProvider(AdditionalInformationProvider additionalProvider) {
        this.additionalProvider = additionalProvider;
    }

}
