package ch.vd.shared.iam.web.filter.auth;

import ch.vd.shared.iam.core.filter.auth.IamAuthentication;
import ch.vd.shared.iam.web.common.IamConstants;
import ch.vd.shared.iam.web.common.RequestHelper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Objects;

public class FormDevLoginSpringFilter extends AbstractSharedIamAuthenticationFilter {

    private static final Logger LOGGER = LoggerFactory.getLogger(FormDevLoginSpringFilter.class);

    public static final String DEV_LOGIN_USERNAME = "dl-username";
    public static final String DEV_LOGIN_ROLES = "dl-roles";
    public static final String DEV_LOGIN_FIRST = "dl-first";
    public static final String DEV_LOGIN_LAST = "dl-last";
    public static final String DEV_LOGIN_EMAIL = "dl-email";
    public static final String SPRING_SECURITY_URL = "/j_spring_security_check";

    private String successUrl;
    private String deniedUrl;

    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;

        IamRequestWrapper wrapper = new IamRequestWrapper(request);

        // Log les headers de la request
        // Le log n'est effectué que si iam.http.headers est mis sur INFO
        RequestHelper.dumpRequestForDebug(request);

        String contextPath = request.getContextPath();
        String requestUri = request.getRequestURI();
        String url = requestUri.substring(contextPath.length());

        // Si on est deja authentifié, on laisse passer
        Authentication givenAuth = SecurityContextHolder.getContext().getAuthentication();
        if (givenAuth != null && givenAuth.isAuthenticated()) {
            long begin = System.nanoTime();

            // Deja auth -> on continue la chaine
            chain.doFilter(request, response);

            long diff = System.nanoTime() - begin;
            response.addHeader("iamstats-app-response-time", "D=" + (diff / 1000));
        }
        // Pas encore authentifié
        else {
            boolean authSucess;
            String redirUrlForLog;
            // On est appelé depuis le devlogin
            if (url.endsWith(SPRING_SECURITY_URL)) {
                String username = request.getParameter(DEV_LOGIN_USERNAME);
                String rolesCommas = request.getParameter(DEV_LOGIN_ROLES);
                if (StringUtils.isNotBlank(username) && StringUtils.isNotBlank(rolesCommas)) {
                    String application = "dl-app";
                    String rolesStr = "";
                    for (String part : rolesCommas.split(",")) {
                        if (rolesStr.length() > 0) {
                            rolesStr += "|";
                        }
                        rolesStr += "cn=" + application + "-" + part + ",dc=etat-de-vaud,dc=ch";
                    }

                    LOGGER.info("Le devlogin est activé. POST sur /" + SPRING_SECURITY_URL + " avec " + username + " / " + rolesStr);

                    String firstName = request.getParameter(DEV_LOGIN_FIRST);
                    String lastName = request.getParameter(DEV_LOGIN_LAST);
                    String email = request.getParameter(DEV_LOGIN_EMAIL);
                    Objects.requireNonNull(firstName, "Pas de first name");
                    Objects.requireNonNull(lastName, "Pas de last name");

                    String authLevelStr = request.getParameter(IamConstants.IAM_HEADER_AUTHLEVEL);
                    Integer authLevel;
                    if (authLevelStr != null) {
                        authLevel = Integer.parseInt(authLevelStr);
                    } else {
                        authLevel = null;
                    }
                    String lastLoginTime = request.getParameter(IamConstants.IAM_HEADER_LASTLOGIN);

                    AuthenticateDTO dto = new AuthenticateDTO(
                            username, username, application, rolesStr, firstName,
                            lastName, email, authLevel, lastLoginTime);
                    authenticate(dto);

                    String query = (String) request.getSession().getAttribute(IAM_AUTH_SAVED_PATH);
                    if (query != null) {
                        LOGGER.info("Le devlogin est successful. Redirection -> " + query);
                        response.sendRedirect(query);
                        redirUrlForLog = query;
                    } else {
                        LOGGER.info("Le devlogin est successful. Redirection -> " + successUrl);
                        relativeRedirect(wrapper, response, successUrl);
                        redirUrlForLog = successUrl;
                    }
                    authSucess = true;
                } else {
                    // Pas de username ni roles
                    LOGGER.warn("Le devlogin est activé, mais pas de username ni roles. Url='" + request.getRequestURI() + "' Redirection -> " + deniedUrl);
                    relativeRedirect(wrapper, response, deniedUrl);
                    redirUrlForLog = deniedUrl;
                    authSucess = false;
                }
            } else {
                // Pas authentifé et pas le bon URL d'appel
                relativeRedirect(wrapper, response, deniedUrl);
                redirUrlForLog = deniedUrl;
                authSucess = false;
            }

            IamAuthentication auth = (IamAuthentication) SecurityContextHolder.getContext().getAuthentication();
            if (authSucess) {
                RequestHelper.logRequestInfos(wrapper, response, "AUTH-OK Redirect to '" + redirUrlForLog + "'", getUsernameForLog(wrapper), auth != null ? auth.getAllRoles() : null);
            } else {
                RequestHelper.logRequestInfos(wrapper, response, "AUTH-KO Redirect to '" + redirUrlForLog + "'", getUsernameForLog(wrapper), auth != null ? auth.getAllRoles() : null);
            }
        }
    }

    public void setSuccessUrl(String successUrl) {
        this.successUrl = successUrl;
    }

    public void setDeniedUrl(String deniedUrl) {
        this.deniedUrl = deniedUrl;
    }
}
