package ch.vd.shared.iam.web.filter.auth;

import ch.vd.shared.iam.core.filter.auth.IamAuthentication;
import ch.vd.shared.iam.web.common.RequestHelper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.Assert;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Objects;

/**
 * Prend les infos d'authentification de IAM dans la request et les met dans le contexte de sécurité Si la request de contient aucune info
 * de sécurité, redirige sur l'url <i>failureUrl</i>
 *
 * @author jec
 */
public class IamAuthenticationSpringFilter extends AbstractSharedIamAuthenticationFilter implements InitializingBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(IamAuthenticationSpringFilter.class);

    private String failureUrl;

    public IamAuthenticationSpringFilter() {
        toString(); // For BP
    }

    @Override
    public void afterPropertiesSet() throws ServletException {
        super.afterPropertiesSet();

        Assert.isTrue(StringUtils.isNotBlank(failureUrl), "'failureUrl' must be set");
    }

    @Override
    public void doFilter(ServletRequest r, ServletResponse resp, FilterChain chain) throws IOException, ServletException {

        HttpServletRequest req = (HttpServletRequest) r;
        HttpServletResponse response = (HttpServletResponse) resp;

        IamRequestWrapper wrapper = new IamRequestWrapper(req);

        // Log les headers de la request
        // Le log n'est effectué que si iam.http.headers est mis sur INFO
        RequestHelper.dumpRequestForDebug(req);

        // Vérifie que le user loggé et celui qui vient de IAM est le même
        invalidateHttpSessionIfNotsameBetweenIamAuthenticationAndRequest(wrapper, response);

        if (requiresAuthentication(wrapper)) {
            IamAuthentication auth = (IamAuthentication) SecurityContextHolder.getContext().getAuthentication();
            RequestHelper.logRequestInfos(wrapper, response, RequestHelper.normalize("NEED", 8), getUsernameForLog(wrapper), auth != null ? auth.getAllRoles() : null);
            boolean authSuccess = doAuthentication(wrapper, response);
            if (!authSuccess) {
                RequestHelper.logRequestInfos(wrapper, response, RequestHelper.normalize("AUTH-KO", 8), getUsernameForLog(wrapper), auth != null ? auth.getAllRoles() : null);
                return;
            }
            RequestHelper.logRequestInfos(wrapper, response, RequestHelper.normalize("AUTH-OK", 8), getUsernameForLog(wrapper), auth != null ? auth.getAllRoles() : null);
        }


        long begin = System.nanoTime();

        // Deja auth -> on continue la chaîne
        chain.doFilter(req, response);

        long diff = System.nanoTime() - begin;
        response.addHeader("iamstats-app-response-time", "D=" + (diff / 1000));
    }

    private void invalidateHttpSessionIfNotsameBetweenIamAuthenticationAndRequest(IamRequestWrapper wrapper, HttpServletResponse response) {
        IamAuthentication auth = (IamAuthentication) SecurityContextHolder.getContext().getAuthentication();
        if (auth != null && auth.isAuthenticated()) {
            // On a le meme user?
            String reqUser = getUsedPrincipal(wrapper);

            String authUser = auth.getName();
            Integer authLevel = auth.getAuthLevel();

            boolean sameUser = Objects.equals(authUser, reqUser);
            boolean sameAuthLevel = Objects.equals(authLevel, wrapper.getAuthLevel());
            if (!sameUser || !sameAuthLevel) {
                wrapper.getSession().invalidate();
            }
        }
    }

    private boolean requiresAuthentication(IamRequestWrapper request) {

        boolean need;

        Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        if (auth == null) {
            need = true;
        } else if (!auth.isAuthenticated()) {
            need = true;
        } else {
            // On a le meme user?
            String username = getUsedPrincipal(request);
            String principal = auth.getName();
            if (principal == null) {
                need = true;
            } else {
                need = username != null && !principal.equals(username);
            }
        }
        return need;
    }

    private boolean doAuthentication(IamRequestWrapper wrapper, HttpServletResponse response) throws IOException {
        SecurityContextHolder.clearContext();

        String username = wrapper.getUserName();
        String application = wrapper.getApplication();
        String rolesStr = wrapper.getRoles();

        if (StringUtils.isBlank(username) || StringUtils.isBlank(application)) {
            // Headers mandatory manquants -> redirige sur la page de failure
            LOGGER.error("Il manque des headers dans la request -> redirige sur '{}' (username={}, application={})", failureUrl, username, application);

            // ... on sauve la request
            String requestUri = wrapper.getRequestURI();
            wrapper.getSession().setAttribute(IAM_AUTH_SAVED_PATH, requestUri);

            // ... redirection
            relativeRedirect(wrapper, response, failureUrl);
            return false;
        }

        Integer authLevel = wrapper.getAuthLevel();
        String lastLoginStr = wrapper.getLastLogin();
        String email = wrapper.getEmail();
        String firstName = wrapper.getFirstName();
        String lastName = wrapper.getLastName();
        String usedPrincipal = getUsedPrincipal(wrapper);

        AuthenticateDTO authenticateDTO = new AuthenticateDTO(usedPrincipal, username, application, rolesStr, firstName, lastName, email, authLevel, lastLoginStr);
        authenticate(authenticateDTO);
        return true;
    }

    public void setFailureUrl(String failureUrl) {
        this.failureUrl = failureUrl;
    }
}
