package ch.vd.shared.iam.web.filter.autorization;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.authentication.AuthenticationCredentialsNotFoundException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.FilterInvocation;
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource;

import javax.servlet.*;
import java.io.IOException;
import java.util.Collection;
import java.util.Objects;

public class ByRoleUrlAutorizationSpringFilter implements Filter, InitializingBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(ByRoleUrlAutorizationSpringFilter.class);
    private static final int ACCESS_GRANTED = +1;
    private static final int ACCESS_DENIED = -1;
    public static final String IS_AUTHENTICATED_FULLY = "IS_AUTHENTICATED_FULLY";
    public static final String IS_ANONYMOUS = "IS_ANONYMOUS";
    public static final String IS_DENIED = "IS_DENIED";

    private FilterInvocationSecurityMetadataSource securityMetadataSource;

    /**
     * Not used (we rely on IoC container lifecycle services instead)
     *
     * @param arg0 ignored
     *
     * @throws ServletException never thrown
     */
    @Override
    public void init(FilterConfig arg0) throws ServletException {
    }

    /**
     * Not used (we rely on IoC container lifecycle services instead)
     */
    @Override
    public void destroy() {
    }

    @Override
    public void afterPropertiesSet() {
        Objects.requireNonNull(securityMetadataSource, "An SecurityMetadataSource is required");
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        FilterInvocation fi = new FilterInvocation(request, response, chain);

        checkAutorization(fi);

        chain.doFilter(request, response);
    }

    protected void checkAutorization(FilterInvocation fi) {
        Objects.requireNonNull(fi, "Object was null");
        boolean debug = LOGGER.isDebugEnabled();

        Collection<ConfigAttribute> attributes = securityMetadataSource.getAttributes(fi);

        if (attributes == null) {
            throw new IllegalArgumentException("Secure object invocation " + fi
                    + " was denied as public invocations are not allowed via this interceptor");
        }

        if (debug) {
            LOGGER.debug("Secure object: " + fi + "; Attributes: " + attributes);
        }

        Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        if (auth == null) {
            credentialsNotFound("An Authentication object was not found in the SecurityContext", fi, attributes);
        }

        // Attempt authorization
        authorize(fi, auth, attributes);

        if (debug) {
            LOGGER.debug("Authorization successful");
        }
    }

    private void authorize(FilterInvocation fi, Authentication auth, Collection<ConfigAttribute> attributes) {
        Objects.requireNonNull(auth, "auth cannot be null");

        {
            for (ConfigAttribute attribute : attributes) {
                if (IS_DENIED.equals(attribute.getAttribute())) {
                    throw new AccessDeniedException("Access is denied");
                }
                if (IS_AUTHENTICATED_FULLY.equals(attribute.getAttribute())) {
                    if (auth.isAuthenticated()) {
                        return; // GRANTED
                    }
                }
                if (IS_ANONYMOUS.equals(attribute.getAttribute())) {
                    return; // GRANTED
                }
            }
        }

        int result = authorizeOnRoles(fi, auth, attributes);
        if (result == ACCESS_DENIED) {
            throw new AccessDeniedException("Access is denied");
        }
    }

    private int authorizeOnRoles(FilterInvocation fi, Authentication auth, Collection<ConfigAttribute> attributes) {
        int result = ACCESS_DENIED;
        Collection<? extends GrantedAuthority> authorities = auth.getAuthorities();

        for (ConfigAttribute attribute : attributes) {
            result = ACCESS_DENIED;

            // Attempt to find a matching granted authority
            for (GrantedAuthority authority : authorities) {
                if (authority.getAuthority() != null
                        && attribute.getAttribute() != null
                        && attribute.getAttribute().contains(authority.getAuthority())) {
                    return ACCESS_GRANTED;
                }
            }
        }

        return result;
    }

    private void credentialsNotFound(String reason, Object secureObject, Collection<ConfigAttribute> configAttribs) {
        throw new AuthenticationCredentialsNotFoundException(reason);
    }

    public void setSecurityMetadataSource(FilterInvocationSecurityMetadataSource newSource) {
        securityMetadataSource = newSource;
    }
}
