package ch.vd.shared.iam.web.filter.remote;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.*;
import java.io.IOException;
import java.util.regex.Pattern;

public class RemoteHostPureServletFilter implements Filter {

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private Pattern[] allowed;
    private Pattern[] denied;

    /**
     * @param request  The servlet request we are processing
     * @param response The servlet response we are creating
     * @param chain    The filter chain we are processing
     *
     * @throws java.io.IOException            if an input/output error occurs
     * @throws javax.servlet.ServletException if a servlet error occurs
     */
    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        String clientAddr = request.getRemoteAddr();

        if (isInvalidAddress(clientAddr)) {
            logger.debug("Denied access to ip {}", clientAddr);
            RemoteHostHelper.handleInvalidAccess(request, response, clientAddr);
            return;
        }

        logger.debug("Access granted to ip {}", clientAddr);
        chain.doFilter(request, response);
    }

    boolean isInvalidAddress(String address) {
        return (denied.length > 0 && RemoteHostHelper.hasMatch(address, denied))
                || (allowed.length > 0 && !RemoteHostHelper.hasMatch(address, allowed));
    }

    /**
     * Destroy method for this filter
     */
    @Override
    public void destroy() {
        allowed = null;
        denied = null;
    }

    /**
     * Init method for this filter.
     * Reads config from config file if parameter "config" is set.
     */
    @Override
    public void init(FilterConfig filterConfig) {
        String allowedIps = filterConfig.getInitParameter("allowed");
        logger.debug("allowed='{}'", allowedIps);
        String denyStr = filterConfig.getInitParameter("denied");
        logger.debug("denied='{}'", denyStr);

        allowed = RemoteHostHelper.parseParam(allowedIps, true);
        denied = RemoteHostHelper.parseParam(denyStr, true);

        if (allowed.length == 0 && denied.length == 0) {
            throw new IllegalArgumentException("Au moins une des deux valeurs [denied/allowed] doit être configurée");
        }
    }
}
