package ch.vd.shared.iam.web.utils;

import ch.vd.shared.iam.core.filter.auth.IamAuthority;
import ch.vd.shared.iam.web.common.RequestHelper;
import ch.vd.shared.iam.web.filter.auth.IamRequestWrapper;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;

import javax.servlet.http.HttpServletRequest;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

public abstract class IamWebUtils {

    private static final String PIPE_SEPARATOR = "\\x7C";
    public static final String X_FORWARDED_HOST = "X-Forwarded-Host";

    public static List<String> parseRoles(IamRequestWrapper wrapper) {
        return parseRoles(wrapper.getApplication(), wrapper.getRoles());
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    public static List<String> parseRoles(String application, String rolesStr) {
        Objects.requireNonNull(application, "'application' cannot be null");

        List<String> result = new ArrayList<>();

        // Example of roles obtained from IAM:
        // Application: finances-demo
        // Role: cn=finances-demo-comptable,dc=etat-de-vaud,dc=ch
        //  -> Role: ROLE_COMPTABLE
        // cn=finances-demo-secretaire_d_office,dc=etat-de-vaud,dc=ch
        // cn=finances-demo-juriste,dc=etat-de-vaud,dc=ch
        // cn=finances-demo-prepose,dc=etat-de-vaud,dc=ch

        // Split roles with separator "pipe":
        String[] roles = rolesStr == null ? new String[]{} : rolesStr.split(PIPE_SEPARATOR);

        for (String role : roles) {
            if (isLDAPRoles(role)) {
                // Don't take "cn=" nor ",dc=.." by splitting:
                String[] roleItems = role.split("[=,]");
                if (roleItems.length > 1) {
                    String appRole = roleItems[1];
                    // Take only those roles that are relevant to this application:
                    handleApplicationRoles(application, result, appRole);
                } else {
                    RequestHelper.LOGGER.error("parseRoles(): Unexpected role={}", role);
                }
            } else {
                // Take only those roles that are relevant to this application:
                handleApplicationRoles(application, result, role);
            }
        }
        return result;
    }

    private static void handleApplicationRoles(String application, List<String> result, String role) {
        if (role.startsWith(application)) {
            Assert.isTrue(role.length() > application.length(), "Erreur dans les headers. application='" + application + "' role='" + role + "'");
            if (role.charAt(application.length()) == '-') {
                // Extract business role by removing application name:
                String subRole = role.substring(1 + application.length());
                result.add("ROLE_" + subRole.toUpperCase());
            } else {
                RequestHelper.LOGGER.debug("parseRoles(): Ignoring role={}", role);
            }
        } else {
            RequestHelper.LOGGER.debug("parseRoles(): Ignoring role={}", role);
        }
    }

    /**
     * Convertit une liste de roles sous forme de String en GrantedAuthority
     */
    public static List<IamAuthority> convertToAuthority(List<String> rolesStr) {
        List<IamAuthority> result = new ArrayList<>();
        if (rolesStr != null) {
            for (String role : rolesStr) {
                result.add(new IamAuthority(role));
            }
        }
        return result;
    }

    public static String calculateRedirect(HttpServletRequest request, String inputUrl) {

        String contextUrl;
        {
            String contextPath = request.getContextPath();
            if (!inputUrl.startsWith("/")) {
                // Relatif au context
                contextUrl = contextPath + "/" + inputUrl;
            } else {
                contextUrl = inputUrl;
            }
        }

        // Get all forwarded hosts (they are separated with comma ","):
        String host = request.getHeader(X_FORWARDED_HOST);

        if (StringUtils.isNotBlank(host)) {
            // Only get the 1st host, which is the RP IAM:
            int commaIndex = host.indexOf(",", 3); // Looking for first comma...
            if (commaIndex > 0) {
                host = host.substring(0, commaIndex); // Take first host = RP IAM
            }
        }

        String redir;
        if (host != null && !host.startsWith("localhost")) {
            redir = "https://" + host + contextUrl;
        } else { // on est sans IAM
            redir = contextUrl;
        }
        return redir;
    }

    public static Date parseIsoTime(String str) {
        Date date;
        if (str != null) {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
            try {
                date = sdf.parse(str);
            } catch (ParseException e) {
                throw new RuntimeException(e);
            }
        } else {
            date = null;
        }
        return date;
    }

    private static boolean isLDAPRoles(String allRoles) {
        return allRoles.contains("=");
    }
}

