package dotty.tools
package dotc
package typer

import core._
import ast.{Trees, TreeTypeMap, untpd, tpd, DesugarEnums}
import util.Spans._
import util.Stats.{record, monitored}
import printing.{Showable, Printer}
import printing.Texts._
import Contexts._
import Types._
import Flags._
import Mode.ImplicitsEnabled
import NameOps._
import NameKinds.{LazyImplicitName, EvidenceParamName}
import Symbols._
import Denotations._
import Types._
import Decorators._
import Names._
import StdNames._
import Constants._
import ProtoTypes._
import ErrorReporting._
import Inferencing.{fullyDefinedType, isFullyDefined}
import Trees._
import transform.SymUtils._
import transform.TypeUtils._
import Hashable._
import util.{SourceFile, NoSource}
import config.{Config, Feature}
import Feature.migrateTo3
import config.Printers.{implicits, implicitsDetailed}
import collection.mutable
import reporting._
import annotation.tailrec

import scala.annotation.internal.sharable
import scala.annotation.threadUnsafe

/** Implicit resolution */
object Implicits:
  import tpd._

  /** An implicit definition `implicitRef` that is visible under a different name, `alias`.
   *  Gets generated if an implicit ref is imported via a renaming import.
   */
  class RenamedImplicitRef(val underlyingRef: TermRef, val alias: TermName) extends ImplicitRef {
    def implicitName(using Context): TermName = alias
  }

  /** An eligible implicit candidate, consisting of an implicit reference and a nesting level */
  case class Candidate(implicitRef: ImplicitRef, kind: Candidate.Kind, level: Int) {
    def ref: TermRef = implicitRef.underlyingRef

    def isExtension = (kind & Candidate.Extension) != 0
    def isConversion = (kind & Candidate.Conversion) != 0
  }
  object Candidate {
    type Kind = Int
    final val None = 0
    final val Value = 1
    final val Conversion = 2
    final val Extension = 4
  }

  /** If `expected` is a selection prototype, does `tp` have an extension
   *  method with the selecting name? False otherwise.
   */
  def hasExtMethod(tp: Type, expected: Type)(using Context) = expected match
    case selProto @ SelectionProto(_: TermName, _, _, _) =>
      tp.memberBasedOnFlags(selProto.extensionName, required = ExtensionMethod).exists
    case _ => false

  def strictEquality(using Context): Boolean =
    ctx.mode.is(Mode.StrictEquality) || Feature.enabled(nme.strictEquality)

  /** A common base class of contextual implicits and of-type implicits which
   *  represents a set of references to implicit definitions.
   */
  abstract class ImplicitRefs(initctx: Context) {
    val irefCtx =
      if (initctx == NoContext) initctx else initctx.retractMode(Mode.ImplicitsEnabled)
    protected given Context = irefCtx

    /** The nesting level of this context. Non-zero only in ContextialImplicits */
    def level: Int = 0

    /** The implicit references */
    def refs: List[ImplicitRef]

    /** If comes from an implicit scope of a type, the companion objects making
     *  up that implicit scope, otherwise the empty set.
     */
    def companionRefs: TermRefSet = TermRefSet.empty

    private var mySingletonClass: ClassSymbol = null

    /** Widen type so that it is neither a singleton type nor a type that inherits from scala.Singleton. */
    private def widenSingleton(tp: Type)(using Context): Type = {
      if (mySingletonClass == null) mySingletonClass = defn.SingletonClass
      val wtp = tp.widenSingleton
      if (wtp.derivesFrom(mySingletonClass)) defn.AnyType else wtp
    }

    protected def isAccessible(ref: TermRef)(using Context): Boolean

    /** Return those references in `refs` that are compatible with type `pt`. */
    protected def filterMatching(pt: Type)(using Context): List[Candidate] = {
      record("filterMatching")

      val considerExtension = pt match
        case ViewProto(_, _: SelectionProto) => true
        case _ => false

      def candidateKind(ref: TermRef)(using Context): Candidate.Kind = { /*trace(i"candidateKind $ref $pt")*/

        def viewCandidateKind(tpw: Type, argType: Type, resType: Type): Candidate.Kind = {

          def methodCandidateKind(mt: MethodType, approx: Boolean) =
            if (mt.isImplicitMethod)
              viewCandidateKind(normalize(mt, pt), argType, resType)
            else if (mt.paramInfos.lengthCompare(1) == 0 && {
                  var formal = widenSingleton(mt.paramInfos.head)
                  if (approx) formal = wildApprox(formal)
                  explore(argType relaxed_<:< formal.widenExpr)
                })
              Candidate.Conversion
            else
              Candidate.None

          tpw match {
            case mt: MethodType =>
              methodCandidateKind(mt, approx = false)
            case poly: PolyType =>
              // We do not need to call ProtoTypes#constrained on `poly` because
              // `candidateKind` is always called with mode TypevarsMissContext enabled.
              poly.resultType match {
                case mt: MethodType =>
                  methodCandidateKind(mt, approx = true)
                case rtp =>
                  viewCandidateKind(wildApprox(rtp), argType, resType)
              }
            case tpw: TermRef => // can't discard overloaded refs
              Candidate.Conversion
              | (if considerExtension then Candidate.Extension else Candidate.None)
            case tpw =>
              // Only direct instances of Function1 and direct or indirect instances of <:< are eligible as views.
              // However, Predef.$conforms is not eligible, because it is a no-op.
              //
              // In principle, it would be cleanest if only implicit methods qualified
              // as implicit conversions. We could achieve that by having standard conversions like
              // this in Predef:
              //
              //    implicit def convertIfConforms[A, B](x: A)(implicit ev: A <:< B): B = ev(a)
              //    implicit def convertIfConverter[A, B](x: A)(implicit ev: Conversion[A, B]): B = ev(a)
              //
              // (Once `<:<` inherits from `Conversion` we only need the 2nd one.)
              // But clauses like this currently slow down implicit search a lot, because
              // they are eligible for all pairs of types, and therefore are tried too often.
              // We emulate instead these conversions directly in the search.
              // The reason for leaving out `Predef_conforms` is that we know it adds
              // nothing since it only relates subtype with supertype.
              //
              // We keep the old behavior under -source 3.0-migration.
              val isFunctionInS2 =
                migrateTo3
                && tpw.derivesFrom(defn.FunctionClass(1))
                && ref.symbol != defn.Predef_conforms
              val isImplicitConversion = tpw.derivesFrom(defn.ConversionClass)
              // An implementation of <:< counts as a view
              val isConforms = tpw.derivesFrom(defn.SubTypeClass)
              val conversionKind =
                if (isFunctionInS2 || isImplicitConversion || isConforms) Candidate.Conversion
                else Candidate.None
              val extensionKind =
                if considerExtension && hasExtMethod(tpw, resType) then Candidate.Extension
                else Candidate.None
              conversionKind | extensionKind
          }
        }

        def valueTypeCandidateKind(tpw: Type): Candidate.Kind = tpw.stripPoly match {
          case tpw: MethodType =>
            if (tpw.isImplicitMethod) Candidate.Value else Candidate.None
          case _ =>
            Candidate.Value
        }

        /** Widen singleton arguments of implicit conversions to their underlying type.
         *  This is necessary so that they can be found eligible for the argument type.
         *  Note that we always take the underlying type of a singleton type as the argument
         *  type, so that we get a reasonable implicit cache hit ratio.
         */
        def adjustSingletonArg(tp: Type): Type = tp.widenSingleton match
          case tp: PolyType =>
            val res = adjustSingletonArg(tp.resType)
            if res eq tp.resType then tp else tp.derivedLambdaType(resType = res)
          case tp: MethodType =>
            tp.derivedLambdaType(paramInfos = tp.paramInfos.mapConserve(widenSingleton))
          case _ =>
            tp.baseType(defn.ConversionClass) match
              case app @ AppliedType(tycon, from :: rest) =>
                val wideFrom = from.widenSingleton
                if wideFrom ne from then app.derivedAppliedType(tycon, wideFrom :: rest)
                else tp
              case _ => tp

        var ckind =
          if !isAccessible(ref) then
            Candidate.None
          else pt match {
            case pt: ViewProto =>
              viewCandidateKind(ref.widen, pt.argType, pt.resType)
            case _: ValueTypeOrProto =>
              if (defn.isFunctionType(pt)) Candidate.Value
              else valueTypeCandidateKind(ref.widen)
            case _ =>
              Candidate.Value
          }

        if (ckind == Candidate.None)
          record("discarded eligible")
        else {
          val ptNorm = normalize(pt, pt) // `pt` could be implicit function types, check i2749
          val refAdjusted =
            if (pt.isInstanceOf[ViewProto]) adjustSingletonArg(ref)
            else ref
          val refNorm = normalize(refAdjusted, pt)
          if (!NoViewsAllowed.isCompatible(refNorm, ptNorm))
            ckind = Candidate.None
        }
        ckind
      }


      if refs.isEmpty && (!considerExtension || companionRefs.isEmpty) then
        Nil
      else
        val nestedCtx = ctx.fresh.addMode(Mode.TypevarsMissContext)
        val candidates = new mutable.ListBuffer[Candidate]
        def tryCandidate(extensionOnly: Boolean)(ref: ImplicitRef) =
          var ckind = explore(candidateKind(ref.underlyingRef))(using nestedCtx)
          if extensionOnly then ckind &= Candidate.Extension
          if ckind != Candidate.None then
            candidates += Candidate(ref, ckind, level)

        if considerExtension then
          companionRefs.foreach(tryCandidate(extensionOnly = true))
        if refs.nonEmpty then
          refs.foreach(tryCandidate(extensionOnly = false))
        candidates.toList
    }
  }

  /** The implicit references coming from the implicit scope of a type.
   *  @param tp              the type determining the implicit scope
   *  @param companionRefs   the companion objects in the implicit scope.
   */
  class OfTypeImplicits(tp: Type, override val companionRefs: TermRefSet)(initctx: Context) extends ImplicitRefs(initctx) {
    assert(initctx.typer != null)
    implicits.println(i"implicit scope of type $tp = ${companionRefs.showAsList}%, %")
    @threadUnsafe lazy val refs: List[ImplicitRef] = {
      val buf = new mutable.ListBuffer[TermRef]
      for (companion <- companionRefs) buf ++= companion.implicitMembers
      buf.toList
    }

    /** The candidates that are eligible for expected type `tp` */
    @threadUnsafe lazy val eligible: List[Candidate] =
      trace(i"eligible($tp), companions = ${companionRefs.showAsList}%, %", implicitsDetailed, show = true) {
        if (refs.nonEmpty && monitored) record(s"check eligible refs in tpe", refs.length)
        filterMatching(tp)
      }

    override def isAccessible(ref: TermRef)(using Context): Boolean =
      ref.symbol.exists && !ref.symbol.is(Private)

    override def toString: String =
      i"OfTypeImplicits($tp), companions = ${companionRefs.showAsList}%, %; refs = $refs%, %."
  }

  /** The implicit references coming from the context.
   *  @param refs      the implicit references made visible by the current context.
   *                   Note: The name of the reference might be different from the name of its symbol.
   *                   In the case of a renaming import a => b, the name of the reference is the renamed
   *                   name, b, whereas the name of the symbol is the original name, a.
   *  @param outerCtx  the next outer context that makes visible further implicits
   */
  class ContextualImplicits(val refs: List[ImplicitRef], val outerImplicits: ContextualImplicits)(initctx: Context) extends ImplicitRefs(initctx) {
    private val eligibleCache = new java.util.IdentityHashMap[Type, List[Candidate]]

    /** The level increases if current context has a different owner or scope than
     *  the context of the next-outer ImplicitRefs. This is however disabled under
     *  Scala2 mode, since we do not want to change the implicit disambiguation then.
     */
    override val level: Int =
      if outerImplicits == null then 1
      else if migrateTo3(using irefCtx)
              || (irefCtx.owner eq outerImplicits.irefCtx.owner)
                 && (irefCtx.scope eq outerImplicits.irefCtx.scope)
                 && !refs.head.implicitName.is(LazyImplicitName)
      then outerImplicits.level
      else outerImplicits.level + 1

    /** Is this the outermost implicits? This is the case if it either the implicits
     *  of NoContext, or the last one before it.
     */
    private def isOuterMost = {
      val finalImplicits = NoContext.implicits
      (this eq finalImplicits) || (outerImplicits eq finalImplicits)
    }

    /** The implicit references that are eligible for type `tp`. */
    def eligible(tp: Type): List[Candidate] =
      if (tp.hash == NotCached) computeEligible(tp)
      else {
        val eligibles = eligibleCache.get(tp)
        if (eligibles != null) {
          def elided(ci: ContextualImplicits): Int = {
            val n = ci.refs.length
            if (ci.isOuterMost) n
            else n + elided(ci.outerImplicits)
          }
          if (monitored) record(s"elided eligible refs", elided(this))
          eligibles
        }
        else if (irefCtx eq NoContext) Nil
        else {
          val result = computeEligible(tp)
          eligibleCache.put(tp, result)
          result
        }
      }

    private def computeEligible(tp: Type): List[Candidate] = /*>|>*/ trace(i"computeEligible $tp in $refs%, %", implicitsDetailed) /*<|<*/ {
      if (monitored) record(s"check eligible refs in irefCtx", refs.length)
      val ownEligible = filterMatching(tp)
      if (isOuterMost) ownEligible
      else if ownEligible.isEmpty then outerImplicits.eligible(tp)
      else
        val outerEligible = outerImplicits.eligible(tp)
        if outerEligible.isEmpty then ownEligible
        else
          val shadowed = ownEligible.map(_.ref.implicitName).toSet
          ownEligible ::: outerEligible.filterConserve(cand => !shadowed.contains(cand.ref.implicitName))
    }

    override def isAccessible(ref: TermRef)(using Context): Boolean =
      ref.symbol.isAccessibleFrom(ref.prefix)

    override def toString: String = {
      val own = i"(implicits: $refs%, %)"
      if (isOuterMost) own else own + "\n " + outerImplicits
    }

    /** This context, or a copy, ensuring root import from symbol `root`
     *  is not present in outer implicits.
     */
    def exclude(root: Symbol): ContextualImplicits =
      if (this == NoContext.implicits) this
      else {
        val outerExcluded = outerImplicits exclude root
        if (irefCtx.importInfo.site.termSymbol == root) outerExcluded
        else if (outerExcluded eq outerImplicits) this
        else new ContextualImplicits(refs, outerExcluded)(irefCtx)
      }
  }

  /** The result of an implicit search */
  sealed abstract class SearchResult extends Showable {
    def tree: Tree
    def toText(printer: Printer): Text = printer.toText(this)
    def recoverWith(other: SearchFailure => SearchResult): SearchResult = this match {
      case _: SearchSuccess => this
      case fail: SearchFailure => other(fail)
    }
    def isSuccess: Boolean = isInstanceOf[SearchSuccess]
  }

  /** A successful search
   *  @param tree   The typed tree that needs to be inserted
   *  @param ref    The implicit reference that succeeded
   *  @param level  The level where the reference was found
   *  @param tstate The typer state to be committed if this alternative is chosen
   */
  case class SearchSuccess(tree: Tree, ref: TermRef, level: Int)(val tstate: TyperState, val gstate: GadtConstraint) extends SearchResult with Showable

  /** A failed search */
  case class SearchFailure(tree: Tree) extends SearchResult {
    final def isAmbiguous: Boolean = tree.tpe.isInstanceOf[AmbiguousImplicits]
    final def reason: SearchFailureType = tree.tpe.asInstanceOf[SearchFailureType]
  }

  object SearchFailure {
    def apply(tpe: SearchFailureType)(using Context): SearchFailure = {
      val id = tpe match
        case tpe: AmbiguousImplicits =>
          untpd.SearchFailureIdent(nme.AMBIGUOUS, s"/* ambiguous: ${tpe.explanation} */")
        case _ =>
          untpd.SearchFailureIdent(nme.MISSING, "/* missing */")
      SearchFailure(id.withTypeUnchecked(tpe))
    }
  }

  abstract class SearchFailureType extends ErrorType {
    def expectedType: Type
    def argument: Tree

    /** A "massaging" function for displayed types to give better info in error diagnostics */
    def clarify(tp: Type)(using Context): Type = tp

    final protected def qualify(using Context): String = expectedType match {
      case SelectionProto(name, mproto, _, _) if !argument.isEmpty =>
        em"provide an extension method `$name` on ${argument.tpe}"
      case NoType =>
        if (argument.isEmpty) em"match expected type"
        else em"convert from ${argument.tpe} to expected type"
      case _ =>
        if (argument.isEmpty) em"match type ${clarify(expectedType)}"
        else em"convert from ${argument.tpe} to ${clarify(expectedType)}"
    }

    /** An explanation of the cause of the failure as a string */
    def explanation(using Context): String

    def msg(using Context): Message = explanation

    /** If search was for an implicit conversion, a note describing the failure
     *  in more detail - this is either empty or starts with a '\n'
     */
    def whyNoConversion(using Context): String = ""
  }

  class NoMatchingImplicits(val expectedType: Type, val argument: Tree, constraint: Constraint = OrderingConstraint.empty)
  extends SearchFailureType {

    /** Replace all type parameters in constraint by their bounds, to make it clearer
     *  what was expected
     */
    override def clarify(tp: Type)(using Context): Type =
      val ctx1 = ctx.fresh.setExploreTyperState()
      ctx1.typerState.constraint = constraint
      inContext(ctx1) {
        val map = new TypeMap {
          def apply(t: Type): Type = t match {
            case t: TypeParamRef =>
              constraint.entry(t) match {
                case NoType => t
                case bounds: TypeBounds => TypeComparer.fullBounds(t)
                case t1 => t1
              }
            case t: TypeVar =>
              t.instanceOpt.orElse(apply(t.origin))
            case _ =>
              mapOver(t)
          }
        }
        map(tp)
      }

    def explanation(using Context): String =
      em"no implicit values were found that $qualify"
    override def toString = s"NoMatchingImplicits($expectedType, $argument)"
  }

  @sharable object NoMatchingImplicits extends NoMatchingImplicits(NoType, EmptyTree, OrderingConstraint.empty)

  @sharable val NoMatchingImplicitsFailure: SearchFailure =
    SearchFailure(NoMatchingImplicits)(using NoContext)

  /** An ambiguous implicits failure */
  class AmbiguousImplicits(val alt1: SearchSuccess, val alt2: SearchSuccess, val expectedType: Type, val argument: Tree) extends SearchFailureType {
    def explanation(using Context): String =
      em"both ${err.refStr(alt1.ref)} and ${err.refStr(alt2.ref)} $qualify"
    override def whyNoConversion(using Context): String = {
      val what = if (expectedType.isInstanceOf[SelectionProto]) "extension methods" else "conversions"
      i"""
         |Note that implicit $what cannot be applied because they are ambiguous;
         |$explanation"""
    }
  }

  class MismatchedImplicit(ref: TermRef,
                           val expectedType: Type,
                           val argument: Tree) extends SearchFailureType {
    def explanation(using Context): String =
      em"${err.refStr(ref)} does not $qualify"
  }

  class DivergingImplicit(ref: TermRef,
                          val expectedType: Type,
                          val argument: Tree) extends SearchFailureType {
    def explanation(using Context): String =
      em"${err.refStr(ref)} produces a diverging implicit search when trying to $qualify"
  }

  class FailedExtension(extApp: Tree, val expectedType: Type) extends SearchFailureType:
    def argument = EmptyTree
    def explanation(using Context) = em"$extApp does not $qualify"
end Implicits

import Implicits._

/** Info relating to implicits that is kept for one run */
trait ImplicitRunInfo:
  self: Run =>

  private val implicitScopeCache = mutable.AnyRefMap[Type, OfTypeImplicits]()

  private def isExcluded(sym: Symbol) =
    if migrateTo3 then false else sym.is(Package) || sym.isPackageObject

  /** Is `sym` an anchor type for which givens may exist? Anchor types are classes,
    *  opaque type aliases, match aliases and abstract types, but not type parameters
    *  or package objects.
    */
  private def isAnchor(sym: Symbol) =
    sym.isClass && !isExcluded(sym)
    || sym.isOpaqueAlias
    || sym.is(Deferred, butNot = Param)
    || sym.info.isInstanceOf[MatchAlias]

  private def computeIScope(rootTp: Type): OfTypeImplicits =

    object collectParts extends TypeTraverser:

      private var provisional: Boolean = _
      private var parts: mutable.LinkedHashSet[Type] = _
      private val partSeen = TypeHashSet()

      def traverse(t: Type) =
        if partSeen.contains(t) then ()
        else if implicitScopeCache.contains(t) then parts += t
        else
          partSeen.addEntry(t)
          t.dealias match
            case t: TypeRef =>
              if isAnchor(t.symbol) then
                parts += t
                traverse(t.prefix)
              else
                traverse(t.underlying)
            case t: TermRef =>
              if !isExcluded(t.symbol) then
                traverse(t.info)
                traverse(t.prefix)
            case t: ThisType if t.cls.is(Module) && t.cls.isStaticOwner =>
              traverse(t.cls.sourceModule.termRef)
            case t: ConstantType =>
              traverse(t.underlying)
            case t: TypeParamRef =>
              traverse(t.underlying)
              if ctx.typerState.constraint.contains(t) then provisional = true
            case t: TermParamRef =>
              traverse(t.underlying)
            case t =>
              traverseChildren(t)

      def apply(tp: Type): (collection.Set[Type], Boolean) =
        provisional = false
        parts = mutable.LinkedHashSet()
        partSeen.clear()
        traverse(tp)
        (parts, provisional)
    end collectParts

    val seen = TypeHashSet()
    val incomplete = TypeHashSet()

    def collectCompanions(tp: Type, parts: collection.Set[Type]): TermRefSet =
      val companions = new TermRefSet

      def iscopeRefs(t: Type): TermRefSet =
        implicitScopeCache.get(t) match
          case Some(is) =>
            is.companionRefs
          case None =>
            if seen.contains(t) then
              incomplete.addEntry(tp) // all references for `t` will be accounted for in `seen` so we return `EmptySet`.
              TermRefSet.empty        // on the other hand, the refs of `tp` are now inaccurate, so `tp` is marked incomplete.
            else
              seen.addEntry(t)
              val is = recur(t)
              if !implicitScopeCache.contains(t) then incomplete.addEntry(tp)
              is.companionRefs
      end iscopeRefs

      def addCompanion(pre: Type, companion: Symbol) =
        if companion.exists && !companion.isAbsent() then
          companions += TermRef(pre, companion)

      def addCompanions(t: Type) = implicitScopeCache.get(t) match
        case Some(iscope) =>
          companions ++= iscope.companionRefs
        case None => t match
          case t: TypeRef =>
            val sym = t.symbol
            val pre = t.prefix
            addPath(pre)
            addCompanion(pre,
              if sym.isClass then sym.companionModule
              else pre.member(sym.name.toTermName)
                .suchThat(companion => companion.is(Module) && companion.owner == sym.owner)
                .symbol)
            if sym.isClass then
              for p <- t.parents do companions ++= iscopeRefs(p)
            else
              companions ++= iscopeRefs(t.underlying)
      end addCompanions

      def addPath(pre: Type): Unit = pre.dealias match
        case pre: ThisType if pre.cls.is(Module) && pre.cls.isStaticOwner =>
          addPath(pre.cls.sourceModule.termRef)
        case pre: TermRef if !isExcluded(pre.symbol) =>
          pre.info match
            case info: SingletonType =>
              addPath(info)
            case info: TypeRef if info.symbol.is(Module) =>
              addCompanion(info.prefix, info.symbol.sourceModule)
              addPath(info.prefix)
            case _ =>
              companions += pre
              addPath(pre.prefix)
        case _ =>

      parts.foreach(addCompanions)
      companions
    end collectCompanions

    def recur(tp: Type): OfTypeImplicits =
      val (parts, provisional) = collectParts(tp)
      val companions = collectCompanions(tp, parts)
      val result = OfTypeImplicits(tp, companions)(runContext)
      if Config.cacheImplicitScopes
         && tp.hash != NotCached
        && !provisional
        && (tp eq rootTp)              // first type traversed is always cached
           || !incomplete.contains(tp) // other types are cached if they are not incomplete
      then implicitScopeCache(tp) = result
      result

    record(i"computeIScope")
    recur(rootTp)
  end computeIScope

  /** The implicit scope of a type `tp`, which is specified by the following definitions.
   *
   *  A reference is an _anchor_ if it refers to an object, a class, a trait, an
   *  abstract type, an opaque type alias, or a match type alias. References to
   *  packages and package objects are anchors only under -source:3.0-migration.
   *
   *  The _anchors_ of a type `T` is a set of references defined as follows:
   *
   *   - If `T` is a reference to an anchor, `T` itself plus, if `T` is of the form
   *     `P#A`, the anchors of `P`.
   *   - If `T` is an alias of `U`, the anchors of `U`.
   *   - If `T` is a reference to a type parameter, the union of the anchors of both of its bounds.
   *   - If `T` is a singleton reference, the anchors of its underlying type, plus,
   *     if `T` is of the form `(P#x).type`, the anchors of `P`.
   *   - If `T` is the this-type of a static object, the anchors of a term reference to that object.
   *   - If `T` is some other type, the union of the anchors of each constituent type of `T`.
   *
   *  The _implicit scope_ of a type `tp` is the smallest set S of term references (i.e. TermRefs)
   *  such that
   *
   *   - If `T` is a reference to a class, S includes a reference to the companion object
   *     of the class, if it exists, as well as the implicit scopes of all of `T`'s parent classes.
   *   - If `T` is a reference to an object, S includes `T` itself as well as
   *     the implicit scopes of all of `T`'s parent classes.
   *   - If `T` is a reference to an opaque type alias named `A`, S includes
   *     a reference to an object `A` defined in the same scope as the type, if it exists,
   *     as well as the implicit scope of `T`'s underlying type or bounds.
   *   - If `T` is a reference to an an abstract type or match type alias named `A`,
   *     S includes a reference to an object `A` defined in the same scope as the type,
   *     if it exists, as well as the implicit scopes of `T`'s lower and upper bound,
   *     if present.
   *   - If `T` is a reference to an anchor of the form `p.A` then S also includes
   *     all term references on the path `p`.
   *   - If `T` is some other type, S includes the implicit scopes of all anchors of `T`.
   */
  def implicitScope(tp: Type)(using Context): OfTypeImplicits =
    implicitScopeCache.get(tp) match
      case Some(is) => is
      case None =>
        record(i"implicitScope")
        val liftToAnchors = new TypeMap:
          override def stopAtStatic = true
          private val seen = TypeHashSet()

          def applyToUnderlying(t: TypeProxy) =
            if seen.contains(t) then
              WildcardType
            else
              seen.addEntry(t)
              t.underlying match
                case TypeBounds(lo, hi) =>
                  if defn.isBottomTypeAfterErasure(lo) then apply(hi)
                  else AndType.make(apply(lo), apply(hi))
                case u => apply(u)

          def apply(t: Type) = t.dealias match
            case t: TypeRef =>
              if t.symbol.isClass || isAnchor(t.symbol) then t else applyToUnderlying(t)
            case t: TypeVar => apply(t.underlying)
            case t: ParamRef => applyToUnderlying(t)
            case t: ConstantType => apply(t.underlying)
            case t => mapOver(t)
        end liftToAnchors
        val liftedTp = liftToAnchors(tp)
        if liftedTp eq tp then
          record(i"implicitScope unlifted")
          computeIScope(tp)
        else
          record(i"implicitScope lifted")
          val liftedIScope = implicitScopeCache.getOrElse(liftedTp, computeIScope(liftedTp))
          OfTypeImplicits(tp, liftedIScope.companionRefs)(runContext)
  end implicitScope

  protected def reset(): Unit =
    implicitScopeCache.clear()
end ImplicitRunInfo

/** The implicit resolution part of type checking */
trait Implicits:
  self: Typer =>

  import tpd._

  override def viewExists(from: Type, to: Type)(using Context): Boolean =
       !from.isError
    && !to.isError
    && !ctx.isAfterTyper
    && ctx.mode.is(Mode.ImplicitsEnabled)
    && from.isValueType
    && (  from.isValueSubType(to)
       || inferView(dummyTreeOfType(from), to)
            (using ctx.fresh.addMode(Mode.ImplicitExploration).setExploreTyperState()).isSuccess
          // TODO: investigate why we can't TyperState#test here
       )

  /** Find an implicit conversion to apply to given tree `from` so that the
   *  result is compatible with type `to`.
   */
  def inferView(from: Tree, to: Type)(using Context): SearchResult = {
    record("inferView")
    if    to.isAny
       || to.isAnyRef
       || to.isRef(defn.UnitClass)
       || from.tpe.isRef(defn.NothingClass)
       || from.tpe.isRef(defn.NullClass)
       || !ctx.mode.is(Mode.ImplicitsEnabled)
       || from.isInstanceOf[Super]
       || (from.tpe eq NoPrefix)
    then NoMatchingImplicitsFailure
    else {
      def adjust(to: Type) = to.stripTypeVar.widenExpr match {
        case SelectionProto(name, memberProto, compat, true) =>
          SelectionProto(name, memberProto, compat, privateOK = false)
        case tp => tp
      }
      try inferImplicit(adjust(to), from, from.span)
      catch {
        case ex: AssertionError =>
          implicits.println(s"view $from ==> $to")
          implicits.println(ctx.typerState.constraint.show)
          implicits.println(TypeComparer.explained(_.isSubType(from.tpe, to)))
          throw ex
      }
    }
  }

  private var synthesizer: Synthesizer = null

  /** Find an implicit argument for parameter `formal`.
   *  Return a failure as a SearchFailureType in the type of the returned tree.
   */
  def inferImplicitArg(formal: Type, span: Span)(using Context): Tree =
    inferImplicit(formal, EmptyTree, span) match
      case SearchSuccess(arg, _, _) => arg
      case fail @ SearchFailure(failed) =>
        if fail.isAmbiguous then failed
        else
          if synthesizer == null then synthesizer = Synthesizer(this)
          synthesizer.tryAll(formal, span).orElse(failed)

  /** Search an implicit argument and report error if not found */
  def implicitArgTree(formal: Type, span: Span)(using Context): Tree = {
    val arg = inferImplicitArg(formal, span)
    if (arg.tpe.isInstanceOf[SearchFailureType])
      report.error(missingArgMsg(arg, formal, ""), ctx.source.atSpan(span))
    arg
  }

  def missingArgMsg(arg: Tree, pt: Type, where: String)(using Context): String = {

    def msg(shortForm: String)(headline: String = shortForm) = arg match {
      case arg: Trees.SearchFailureIdent[?] =>
        shortForm
      case _ =>
        arg.tpe match {
          case tpe: SearchFailureType =>
            val original = arg match
              case Inlined(call, _, _) => call
              case _ => arg

            i"""$headline.
              |I found:
              |
              |    ${original.show.replace("\n", "\n    ")}
              |
              |But ${tpe.explanation}."""
        }
    }

    def location(preposition: String) = if (where.isEmpty) "" else s" $preposition $where"

    /** Extract a user defined error message from a symbol `sym`
     *  with an annotation matching the given class symbol `cls`.
     */
    def userDefinedMsg(sym: Symbol, cls: Symbol) = for {
      ann <- sym.getAnnotation(cls)
      Trees.Literal(Constant(msg: String)) <- ann.argument(0)
    }
    yield msg


    arg.tpe match {
      case ambi: AmbiguousImplicits =>
        object AmbiguousImplicitMsg {
          def unapply(search: SearchSuccess): Option[String] =
            userDefinedMsg(search.ref.symbol, defn.ImplicitAmbiguousAnnot)
        }

        /** Construct a custom error message given an ambiguous implicit
         *  candidate `alt` and a user defined message `raw`.
         */
        def userDefinedAmbiguousImplicitMsg(alt: SearchSuccess, raw: String) = {
          val params = alt.ref.underlying match {
            case p: PolyType => p.paramNames.map(_.toString)
            case _           => Nil
          }
          def resolveTypes(targs: List[Tree])(using Context) =
            targs.map(a => fullyDefinedType(a.tpe, "type argument", a.span))

          // We can extract type arguments from:
          //   - a function call:
          //     @implicitAmbiguous("msg A=${A}")
          //     implicit def f[A](): String = ...
          //     implicitly[String] // found: f[Any]()
          //
          //   - an eta-expanded function:
          //     @implicitAmbiguous("msg A=${A}")
          //     implicit def f[A](x: Int): String = ...
          //     implicitly[Int => String] // found: x => f[Any](x)

          val call = closureBody(alt.tree) // the tree itself if not a closure
          val (_, targs, _) = decomposeCall(call)
          val args = resolveTypes(targs)(using ctx.fresh.setTyperState(alt.tstate))
          err.userDefinedErrorString(raw, params, args)
        }

        (ambi.alt1, ambi.alt2) match {
          case (alt @ AmbiguousImplicitMsg(msg), _) =>
            userDefinedAmbiguousImplicitMsg(alt, msg)
          case (_, alt @ AmbiguousImplicitMsg(msg)) =>
            userDefinedAmbiguousImplicitMsg(alt, msg)
          case _ =>
            msg(s"ambiguous implicit arguments: ${ambi.explanation}${location("of")}")(
                s"ambiguous implicit arguments of type ${pt.show} found${location("for")}")
        }

      case _ =>
        val userDefined = userDefinedMsg(pt.typeSymbol, defn.ImplicitNotFoundAnnot).map(raw =>
          err.userDefinedErrorString(
            raw,
            pt.typeSymbol.typeParams.map(_.name.unexpandedName.toString),
            pt.widenExpr.dropDependentRefinement.argInfos))

        def hiddenImplicitsAddendum: String =

          def hiddenImplicitNote(s: SearchSuccess) =
            em"\n\nNote: given instance ${s.ref.symbol.showLocated} was not considered because it was not imported with `import given`."

          def findHiddenImplicitsCtx(c: Context): Context =
            if c == NoContext then c
            else c.freshOver(findHiddenImplicitsCtx(c.outer)).addMode(Mode.FindHiddenImplicits)

          val normalImports = arg.tpe match
            case fail: SearchFailureType =>
              if (fail.expectedType eq pt) || isFullyDefined(fail.expectedType, ForceDegree.none) then
                inferImplicit(fail.expectedType, fail.argument, arg.span)(
                  using findHiddenImplicitsCtx(ctx)) match {
                  case s: SearchSuccess => hiddenImplicitNote(s)
                  case f: SearchFailure =>
                    f.reason match {
                      case ambi: AmbiguousImplicits => hiddenImplicitNote(ambi.alt1)
                      case r => ""
                    }
                }
              else
                // It's unsafe to search for parts of the expected type if they are not fully defined,
                // since these come with nested contexts that are lost at this point. See #7249 for an
                // example where searching for a nested type causes an infinite loop.
                ""

          def suggestedImports = importSuggestionAddendum(pt)
          if normalImports.isEmpty then suggestedImports else normalImports
        end hiddenImplicitsAddendum

        msg(userDefined.getOrElse(
          em"no implicit argument of type $pt was found${location("for")}"))() ++
        hiddenImplicitsAddendum
    }
  }

  /** A string indicating the formal parameter corresponding to a  missing argument */
  def implicitParamString(paramName: TermName, methodStr: String, tree: Tree)(using Context): String =
    tree match {
      case Select(qual, nme.apply) if defn.isFunctionType(qual.tpe.widen) =>
        val qt = qual.tpe.widen
        val qt1 = qt.dealiasKeepAnnots
        def addendum = if (qt1 eq qt) "" else (i"\nwhich is an alias of: $qt1")
        em"parameter of ${qual.tpe.widen}$addendum"
      case _ =>
        em"${ if paramName.is(EvidenceParamName) then "an implicit parameter"
              else s"parameter $paramName" } of $methodStr"
    }

  /** An Eql[T, U] instance is assumed
   *   - if one of T, U is an error type, or
   *   - if one of T, U is a subtype of the lifted version of the other,
   *     unless strict equality is set.
   */
  def assumedCanEqual(ltp: Type, rtp: Type)(using Context) = {
    // Map all non-opaque abstract types to their upper bound.
    // This is done to check whether such types might plausibly be comparable to each other.
    val lift = new TypeMap {
      def apply(t: Type): Type = t match {
        case t: TypeRef =>
          t.info match {
            case TypeBounds(lo, hi) if lo.ne(hi) && !t.symbol.is(Opaque) => apply(hi)
            case _ => t
          }
        case t: RefinedType =>
          apply(t.parent)
        case _ =>
          if (variance > 0) mapOver(t) else t
      }
    }

    ltp.isError
    || rtp.isError
    || !strictEquality && (ltp <:< lift(rtp) || rtp <:< lift(ltp))
  }

  /** Check that equality tests between types `ltp` and `rtp` make sense */
  def checkCanEqual(ltp: Type, rtp: Type, span: Span)(using Context): Unit =
    if (!ctx.isAfterTyper && !assumedCanEqual(ltp, rtp)) {
      val res = implicitArgTree(defn.EqlClass.typeRef.appliedTo(ltp, rtp), span)
      implicits.println(i"Eql witness found for $ltp / $rtp: $res: ${res.tpe}")
    }

  /** Find an implicit parameter or conversion.
   *  @param pt              The expected type of the parameter or conversion.
   *  @param argument        If an implicit conversion is searched, the argument to which
   *                         it should be applied, EmptyTree otherwise.
   *  @param span            The position where errors should be reported.
   */
  def inferImplicit(pt: Type, argument: Tree, span: Span)(using Context): SearchResult =
    trace(s"search implicit ${pt.show}, arg = ${argument.show}: ${argument.tpe.show}", implicits, show = true) {
      record("inferImplicit")
      assert(ctx.phase.allowsImplicitSearch,
        if (argument.isEmpty) i"missing implicit parameter of type $pt after typer"
        else i"type error: ${argument.tpe} does not conform to $pt${err.whyNoMatchStr(argument.tpe, pt)}")
      val result0 =
        try ImplicitSearch(pt, argument, span).bestImplicit
        catch case ce: CyclicReference =>
          ce.inImplicitSearch = true
          throw ce

      val result =
        result0 match {
          case result: SearchSuccess =>
            result.tstate.commit()
            ctx.gadt.restore(result.gstate)
            implicits.println(i"success: $result")
            implicits.println(i"committing ${result.tstate.constraint} yielding ${ctx.typerState.constraint} in ${ctx.typerState}")
            result
          case result: SearchFailure if result.isAmbiguous =>
            val deepPt = pt.deepenProto
            if (deepPt ne pt) inferImplicit(deepPt, argument, span)
            else if (migrateTo3 && !ctx.mode.is(Mode.OldOverloadingResolution))
              withMode(Mode.OldOverloadingResolution)(inferImplicit(pt, argument, span)) match {
                case altResult: SearchSuccess =>
                  report.migrationWarning(
                    s"According to new implicit resolution rules, this will be ambiguous:\n${result.reason.explanation}",
                    ctx.source.atSpan(span))
                  altResult
                case _ =>
                  result
              }
            else result
          case NoMatchingImplicitsFailure =>
            SearchFailure(new NoMatchingImplicits(pt, argument, ctx.typerState.constraint))
          case _ =>
            result0
        }
      // If we are at the outermost implicit search then emit the implicit dictionary, if any.
      ctx.searchHistory.emitDictionary(span, result)
    }

  /** Try to typecheck an implicit reference */
  def typedImplicit(cand: Candidate, pt: Type, argument: Tree, span: Span)(using Context): SearchResult =  trace(i"typed implicit ${cand.ref}, pt = $pt, implicitsEnabled == ${ctx.mode is ImplicitsEnabled}", implicits, show = true) {
    if ctx.run.isCancelled then NoMatchingImplicitsFailure
    else
      record("typedImplicit")
      val ref = cand.ref
      val generated: Tree = tpd.ref(ref).withSpan(span.startPos)
      val locked = ctx.typerState.ownedVars
      val adapted =
        if (argument.isEmpty)
          adapt(generated, pt.widenExpr, locked)
        else {
          def untpdGenerated = untpd.TypedSplice(generated)
          def tryConversion(using Context) = {
            val untpdConv =
              if (ref.symbol.is(Given))
                untpd.Select(
                  untpd.TypedSplice(
                    adapt(generated,
                      defn.ConversionClass.typeRef.appliedTo(argument.tpe, pt),
                      locked)),
                  nme.apply)
              else untpdGenerated
            typed(
              untpd.Apply(untpdConv, untpd.TypedSplice(argument) :: Nil),
              pt, locked)
          }
          pt match
            case selProto @ SelectionProto(_: TermName, mbrType, _, _) if cand.isExtension =>
              def tryExtension(using Context) =
                extMethodApply(untpd.Select(untpdGenerated, selProto.extensionName), argument, mbrType)
              if cand.isConversion then
                val extensionCtx, conversionCtx = ctx.fresh.setNewTyperState()
                val extensionResult = tryExtension(using extensionCtx)
                val conversionResult = tryConversion(using conversionCtx)
                if !extensionCtx.reporter.hasErrors then
                  extensionCtx.typerState.commit()
                  if !conversionCtx.reporter.hasErrors then
                    report.error(em"ambiguous implicit: $generated is eligible both as an implicit conversion and as an extension method container")
                  extensionResult
                else
                  conversionCtx.typerState.commit()
                  conversionResult
              else tryExtension
            case _ =>
              tryConversion
        }
      if ctx.reporter.hasErrors
         || !cand.ref.symbol.isAccessibleFrom(cand.ref.prefix)
      then
        ctx.reporter.removeBufferedMessages
        adapted.tpe match {
          case _: SearchFailureType => SearchFailure(adapted)
          case _ =>
            // Special case for `$conforms` and `<:<.refl`. Showing them to the users brings
            // no value, so we instead report a `NoMatchingImplicitsFailure`
            if (adapted.symbol == defn.Predef_conforms || adapted.symbol == defn.SubType_refl)
              NoMatchingImplicitsFailure
            else
              SearchFailure(adapted.withType(new MismatchedImplicit(ref, pt, argument)))
        }
      else
        val returned =
          if (cand.isExtension) Applications.ExtMethodApply(adapted)
          else adapted
        SearchSuccess(returned, ref, cand.level)(ctx.typerState, ctx.gadt)
    }

  /** An implicit search; parameters as in `inferImplicit` */
  class ImplicitSearch(protected val pt: Type, protected val argument: Tree, span: Span)(using Context):
    assert(argument.isEmpty || argument.tpe.isValueType || argument.tpe.isInstanceOf[ExprType],
        em"found: $argument: ${argument.tpe}, expected: $pt")

    private def nestedContext() =
      ctx.fresh.setMode(ctx.mode &~ Mode.ImplicitsEnabled)

    private def isCoherent = pt.isRef(defn.EqlClass)

    val wideProto = pt.widenExpr

    /** The expected type where parameters and uninstantiated typevars are replaced by wildcard types */
    val wildProto: Type =
      if argument.isEmpty then wildApprox(pt)
      else ViewProto(wildApprox(argument.tpe.widen), wildApprox(pt))
        // Not clear whether we need to drop the `.widen` here. All tests pass with it in place, though.

    val isNot: Boolean = wildProto.classSymbol == defn.NotClass

    /** Try to type-check implicit reference, after checking that this is not
      * a diverging search
      */
    def tryImplicit(cand: Candidate, contextual: Boolean): SearchResult =
      if checkDivergence(cand) then
        SearchFailure(new DivergingImplicit(cand.ref, wideProto, argument))
      else {
        val history = ctx.searchHistory.nest(cand, pt)
        val result =
          typedImplicit(cand, pt, argument, span)(using nestedContext().setNewTyperState().setFreshGADTBounds.setSearchHistory(history))
        result match {
          case res: SearchSuccess =>
            ctx.searchHistory.defineBynameImplicit(wideProto, res)
          case _ =>
            result
        }
      }

    /** Search a list of eligible implicit references */
    private def searchImplicit(eligible: List[Candidate], contextual: Boolean): SearchResult =

      /** Compare previous success with reference and level to determine which one would be chosen, if
       *  an implicit starting with the reference was found.
       */
      def compareCandidate(prev: SearchSuccess, ref: TermRef, level: Int): Int =
        if (prev.ref eq ref) 0
        else if (prev.level != level) prev.level - level
        else explore(compare(prev.ref, ref))(using nestedContext())

      /** If `alt1` is also a search success, try to disambiguate as follows:
       *    - If alt2 is preferred over alt1, pick alt2, otherwise return an
       *      ambiguous implicits error.
       */
      def disambiguate(alt1: SearchResult, alt2: SearchSuccess) = alt1 match
        case alt1: SearchSuccess =>
          var diff = compareCandidate(alt1, alt2.ref, alt2.level)
          assert(diff <= 0)   // diff > 0 candidates should already have been eliminated in `rank`
          if diff == 0 then
            // Fall back: if both results are extension method applications,
            // compare the extension methods instead of their wrappers.
            object extMethodApply:
              def unapply(t: Tree): Option[Type] = t match
                case t: Applications.ExtMethodApply => Some(methPart(stripApply(t.app)).tpe)
                case _ => None
            end extMethodApply

            (alt1.tree, alt2.tree) match
              case (extMethodApply(ref1: TermRef), extMethodApply(ref2: TermRef)) =>
                diff = compare(ref1, ref2)
              case _ =>

          if diff < 0 then alt2
          else if diff > 0 then alt1
          else SearchFailure(new AmbiguousImplicits(alt1, alt2, pt, argument))
        case _: SearchFailure => alt2

      /** Faced with an ambiguous implicits failure `fail`, try to find another
       *  alternative among `pending` that is strictly better than both ambiguous
       *  alternatives.  If that fails, return `fail`
       */
      def healAmbiguous(pending: List[Candidate], fail: SearchFailure) = {
        val ambi = fail.reason.asInstanceOf[AmbiguousImplicits]
        val newPending = pending.filter(cand =>
          compareCandidate(ambi.alt1, cand.ref, cand.level) < 0 &&
          compareCandidate(ambi.alt2, cand.ref, cand.level) < 0)
        rank(newPending, fail, Nil).recoverWith(_ => fail)
      }

      /** Try to find a best matching implicit term among all the candidates in `pending`.
       *  @param pending   The list of candidates that remain to be tested
       *  @param found     The result obtained from previously tried candidates
       *  @param rfailures A list of all failures from previously tried candidates in reverse order
       *
       *  The scheme is to try candidates one-by-one. If a trial is successful:
       *   - if the query term is a `Not[T]` treat it a failure,
       *   - otherwise, if a previous search was also successful, handle the ambiguity
       *     in `disambiguate`,
       *   - otherwise, continue the search with all candidates that are not strictly
       *     worse than the successful candidate.
       *  If a trial failed:
       *    - if the query term is a `Not[T]` treat it as a success,
       *    - otherwise, if the failure is an ambiguity, try to heal it (see @healAmbiguous)
       *      and return an ambiguous error otherwise. However, under Scala2 mode this is
       *      treated as a simple failure, with a warning that semantics will change.
       *    - otherwise add the failure to `rfailures` and continue testing the other candidates.
       */
      def rank(pending: List[Candidate], found: SearchResult, rfailures: List[SearchFailure]): SearchResult =
        pending match {
          case cand :: remaining =>
            negateIfNot(tryImplicit(cand, contextual)) match {
              case fail: SearchFailure =>
                if (fail.isAmbiguous)
                  if migrateTo3 then
                    val result = rank(remaining, found, NoMatchingImplicitsFailure :: rfailures)
                    if (result.isSuccess)
                      warnAmbiguousNegation(fail.reason.asInstanceOf[AmbiguousImplicits])
                    result
                  else healAmbiguous(remaining, fail)
                else rank(remaining, found, fail :: rfailures)
              case best: SearchSuccess =>
                if (ctx.mode.is(Mode.ImplicitExploration) || isCoherent)
                  best
                else disambiguate(found, best) match {
                  case retained: SearchSuccess =>
                    val newPending =
                      if (retained eq found) || remaining.isEmpty then remaining
                      else remaining.filter(cand =>
                        compareCandidate(retained, cand.ref, cand.level) <= 0)
                    rank(newPending, retained, rfailures)
                  case fail: SearchFailure =>
                    healAmbiguous(remaining, fail)
                }
            }
          case nil =>
            if (rfailures.isEmpty) found
            else found.recoverWith(_ => rfailures.reverse.maxBy(_.tree.treeSize))
        }

      def negateIfNot(result: SearchResult) =
        if (isNot)
          result match {
            case _: SearchFailure =>
              SearchSuccess(ref(defn.Not_value), defn.Not_value.termRef, 0)(
                ctx.typerState.fresh().setCommittable(true),
                ctx.gadt
              )
            case _: SearchSuccess =>
              NoMatchingImplicitsFailure
          }
        else result

      def warnAmbiguousNegation(ambi: AmbiguousImplicits) =
        report.migrationWarning(
          i"""Ambiguous implicits ${ambi.alt1.ref.symbol.showLocated} and ${ambi.alt2.ref.symbol.showLocated}
             |seem to be used to implement a local failure in order to negate an implicit search.
             |According to the new implicit resolution rules this is no longer possible;
             |the search will fail with a global ambiguity error instead.
             |
             |Consider using the scala.util.Not class to implement similar functionality.""",
             ctx.source.atSpan(span))

      /** A relation that influences the order in which implicits are tried.
       *  We prefer (in order of importance)
       *   1. more deeply nested definitions
       *   2. definitions with fewer implicit parameters
       *   3. definitions in subclasses
       *  The reason for (2) is that we want to fail fast if the search type
       *  is underconstrained. So we look for "small" goals first, because that
       *  will give an ambiguity quickly.
       */
      def prefer(cand1: Candidate, cand2: Candidate): Boolean =
        val level1 = cand1.level
        val level2 = cand2.level
        if level1 > level2 then return true
        if level1 < level2 then return false
        val sym1 = cand1.ref.symbol
        val sym2 = cand2.ref.symbol
        val arity1 = sym1.info.firstParamTypes.length
        val arity2 = sym2.info.firstParamTypes.length
        if arity1 < arity2 then return true
        if arity1 > arity2 then return false
        compareOwner(sym1, sym2) == 1

      /** Sort list of implicit references according to `prefer`.
       *  This is just an optimization that aims at reducing the average
       *  number of candidates to be tested.
       */
      def sort(eligible: List[Candidate]) = eligible match {
        case Nil => eligible
        case e1 :: Nil => eligible
        case e1 :: e2 :: Nil =>
          if (prefer(e2, e1)) e2 :: e1 :: Nil
          else eligible
        case _ =>
          try eligible.sortWith(prefer)
          catch case ex: IllegalArgumentException =>
            // diagnostic to see what went wrong
            for
              e1 <- eligible
              e2 <- eligible
              if prefer(e1, e2)
              e3 <- eligible
              if prefer(e2, e3) && !prefer(e1, e3)
            do
              val es = List(e1, e2, e3)
              println(i"transitivity violated for $es%, %\n ${es.map(_.implicitRef.underlyingRef.symbol.showLocated)}%, %")
            throw ex
      }

      rank(sort(eligible), NoMatchingImplicitsFailure, Nil)
    end searchImplicit

    private def searchImplicit(contextual: Boolean): SearchResult =
      val eligible =
        if contextual then ctx.implicits.eligible(wildProto)
        else implicitScope(wildProto).eligible
      searchImplicit(eligible, contextual) match
        case result: SearchSuccess =>
          result
        case failure: SearchFailure =>
          failure.reason match
            case _: AmbiguousImplicits => failure
            case reason =>
              if contextual then
                searchImplicit(contextual = false).recoverWith {
                  failure2 => failure2.reason match
                    case _: AmbiguousImplicits => failure2
                    case _ =>
                      reason match
                        case (_: DivergingImplicit) => failure
                        case _ => List(failure, failure2).maxBy(_.tree.treeSize)
                }
              else failure
    end searchImplicit

    /** Find a unique best implicit reference */
    def bestImplicit: SearchResult =
      // Before searching for contextual or implicit scope candidates we first check if
      // there is an under construction or already constructed term with which we can tie
      // the knot.
      //
      // Since any suitable term found is defined as part of this search it will always be
      // effectively in a more inner context than any other definition provided by
      // explicit definitions. Consequently these terms have the highest priority and no
      // other candidates need to be considered.
      recursiveRef match
        case ref: TermRef =>
          SearchSuccess(tpd.ref(ref).withSpan(span.startPos), ref, 0)(ctx.typerState, ctx.gadt)
        case _ =>
          searchImplicit(contextual = true)
    end bestImplicit

    def implicitScope(tp: Type): OfTypeImplicits = ctx.run.implicitScope(tp)

    /** All available implicits, without ranking */
    def allImplicits: Set[TermRef] = {
      val contextuals = ctx.implicits.eligible(wildProto).map(tryImplicit(_, contextual = true))
      val inscope = implicitScope(wildProto).eligible.map(tryImplicit(_, contextual = false))
      (contextuals.toSet ++ inscope).collect {
        case success: SearchSuccess => success.ref
      }
    }

    /** Fields needed for divergence checking */
    @threadUnsafe lazy val ptCoveringSet = wideProto.coveringSet
    @threadUnsafe lazy val ptSize = wideProto.typeSize
    @threadUnsafe lazy val wildPt = wildApprox(wideProto)

    /**
    * Check if the supplied candidate implicit and target type indicate a diverging
    * implicit search.
    *
    * @param cand The candidate implicit to be explored.
    * @param pt   The target type for the above candidate.
    * @result     True if this candidate/pt are divergent, false otherwise.
    */
    def checkDivergence(cand: Candidate): Boolean =
      // For full details of the algorithm see the SIP:
      //   https://docs.scala-lang.org/sips/byname-implicits.html
      util.Stats.record("checkDivergence")

      // Unless we are able to tie a recursive knot, we report divergence if there is an
      // open implicit using the same candidate implicit definition which has a type which
      // is larger (see `typeSize`) and is constructed using the same set of types and type
      // constructors (see `coveringSet`).
      //
      // We are able to tie a recursive knot if there is compatible term already under
      // construction which is separated from this context by at least one by name argument
      // as we ascend the chain of open implicits to the outermost search context.

      @tailrec
      def loop(history: SearchHistory, belowByname: Boolean): Boolean =
        history match
          case prev @ OpenSearch(cand1, tp, outer) =>
            if cand1.ref eq cand.ref then
              lazy val wildTp = wildApprox(tp.widenExpr)
              if belowByname && (wildTp <:< wildPt) then
                fullyDefinedType(tp, "by-name implicit parameter", span)
                false
              else if prev.typeSize > ptSize || prev.coveringSet != ptCoveringSet then
                loop(outer, tp.isByName || belowByname)
              else
                prev.typeSize < ptSize
                || wildTp =:= wildPt
                || loop(outer, tp.isByName || belowByname)
            else loop(outer, tp.isByName || belowByname)
          case _ => false

      loop(ctx.searchHistory, pt.isByName)
    end checkDivergence

    /**
     * Return the reference, if any, to a term under construction or already constructed in
     * the current search history corresponding to the supplied target type.
     *
     * A term is eligible if its type is a subtype of the target type and either it has
     * already been constructed and is present in the current implicit dictionary, or it is
     * currently under construction and is separated from the current search context by at
     * least one by name argument position.
     *
     * Note that because any suitable term found is defined as part of this search it will
     * always be effectively in a more inner context than any other definition provided by
     * explicit definitions. Consequently these terms have the highest priority and no other
     * candidates need to be considered.
     *
     * @param pt  The target type being searched for.
     * @result    The corresponding dictionary reference if any, NoType otherwise.
     */
    def recursiveRef: Type =
      val found = ctx.searchHistory.refBynameImplicit(wideProto)
      if found.exists then
        found
      else if !ctx.searchHistory.byname && !pt.isByName then
        NoType // No recursion unless at least one open implicit is by name ...
      else
        // We are able to tie a recursive knot if there is compatible term already under
        // construction which is separated from this context by at least one by name
        // argument as we ascend the chain of open implicits to the outermost search
        // context.
        @tailrec
        def loop(history: SearchHistory, belowByname: Boolean): Type =
          history match
            case OpenSearch(cand, tp, outer) =>
              if (belowByname || tp.isByName) && tp.widenExpr <:< wideProto then tp
              else loop(outer, belowByname || tp.isByName)
            case _ => NoType

        loop(ctx.searchHistory, pt.isByName) match
          case NoType => NoType
          case tp => ctx.searchHistory.linkBynameImplicit(tp.widenExpr)
    end recursiveRef
  end ImplicitSearch
end Implicits

/**
 * Records the history of currently open implicit searches.
 *
 * A search history maintains a list of open implicit searches (`openSearchPairs`) a shortcut flag
 * indicating whether any of these are by name (`byname`) and a reference to the root
 * search history (`root`) which in turn maintains a possibly empty dictionary of
 * recursive implicit terms constructed during this search.
 *
 * A search history provides operations to create a nested search history, check for
 * divergence, enter by name references and definitions in the implicit dictionary, lookup
 * recursive references and emit a complete implicit dictionary when the outermost search
 * is complete.
 */
abstract class SearchHistory:
  val root: SearchRoot
  /** Does this search history contain any by name implicit arguments. */
  val byname: Boolean
  def openSearchPairs: List[(Candidate, Type)]

  /**
   * Create the state for a nested implicit search.
   * @param cand The candidate implicit to be explored.
   * @param pt   The target type for the above candidate.
   * @result     The nested history.
   */
  def nest(cand: Candidate, pt: Type)(using Context): OpenSearch = OpenSearch(cand, pt, this)

  // The following are delegated to the root of this search history.
  def linkBynameImplicit(tpe: Type)(using Context): TermRef =
    root.linkBynameImplicit(tpe)
  def refBynameImplicit(tpe: Type)(using Context): Type =
    root.refBynameImplicit(tpe)
  def defineBynameImplicit(tpe: Type, result: SearchSuccess)(using Context): SearchResult =
    root.defineBynameImplicit(tpe, result)

  // This is NOOP unless at the root of this search history.
  def emitDictionary(span: Span, result: SearchResult)(using Context): SearchResult = result

  override def toString: String = s"SearchHistory(open = $openSearchPairs, byname = $byname)"
end SearchHistory

case class OpenSearch(cand: Candidate, pt: Type, outer: SearchHistory)(using Context) extends SearchHistory:
  val root = outer.root
  val byname = outer.byname || pt.isByName
  def openSearchPairs = (cand, pt) :: outer.openSearchPairs

  // The typeSize and coveringSet of the current search.
  // Note: It is important to cache size and covering sets since types
  // in search histories can contain type variables that can be instantiated
  // by nested implicit searches, thus leading to types in search histories
  // that grow larger the deeper the search gets. This can mask divergence.
  // An example is in neg/9504.scala
  lazy val typeSize = pt.typeSize
  lazy val coveringSet = pt.coveringSet
end OpenSearch

/**
 * The the state corresponding to the outermost context of an implicit searcch.
 */
final class SearchRoot extends SearchHistory:
  val root = this
  val byname = false
  def openSearchPairs = Nil

  /** The dictionary of recursive implicit types and corresponding terms for this search. */
  var myImplicitDictionary: mutable.Map[Type, (TermRef, tpd.Tree)] = null
  private def implicitDictionary =
    if myImplicitDictionary == null then
      myImplicitDictionary = mutable.Map.empty[Type, (TermRef, tpd.Tree)]
    myImplicitDictionary

  /**
   * Link a reference to an under-construction implicit for the provided type to its
   * defining occurrence via the implicit dictionary, creating a dictionary entry for this
   * type if one does not yet exist.
   *
   * @param tpe  The type to link.
   * @result     The TermRef of the corresponding dictionary entry.
   */
  override def linkBynameImplicit(tpe: Type)(using Context): TermRef =
    implicitDictionary.get(tpe) match {
      case Some((ref, _)) => ref
      case None =>
        val lazyImplicit = newLazyImplicit(tpe)
        val ref = lazyImplicit.termRef
        implicitDictionary.put(tpe, (ref, tpd.EmptyTree))
        ref
    }

  /**
   * Look up an implicit dictionary entry by type.
   *
   * If present yield the TermRef corresponding to the eventual dictionary entry,
   * otherwise NoType.
   *
   * @param tpe The type to look up.
   * @result    The corresponding TermRef, or NoType if none.
   */
  override def refBynameImplicit(tpe: Type)(using Context): Type =
    implicitDictionary.get(tpe) match
      case Some((tp, _)) => tp
      case None => NoType

  /**
   * Define a pending dictionary entry if any.
   *
   * If the provided type corresponds to an under-construction by name implicit, then use
   * the tree contained in the provided SearchSuccess as its definition, returning an
   * updated result referring to dictionary entry. Otherwise return the SearchSuccess
   * unchanged.
   *
   * @param  tpe    The type for which the entry is to be defined
   * @param  result The SearchSuccess corresponding to tpe
   * @result        A SearchResult referring to the newly created dictionary entry if tpe
   *                is an under-construction by name implicit, the provided result otherwise.
   */
  override def defineBynameImplicit(tpe: Type, result: SearchSuccess)(using Context): SearchResult =
    implicitDictionary.get(tpe) match {
      case Some((ref, _)) =>
        implicitDictionary.put(tpe, (ref, result.tree))
        SearchSuccess(tpd.ref(ref).withSpan(result.tree.span), result.ref, result.level)(result.tstate, result.gstate)
      case None => result
    }

  /**
   * Emit the implicit dictionary at the completion of an implicit search.
   *
   * @param span   The position at which the search is elaborated.
   * @param result The result of the search prior to substitution of recursive references.
   * @result       The elaborated result, comprising the implicit dictionary and a result tree
   *               substituted with references into the dictionary.
   */
  override def emitDictionary(span: Span, result: SearchResult)(using Context): SearchResult =
    if (myImplicitDictionary == null || implicitDictionary.isEmpty) result
    else
      result match {
        case failure: SearchFailure => failure
        case success @ SearchSuccess(tree, _, _) =>
          import tpd._

          // We might have accumulated dictionary entries for by name implicit arguments
          // which are not in fact used recursively either directly in the outermost result
          // term, or indirectly via other dictionary entries. We prune these out, recursively
          // eliminating entries until all remaining entries are at least transtively referred
          // to in the outermost result term.
          @tailrec
          def prune(trees: List[Tree], pending: List[(TermRef, Tree)], acc: List[(TermRef, Tree)]): List[(TermRef, Tree)] = pending match {
            case Nil => acc
            case ps =>
              val (in, out) = ps.partition {
                case (vref, rhs) =>
                  trees.exists(_.existsSubTree {
                    case id: Ident => id.symbol == vref.symbol
                    case _ => false
                  })
              }
              if (in.isEmpty) acc
              else prune(in.map(_._2) ++ trees, out, in ++ acc)
          }

          val pruned = prune(List(tree), implicitDictionary.map(_._2).toList, Nil)
          myImplicitDictionary = null
          if (pruned.isEmpty) result
          else if (pruned.exists(_._2 == EmptyTree)) NoMatchingImplicitsFailure
          else {
            // If there are any dictionary entries remaining after pruning, construct a dictionary
            // class of the form,
            //
            // class <dictionary> {
            //   val $_lazy_implicit_$0 = ...
            //   ...
            //   val $_lazy_implicit_$n = ...
            // }
            //
            // Where the RHSs of the $_lazy_implicit_$n are the terms used to populate the dictionary
            // via defineByNameImplicit.
            //
            // The returned search result is then of the form,
            //
            // {
            //   class <dictionary> { ... }
            //   val $_lazy_implicit_$nn = new <dictionary>
            //   result.tree // with dictionary references substituted in
            // }

            val parents = List(defn.ObjectType, defn.SerializableType)
            val classSym = newNormalizedClassSymbol(ctx.owner, LazyImplicitName.fresh().toTypeName, Synthetic | Final, parents, coord = span)
            val vsyms = pruned.map(_._1.symbol)
            val nsyms = vsyms.map(vsym => newSymbol(classSym, vsym.name, EmptyFlags, vsym.info, coord = span).entered)
            val vsymMap = (vsyms zip nsyms).toMap

            val rhss = pruned.map(_._2)
            // Substitute dictionary references into dictionary entry RHSs
            val rhsMap = new TreeTypeMap(treeMap = {
              case id: Ident if vsymMap.contains(id.symbol) =>
                tpd.ref(vsymMap(id.symbol))(using ctx.withSource(id.source)).withSpan(id.span)
              case tree => tree
            })
            val nrhss = rhss.map(rhsMap(_))

            val vdefs = (nsyms zip nrhss) map {
              case (nsym, nrhs) => ValDef(nsym.asTerm, nrhs.changeNonLocalOwners(nsym))
            }

            val constr = newConstructor(classSym, Synthetic, Nil, Nil).entered
            val classDef = ClassDef(classSym, DefDef(constr), vdefs)

            val valSym = newLazyImplicit(classSym.typeRef, span)
            val inst = ValDef(valSym, New(classSym.typeRef, Nil))

            // Substitute dictionary references into outermost result term.
            val resMap = new TreeTypeMap(treeMap = {
              case id: Ident if vsymMap.contains(id.symbol) =>
                Select(tpd.ref(valSym), id.name)
              case tree => tree
            })

            val res = resMap(tree)

            val blk = Block(classDef :: inst :: Nil, res).withSpan(span)

            success.copy(tree = blk)(success.tstate, success.gstate)
          }
      }
end SearchRoot

/** A set of term references where equality is =:= */
sealed class TermRefSet(using Context):
  private val elems = new java.util.LinkedHashMap[TermSymbol, Type | List[Type]]

  def isEmpty = elems.size == 0

  def += (ref: TermRef): Unit =
    val pre = ref.prefix
    val sym = ref.symbol.asTerm
    elems.get(sym) match
      case null =>
        elems.put(sym, pre)
      case prefix: Type =>
        if !(prefix =:= pre) then elems.put(sym, pre :: prefix :: Nil)
      case prefixes: List[Type] =>
        if !prefixes.exists(_ =:= pre) then elems.put(sym, pre :: prefixes)

  def ++= (that: TermRefSet): Unit =
    if !that.isEmpty then that.foreach(+=)

  def foreach[U](f: TermRef => U): Unit =
    elems.forEach((sym: TermSymbol, prefixes: Type | List[Type]) =>
      prefixes match
        case prefix: Type => f(TermRef(prefix, sym))
        case prefixes: List[Type] => prefixes.foreach(pre => f(TermRef(pre, sym))))

  // used only for debugging
  def showAsList: List[TermRef] = {
    val buffer = new mutable.ListBuffer[TermRef]
    foreach(tr => buffer += tr)
    buffer.toList
  }

  override def toString = showAsList.toString

object TermRefSet:

  @sharable val empty = new TermRefSet(using NoContext):
    override def += (ref: TermRef): Unit = throw UnsupportedOperationException("+=")

end TermRefSet

