package ch.epfl.lamp.grading

import org.scalatest.Reporter
import org.scalatest.events._
import java.io.File
import collection.mutable.ListBuffer
import org.scalatest.exceptions.TestFailedException

class GradingReporter extends Reporter {

  val outfile: File = {
    val prop = System.getProperty("scalatest.reportFile")
    if (prop == null) sys.error("scalatest.reportFile property not defined")
    new File(prop)
  }

  private def writeOutfile(s: String): Unit = {
    val p = new java.io.PrintWriter(outfile)
    try {
      p.print(s)
    } finally {
      p.close()
    }
  }

  def extractWeight(s: String): (Int, String) = {
    val (num, rest) = s.span(c => c != '\n')
    try {
      (num.toInt, rest.drop(1))
    } catch {
      case e: NumberFormatException =>
        sys.error("Could not extract weight from test name string\n"+ s)
    }
  }

  private var totalWeight = 0
  private var failWeight = 0
  private var successWeight = 0

  private val log = new ListBuffer[String]()

  def logTestFailure(name: String, message: String, weight: Int): Unit = {
    log += "[Test Description] "+ name
    log += "[Observed Error] "+ message
    log += "[Lost Points] "+ weight + "\n"
  }

  def initialize(): Unit = {
    totalWeight = 0
    failWeight = 0
    successWeight = 0
  }

  def writeResults(): Unit = {
    val scoreText =
      "Your solution achieved a testing score of %d out of %d.\n".format(
          successWeight, totalWeight)
    val header = scoreText + {
      if (successWeight == totalWeight) "Great job!!!\n\n"
      else {
        """Below you can see a short feedback for every test that failed, indicating the reason
          |for the test failure and how many points you lost for each individual test.
          |
          |""".stripMargin
      }
    }

    val sumErrorMessage = {
      if (failWeight + successWeight != totalWeight) {
        """An internal error happened while testing your code. Please send your entire feedback message
          |to one of the teaching assistants. success %d, fail %d, total %d
          |
          |""".stripMargin.format(successWeight, failWeight, totalWeight)
      } else ""
    }

    val feedback =
      header + sumErrorMessage + log.mkString("\n").trim // trim removes newlines at the end

    val weightValues = successWeight.toString + ";" + totalWeight.toString + "\n"

    try {
      writeOutfile(weightValues + feedback)
    } catch {
      case e: java.io.IOException =>
        val m = "Error writing test results to file\n"+ e.toString
        sys.error(m)
    }
  }

  /*
   * Event fields:
   *  - testName: the string that the test writer passes to the `test` method,
   *    i.e., `test("testName") {...}`
   *  - message: the reason for the test failure, a message generated by
   *    ScalaTest, e.g., "2 did not equal 3"
   */

  def apply(event: Event): Unit = {
    event match {
      /* We don't get a `TestStarting` for ignored tests, but we do get here
       * one tests that use `pending`.
       */
      case e: TestStarting =>
//        println("[STRT] "+ extractWeight(e.testName)._2)
        val (weigth, _) = extractWeight(e.testName)
        totalWeight += weigth

      case e: TestFailed =>
//        println("[FAIL] "+ extractWeight(e.testName)._2)
        val (weight, name) = extractWeight(e.testName)
        failWeight += weight
        val msg = e.message + (e.throwable match {
          case None => ""
          case Some(testFailed: TestFailedException) =>
            ""
          case Some(thrown) =>
            /* The standard output is captured by sbt and printed as
             * `testing tool debug output` in the feedback.
             */
            println("[test failure log] test name: "+ name)
            println(exceptionString(thrown) + "\n\n")
            "\n  [exception was thrown] detailed error message in debug output section below"
        })
        logTestFailure(name, msg, weight)

      case e: TestSucceeded =>
//        println("[SUCC] "+ extractWeight(e.testName)._2)
        val (weight, _) = extractWeight(e.testName)
        successWeight += weight

      case e: TestPending =>
//        println("[PEND] "+ extractWeight(e.testName)._2)
        val (weight, _) = extractWeight(e.testName)
        successWeight += weight

      case e: RunStarting =>
//        println("[INFO] Starting new run")
        initialize()

      case e: RunStopped =>
        writeResults()

      case e: RunAborted =>
        writeResults()

      case e: RunCompleted =>
        writeResults()

      case _ => ()
    }
  }

  def exceptionString(e: Throwable): String =
    e.toString +"\n"+ e.getStackTrace.take(25).map(_.toString).mkString("\n")
}
