package strawman
package collection


import mutable.{ GrowableBuilder, Builder }

class PrefixMap[A]
  extends mutable.Map[String, A]
    with mutable.MapOps[String, A, mutable.Map, PrefixMap[A]]
    with StrictOptimizedIterableOps[(String, A), mutable.Iterable, PrefixMap[A]] {

  import scala.Predef.{ augmentString => _ }

  var suffixes: immutable.Map[Char, PrefixMap[A]] = immutable.Map.empty
  var value: Option[A] = None

  def get(s: String): Option[A] =
    if (s.isEmpty) value
    else suffixes get (stringToStringOps(s)(0)) flatMap (_.get(s substring 1))

  def withPrefix(s: String): PrefixMap[A] =
    if (s.isEmpty) this
    else {
      val leading = stringToStringOps(s)(0)
      suffixes get leading match {
        case None =>
          suffixes = suffixes + (leading -> empty)
        case _ =>
      }
      suffixes(leading) withPrefix (s substring 1)
    }

  def iterator(): Iterator[(String, A)] =
    (for (v <- value.iterator.toStrawman) yield ("", v)) ++
      (for ((chr, m) <- suffixes.iterator();
            (s, v) <- m.iterator()) yield (chr +: stringToStringOps(s), v))

  def add(kv: (String, A)): this.type = {
    withPrefix(kv._1).value = Some(kv._2)
    this
  }

  def subtract(s: String): this.type  = {
    if (s.isEmpty) { val prev = value; value = None; prev }
    else suffixes get (stringToStringOps(s)(0)) flatMap (_.remove(s substring 1))
    this
  }

  def empty = new PrefixMap[A]

  // Overloading of transformation methods that should return a PrefixMap
  def map[B](f: ((String, A)) => (String, B)): PrefixMap[B] = PrefixMap.from(View.Map(this, f))
  def flatMap[B](f: ((String, A)) => IterableOnce[(String, B)]): PrefixMap[B] = PrefixMap.from(View.FlatMap(this, f))

  // Override concat method to refine its return type
  override def concat[B >: A](suffix: Iterable[(String, B)]): PrefixMap[B] = PrefixMap.from(View.Concat(this, suffix))

  // Members declared in strawman.collection.mutable.Clearable
  def clear(): Unit = suffixes = immutable.Map.empty
  // Members declared in strawman.collection.IterableOps
  protected[this] def fromSpecificIterable(coll: Iterable[(String, A)]): PrefixMap[A] = PrefixMap.from(coll)
  protected[this] def newSpecificBuilder(): Builder[(String, A), PrefixMap[A]] = PrefixMap.newBuilder()
  def mapFactory: MapFactory[mutable.Map] = mutable.Map
  protected[this] def mapFromIterable[K2, V2](it: Iterable[(K2, V2)]): mutable.Map[K2,V2] = mapFactory.from(it)
}

object PrefixMap {
  def empty[A] = new PrefixMap[A]

  def from[A](source: IterableOnce[(String, A)]): PrefixMap[A] =
    source match {
      case pm: PrefixMap[A] => pm
      case _ => (newBuilder() ++= source).result()
    }

  def apply[A](kvs: (String, A)*): PrefixMap[A] = from(kvs.toStrawman)

  def newBuilder[A](): Builder[(String, A), PrefixMap[A]] =
    new GrowableBuilder[(String, A), PrefixMap[A]](empty)

  implicit def toFactory[A](self: this.type): Factory[(String, A), PrefixMap[A]] =
    new Factory[(String, A), PrefixMap[A]] {
      def fromSpecific(it: IterableOnce[(String, A)]): PrefixMap[A] = self.from(it)
      def newBuilder(): Builder[(String, A), PrefixMap[A]] = self.newBuilder()
    }

}

