package mycollections

import strawman.collection._
import strawman.collection.mutable.{Builder, ImmutableBuilder}

final class Capped[A] private (val capacity: Int, val length: Int, offset: Int, elems: Array[Any])
  extends immutable.Iterable[A]
    with IterableOps[A, Capped, Capped[A]]
    with StrictOptimizedIterableOps[A, Capped, Capped[A]] { self =>

  def this(capacity: Int) =
    this(capacity, length = 0, offset = 0, elems = Array.ofDim(capacity))

  val iterableFactory: IterableFactory[Capped] = new CappedFactory(capacity)
  protected[this] def fromSpecificIterable(coll: Iterable[A]): Capped[A] = iterableFactory.from(coll)
  protected[this] def newSpecificBuilder(): Builder[A, Capped[A]] = iterableFactory.newBuilder()

  override def knownSize: Int = length

  def iterator(): Iterator[A] = view.iterator()

  override def view: IndexedView[A] = new IndexedView[A] {
    def length: Int = self.length
    def apply(i: Int): A = self(i)
  }

  def apply(i: Int): A = elems((i + offset) % capacity).asInstanceOf[A]

  def appended[B >: A](elem: B): Capped[B] = {
    val newElems = Array.ofDim[Any](capacity)
    Array.copy(elems, 0, newElems, 0, capacity)
    val (newOffset, newLength) =
      if (length == capacity) {
        newElems(offset) = elem
        ((offset + 1) % capacity, length)
      } else {
        newElems(length) = elem
        (offset, length + 1)
      }
    new Capped[B](capacity, newLength, newOffset, newElems)
  }

  @`inline` def :+ [B >: A](elem: B): Capped[B] = appended(elem)

}

class CappedFactory(capacity: Int) extends IterableFactory[Capped] {

  def from[A](source: IterableOnce[A]): Capped[A] =
    source match {
      case cs: Capped[A] if cs.capacity == capacity => cs
      case _ => (newBuilder[A]() ++= source).result()
    }

  def empty[A]: Capped[A] = new Capped[A](capacity)

  def newBuilder[A](): Builder[A, Capped[A]] =
    new ImmutableBuilder[A, Capped[A]](empty) {
      def addOne(elem: A): this.type = { elems = elems :+ elem; this }
    }

}

object Usage {

  object Capped8 extends CappedFactory(capacity = 8)

  val cs1 = Capped8(1, 2, 3, 4, 5)

  println(cs1)

  val cs2 = cs1 ++ immutable.List.fill(5)(0)
  println(cs2)

  println(cs2.filter(x => x % 2 == 1))

  println(cs1.take(2))
  println(cs1.drop(2))

}

class Capped1[A] private (val capacity: Int, val length: Int, offset: Int, elems: Array[Any])
  extends immutable.Iterable[A] { self =>

  def this(capacity: Int) =
    this(capacity, length = 0, offset = 0, elems = Array.ofDim(capacity))

  def appended[B >: A](elem: B): Capped1[B] = {
    val newElems = Array.ofDim[Any](capacity)
    Array.copy(elems, 0, newElems, 0, capacity)
    val (newOffset, newLength) =
      if (length == capacity) {
        newElems(offset) = elem
        ((offset + 1) % capacity, length)
      } else {
        newElems(length) = elem
        (offset, length + 1)
      }
    new Capped1[B](capacity, newLength, newOffset, newElems)
  }

  @`inline` def :+ [B >: A](elem: B): Capped1[B] = appended(elem)

  def apply(i: Int): A = elems((i + offset) % capacity).asInstanceOf[A]

  def iterator(): Iterator[A] = new AbstractIterator[A] {
    private var current = 0
    def hasNext = current < self.length
    def next(): A = {
      val elem = self(current)
      current += 1
      elem
    }
  }

  def iterableFactory: IterableFactory[immutable.Iterable] = immutable.Iterable
  protected[this] def fromSpecificIterable(coll: Iterable[A]): immutable.Iterable[A] = iterableFactory.from(coll)
  protected[this] def newSpecificBuilder(): Builder[A, immutable.Iterable[A]] = iterableFactory.newBuilder()

}

class Capped2[A] private (val capacity: Int, val length: Int, offset: Int, elems: Array[Any])
  extends immutable.Iterable[A]
    with IterableOps[A, Capped2, Capped2[A]] { self =>

  def this(capacity: Int) =
    this(capacity, length = 0, offset = 0, elems = Array.ofDim(capacity))

  def appended[B >: A](elem: B): Capped2[B] = {
    val newElems = Array.ofDim[Any](capacity)
    Array.copy(elems, 0, newElems, 0, capacity)
    val (newOffset, newLength) =
      if (length == capacity) {
        newElems(offset) = elem
        ((offset + 1) % capacity, length)
      } else {
        newElems(length) = elem
        (offset, length + 1)
      }
    new Capped2[B](capacity, newLength, newOffset, newElems)
  }

  @`inline` def :+ [B >: A](elem: B): Capped2[B] = appended(elem)

  def apply(i: Int): A = elems((i + offset) % capacity).asInstanceOf[A]

  def iterator(): Iterator[A] = new AbstractIterator[A] {
    private var current = 0
    def hasNext = current < self.length
    def next(): A = {
      val elem = self(current)
      current += 1
      elem
    }
  }

  val iterableFactory: IterableFactory[Capped2] = new Capped2Factory(capacity)
  protected[this] def fromSpecificIterable(coll: Iterable[A]): Capped2[A] = iterableFactory.from(coll)
  protected[this] def newSpecificBuilder(): Builder[A, Capped2[A]] = iterableFactory.newBuilder()

}

class Capped2Factory(capacity: Int) extends IterableFactory[Capped2] {

  def from[A](source: IterableOnce[A]): Capped2[A] =
    source match {
      case cs: Capped2[A] if cs.capacity == capacity => cs
      case _ => (newBuilder[A]() ++= source).result()
    }

  def empty[A]: Capped2[A] = new Capped2[A](capacity)

  def newBuilder[A](): Builder[A, Capped2[A]] =
    new ImmutableBuilder[A, Capped2[A]](empty) {
      def addOne(elem: A): this.type = { elems = elems :+ elem; this }
    }

}