package scala.spores

import scala.language.implicitConversions
import scala.language.experimental.macros
import scala.reflect.macros.whitebox
import scala.spores.util.Feedback

/** Implicit conversion between spores and spores with excluded types. */
object Conversions {
  // TODO(jvican): Consider importing all this into the spores package
  implicit def toExcluded[T, R, A](s: Spore[T, R]): Spore[T, R] {
    type Excluded = A
  } = macro SporeTranslator.toExcludedSpore[T, R, A]

  implicit def toExcluded[R, A](s: NullarySpore[R]): NullarySpore[R] {
    type Excluded = A
  } = macro SporeTranslator.toExcludedNullary[R, A]

  implicit def toExcluded[T1, T2, R, A](
      s: Spore2[T1, T2, R]): Spore2[T1, T2, R] { type Excluded = A } =
    macro SporeTranslator.toExcludedSpore2[T1, T2, R, A]

  implicit def toExcluded[T1, T2, T3, R, A](
      s: Spore3[T1, T2, T3, R]): Spore3[T1, T2, T3, R] { type Excluded = A } =
    macro SporeTranslator.toExcludedSpore3[T1, T2, T3, R, A]

  implicit def toExcluded[T1, T2, T3, T4, R, A](
      s: Spore4[T1, T2, T3, T4, R]): Spore4[T1, T2, T3, T4, R] {
    type Excluded = A
  } =
    macro SporeTranslator.toExcludedSpore4[T1, T2, T3, T4, R, A]

  implicit def toExcluded[T1, T2, T3, T4, T5, R, A](
      s: Spore5[T1, T2, T3, T4, T5, R]): Spore5[T1, T2, T3, T4, T5, R] {
    type Excluded = A
  } =
    macro SporeTranslator.toExcludedSpore5[T1, T2, T3, T4, T5, R, A]

  implicit def toExcluded[T1, T2, T3, T4, T5, T6, R, A](s: Spore6[
    T1,
    T2,
    T3,
    T4,
    T5,
    T6,
    R]): Spore6[T1, T2, T3, T4, T5, T6, R] {
    type Excluded = A
  } =
    macro SporeTranslator.toExcludedSpore6[T1, T2, T3, T4, T5, T6, R, A]

  implicit def toExcluded[T1, T2, T3, T4, T5, T6, T7, R, A](
      s: Spore7[T1, T2, T3, T4, T5, T6, T7, R]): Spore7[T1,
                                                        T2,
                                                        T3,
                                                        T4,
                                                        T5,
                                                        T6,
                                                        T7,
                                                        R] {
    type Excluded = A
  } =
    macro SporeTranslator.toExcludedSpore7[T1, T2, T3, T4, T5, T6, T7, R, A]
}

object SporeTranslator {

  /** Creates a compile error if a reference for a type in the spore
    * is any of the type members of the defined `Excluded` type.
    *
    * @tparam A Single type or tuple.
    * @param s Block expression generated by the `SporeGenerator` as follows:
    *          {{{
    *          {
    *            class anonclass extends Spore...[...] {...}
    *            new anonclass(...)
    *          }
    *          }}}
    * @return  Modified expression:
    *          {{{
    *          {
    *            class anonclass extends Spore...[...] {...}
    *            new anonclass(...) { type Excluded = (..._ }
    *          }
    *          }}}
    */
  def constructTree[A: c.WeakTypeTag](c: whitebox.Context)(
      s: c.Tree): c.universe.Tree = {
    import c.universe._
    val atpe = weakTypeOf[A]

    // TODO(jvican): Increase the number of types here
    val avoidedList: List[c.universe.Type] = {
      if (atpe <:< weakTypeOf[(Any, Any)] ||
          atpe <:< weakTypeOf[(Any, Any, Any)] ||
          atpe <:< weakTypeOf[(Any, Any, Any, Any)] ||
          atpe <:< weakTypeOf[(Any, Any, Any, Any, Any)] ||
          atpe <:< weakTypeOf[(Any, Any, Any, Any, Any, Any)])
        atpe.typeArgs
      else List[c.universe.Type](atpe)
    }

    object traverser extends Traverser {
      var mentionedTypes = List[TypeTree]()
      override def traverse(tree: Tree): Unit = tree match {
        case tt @ TypeTree() => mentionedTypes = tt :: mentionedTypes
        case _ => super.traverse(tree)
      }
    }
    traverser.traverse(s)
    debug(s"${showCode(s)}")
    debug(s"Traversed: ${traverser.mentionedTypes}")

    val NothingType = typeOf[Nothing]
    /* Check that btm is indeed the
     * bottom type and that tpe is not */
    def isBottomType(btm: Type, tpe: Type) =
      btm =:= NothingType && !(tpe =:= btm)

    /* This is the check: compiler error if some TypeTree
     * in 's' has a type that is <:< of something in A */
    traverser.mentionedTypes.foreach { t =>
      avoidedList.foreach { at =>
        if (t.tpe <:< at && !isBottomType(t.tpe, at)) {
          c.abort(t.pos,
                  Feedback.InvalidReferenceToExcludedType(t.tpe.toString))
        }
      }
    }

    /* Divide the spore into pieces that are put together
     * to create a Spore[...] {type Excluded = ...} */
    val Block(stmts, newInstance) = s
    val correctFormat = stmts.headOption.exists {
      case sporeDef: ClassDef => true
      case _ => false
    }

    if (!correctFormat) c.abort(s.pos, Feedback.MissingSporeClassDef)
    else {
      val sporeDef = stmts.head
      val sporeSym = sporeDef.symbol
      val q"new ${ _ }(...$constructorArgs)" = newInstance
      val excludedSporeInstantiation =
        q"""
        $sporeDef
        new $sporeSym(...$constructorArgs) {type Excluded = $atpe}
      """

      debug(s"Excluded transformed spore:\n$excludedSporeInstantiation")
      excludedSporeInstantiation
    }
  }

  def toExcludedNullary[R: c.WeakTypeTag, A: c.WeakTypeTag](
      c: whitebox.Context)(s: c.Expr[NullarySpore[R]])
    : c.Expr[NullarySpore[R] { type Excluded = A }] = {
    c.Expr[NullarySpore[R] { type Excluded = A }](constructTree[A](c)(s.tree))
  }

  def toExcludedSpore[T: c.WeakTypeTag, R: c.WeakTypeTag, A: c.WeakTypeTag](
      c: whitebox.Context)(
      s: c.Expr[Spore[T, R]]): c.Expr[Spore[T, R] { type Excluded = A }] = {
    c.Expr[Spore[T, R] { type Excluded = A }](constructTree[A](c)(s.tree))
  }

  def toExcludedSpore2[T1: c.WeakTypeTag,
                       T2: c.WeakTypeTag,
                       R: c.WeakTypeTag,
                       A: c.WeakTypeTag](c: whitebox.Context)(
      s: c.Expr[Spore2[T1, T2, R]])
    : c.Expr[Spore2[T1, T2, R] { type Excluded = A }] = {
    c.Expr[Spore2[T1, T2, R] { type Excluded = A }](
      constructTree[A](c)(s.tree))
  }

  def toExcludedSpore3[T1: c.WeakTypeTag,
                       T2: c.WeakTypeTag,
                       T3: c.WeakTypeTag,
                       R: c.WeakTypeTag,
                       A: c.WeakTypeTag](c: whitebox.Context)(
      s: c.Expr[Spore3[T1, T2, T3, R]])
    : c.Expr[Spore3[T1, T2, T3, R] { type Excluded = A }] = {
    c.Expr[Spore3[T1, T2, T3, R] { type Excluded = A }](
      constructTree[A](c)(s.tree))
  }

  def toExcludedSpore4[T1: c.WeakTypeTag,
                       T2: c.WeakTypeTag,
                       T3: c.WeakTypeTag,
                       T4: c.WeakTypeTag,
                       R: c.WeakTypeTag,
                       A: c.WeakTypeTag](c: whitebox.Context)(
      s: c.Expr[Spore4[T1, T2, T3, T4, R]])
    : c.Expr[Spore4[T1, T2, T3, T4, R] { type Excluded = A }] = {
    c.Expr[Spore4[T1, T2, T3, T4, R] { type Excluded = A }](
      constructTree[A](c)(s.tree))
  }

  def toExcludedSpore5[T1: c.WeakTypeTag,
                       T2: c.WeakTypeTag,
                       T3: c.WeakTypeTag,
                       T4: c.WeakTypeTag,
                       T5: c.WeakTypeTag,
                       R: c.WeakTypeTag,
                       A: c.WeakTypeTag](c: whitebox.Context)(
      s: c.Expr[Spore5[T1, T2, T3, T4, T5, R]])
    : c.Expr[Spore5[T1, T2, T3, T4, T5, R] { type Excluded = A }] = {
    c.Expr[Spore5[T1, T2, T3, T4, T5, R] { type Excluded = A }](
      constructTree[A](c)(s.tree))
  }

  def toExcludedSpore6[T1: c.WeakTypeTag,
                       T2: c.WeakTypeTag,
                       T3: c.WeakTypeTag,
                       T4: c.WeakTypeTag,
                       T5: c.WeakTypeTag,
                       T6: c.WeakTypeTag,
                       R: c.WeakTypeTag,
                       A: c.WeakTypeTag](c: whitebox.Context)(
      s: c.Expr[Spore6[T1, T2, T3, T4, T5, T6, R]])
    : c.Expr[Spore6[T1, T2, T3, T4, T5, T6, R] { type Excluded = A }] = {
    c.Expr[Spore6[T1, T2, T3, T4, T5, T6, R] { type Excluded = A }](
      constructTree[A](c)(s.tree))
  }

  def toExcludedSpore7[T1: c.WeakTypeTag,
                       T2: c.WeakTypeTag,
                       T3: c.WeakTypeTag,
                       T4: c.WeakTypeTag,
                       T5: c.WeakTypeTag,
                       T6: c.WeakTypeTag,
                       T7: c.WeakTypeTag,
                       R: c.WeakTypeTag,
                       A: c.WeakTypeTag](c: whitebox.Context)(
      s: c.Expr[Spore7[T1, T2, T3, T4, T5, T6, T7, R]])
    : c.Expr[Spore7[T1, T2, T3, T4, T5, T6, T7, R] { type Excluded = A }] = {
    c.Expr[Spore7[T1, T2, T3, T4, T5, T6, T7, R] { type Excluded = A }](
      constructTree[A](c)(s.tree))
  }
}
