package tastyquery

import tastyquery.Spans.Span
import tastyquery.Spans.NoSpan

final class SourcePosition private[tastyquery] (val sourceFile: SourceFile, private[tastyquery] val span: Span):
  override def toString(): String = s"$sourceFile:$span"

  private[tastyquery] def isAuto: Boolean = !span.exists && sourceFile != SourceFile.NoSource

  /** True if this source position is unknown. */
  def isUnknown: Boolean = !span.exists

  /** Tests whether this position gives line/column information.
    *
    * If this method returns `false`, the `*Line` and `*Column` methods will
    * throw `UnsupportedOperationException`.
    */
  def hasLineColumnInformation: Boolean = sourceFile.hasLineColumnInformation

  /** The offset of the main point of this position.
    *
    * The main point is where the `^` of an error message would typically be pointing.
    */
  def pointOffset: Int = span.point

  /** The line number of the main point of this position.
    *
    * The main point is where the `^` of an error message would typically be pointing.
    */
  def pointLine: Int = sourceFile.offsetToLine(pointOffset)

  /** The column number of the main point of this position.
    *
    * The main point is where the `^` of an error message would typically be pointing.
    */
  def pointColumn: Int = sourceFile.offsetToColumn(pointOffset)

  /** The offset of the start of this position. */
  def startOffset: Int = span.start

  /** The line number of the start of this position. */
  def startLine: Int = sourceFile.offsetToLine(startOffset)

  /** The column number of the start of this position. */
  def startColumn: Int = sourceFile.offsetToColumn(startOffset)

  /** The offset of the end of this position. */
  def endOffset: Int = span.end

  /** The line number of the end of this position. */
  def endLine: Int = sourceFile.offsetToLine(endOffset)

  /** The column number of the end of this position. */
  def endColumn: Int = sourceFile.offsetToColumn(endOffset)

  /** Is this position synthetic?
    *
    * A synthetic position is given to trees that are not found in source code,
    * but generated by the compiler. Examples are inferred type parameters or
    * summoned implicit values.
    */
  def isSynthetic: Boolean = span.isSynthetic

  /** Is this position source-derived, i.e., not synthetic? */
  def isSourceDerived: Boolean = span.isSourceDerived

  /** Is this a zero-extent position, i.e., one for which the start and end coincide? */
  def isZeroExtent: Boolean = span.isZeroExtent

  private[tastyquery] def union(that: SourcePosition): SourcePosition =
    if this.isUnknown then that
    else if that.isUnknown then this
    else
      assert(this.sourceFile == that.sourceFile, s"${this.sourceFile} != ${that.sourceFile}")
      SourcePosition(sourceFile, this.span.union(that.span))
end SourcePosition

object SourcePosition:
  val NoPosition: SourcePosition = new SourcePosition(SourceFile.NoSource, NoSpan)

  private[tastyquery] def auto(source: SourceFile, span: Span): SourcePosition =
    new SourcePosition(source, span)
end SourcePosition
