package ch.hevs.gdx2d.lib.physics;

import ch.hevs.gdx2d.components.physics.utils.PhysicsConstants;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.physics.box2d.Body;
import com.badlogic.gdx.physics.box2d.World;

import java.util.Vector;

/**
 * Gets the single and only physics world, implements a sort of singleton design
 * pattern (this is not a completely standard singleton because the class
 * which is generated (World) is not of the type of the containing class (PhysicsWorld).
 *
 * @author Pierre-André Mudry (mui)
 * @version 1.02
 */
public class PhysicsWorld {
	// Contains the object to be removed at each simulation step
	private static final Vector<Body> toRemove = new Vector<Body>();

	private static PhysicsWorld instance = null;
	private World w;

	private static float accumulator;
	private static float step = PhysicsConstants.STEP_SIZE;

	// Exists only to defeat normal instantiation
	private PhysicsWorld() {
		w = new World(new Vector2(0, PhysicsConstants.GRAVITY_VALUE), true);
	}

	/**
	 * @return The single and only World instance
	 */
	public static World getInstance() {
		if (instance == null) {
			instance = new PhysicsWorld();
		}
		return instance.w;
	}

	/**
	 * Schedules an object for deletion when it is doable
	 *
	 * @param obj The object to remove from simulation
	 */
	static public void scheduleForDeletion(Body obj) {
		assert (obj != null);
		toRemove.add(obj);
	}

	/**
	 * Call this to update the physics simulation
	 */
	static public void updatePhysics() {
		updatePhysics(Gdx.graphics.getDeltaTime());
	}

	/**
	 * Call this to update the physics simulation
	 *
	 * @param dt The amount of time that should be simulated
	 */
	static public void updatePhysics(float dt) {
		if (instance == null)
			getInstance();

		accumulator += dt;

		while (accumulator >= step) {
			instance.w.step(step, PhysicsConstants.VELOCITY_IT, PhysicsConstants.POSITION_IT);
			accumulator -= step / PhysicsConstants.SPEEDUP;
		}

		/**
		 *  Handles object deletions 
		 */
		for (Body obj : toRemove) {
			assert (obj != null);
			obj.setUserData(null);
			instance.w.destroyBody(obj);
		}

		toRemove.clear();
	}

	/**
	 * To destroy it (required for JNI calls)
	 */
	public static void dispose() {
		if (instance != null) {
			instance.w.dispose();
			instance.w = new World(new Vector2(0, PhysicsConstants.GRAVITY_VALUE), true);
		}
	}
}