package ch.hevs.medgift.epad.plugins.service;

import ch.hevs.medgift.plugins.common.models.*;

import java.util.List;

import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

/**
 * Persistence data
 * @author Himmouche Abderrahmane
 *
 */
@Stateless
public class PersistenceBean implements PersistenceBeanLocal {


    @PersistenceContext(unitName = "Epad_Medgift_WS")
    private EntityManager em;

    public void savePatient(Patient patient) {
        if (!this.checkPatient(patient.getEpadId()))
            em.persist(patient);
    }

    /**
     * Used to extract ROI features, persist data and update model
     *
     * @param patient data
     * @return patient from DB if exists else return the given patient
     */
    @Override //Return patient from DB if exists else return the given patient
    public Patient findPatient(Patient patient) {
        Query query = em.createQuery("SELECT  p FROM Patient p WHERE p.epadId = :id");
        query.setParameter("id", patient.getEpadId());
        @SuppressWarnings("unchecked")
        List<Patient> patients = query.getResultList();
        //System.out.println(patients.get(0).getEpadId());
        Patient fromDB = patients.isEmpty() ? null : patients.get(0);
        Patient result = fromDB != null ? fromDB : patient;
        return result;
    }

    @Override
    public boolean checkPatient(String epadId) {
        Query query = em.createQuery("Select COUNT(p.id) FROM Patient p WHERE p.epadId = :id");
        query.setParameter("id", epadId);
        //Patient p = (Patient) query.getSingleResult();
        Long count = (Long) query.getSingleResult();
        if (count < 1) {
            System.out.println("Patient exists !");
            return false;
        }
        System.out.println("New Person is added!");
        return true;
    }

    @Override
    public void saveImage(ImageReferenceEntity image) {
        if (!this.checkImage(image.getSopInstanceUid()))
            em.persist(image);

    }

    @Override
    public boolean checkImage(String sopInstanceUid) {
        Query query = em.createQuery("Select COUNT(i.id) FROM ImageReferenceEntity i WHERE i.sopInstanceUid = :uid");
        query.setParameter("uid", sopInstanceUid);
        Long count = (Long) query.getSingleResult();
        if (count < 1) {
            System.out.println("Image does not exist !");
            return false;
        }
        System.out.println("Image exists !");
        return true;
    }

    @Override
    public ImageReferenceEntity findImage(ImageReferenceEntity image) {
        Query query = em.createQuery("select  i FROM ImageReferenceEntity i WHERE i.sopInstanceUid = :uid");
        query.setParameter("uid", image.getSopInstanceUid());
        @SuppressWarnings("unchecked")
        List<ImageReferenceEntity> images = query.getResultList();
        ImageReferenceEntity fromDB = images.isEmpty() ? null : images.get(0);
        ImageReferenceEntity result = fromDB != null ? fromDB : image;
        return result;
    }

    @Override
    public void saveUser(User user) {
        // TODO Auto-generated method stub

    }

    @Override
    public void saveQuantitativeImagingFeatures(QuantitativeImagingFeatures features) {
        em.persist(features);
    }

    @Override
    public void saveAnnotation(ImageAnnotation annotation) {
        if (!this.checkAnnotation(annotation.getEpadId()))
            em.persist(annotation);

    }


    @Override
    public boolean checkAnnotation(String epadId) {
        Query query = em.createQuery("Select COUNT(a.id) FROM ImageAnnotation a WHERE a.epadId = :epadId");
        query.setParameter("epadId", epadId);
        Long count = (Long) query.getSingleResult();
        if (count < 1) {
            System.out.println("Annotation exists !");
            return false;
        }
        System.out.println("New Annotation is added!");
        return true;
    }

    @Override
    public ImageAnnotation findAnnotation(ImageAnnotation annotation) {
        Query query = em.createQuery("select a FROM ImageAnnotation a WHERE a.epadId = :epadId");
        query.setParameter("epadId", annotation.getEpadId());
        @SuppressWarnings("unchecked")
        List<ImageAnnotation> annotations = query.getResultList();
        ImageAnnotation fromDB = annotations.isEmpty() ? null : annotations.get(0);
        ImageAnnotation result = fromDB != null ? fromDB : annotation;
        return result;
    }

    @Override
    public boolean checkUser(String loginName) {
        // TODO Auto-generated method stub
        return false;
    }

    @Override
    public User findUser(User user) {

        Query query = em.createQuery("select u FROM User u WHERE u.loginName = :name");
        query.setParameter("name", user.getLoginName());
        @SuppressWarnings("unchecked")
        List<User> users = query.getResultList();
        User fromDB = users.isEmpty() ? null : users.get(0);
        User result = fromDB != null ? fromDB : user;
        return result;
    }

    /**
     * Persist all data received from ePAD
     * Persist features analysis of a given ROI
     *
     * @param epadPost
     */
    @Override
    @TransactionAttribute(value = TransactionAttributeType.REQUIRED)
    public void persistAllDataWithoutPatchs(EpadAnalysisPost epadPost,
                                            EpadAnalysisResponse epadResponse) {
        //Patient
        Patient patient = new Patient();
        patient.setEpadId(epadPost.getPatient_epadId());
        patient = this.findPatient(patient); //Check if the patient exists in D

        patient.setBirthdate(epadPost.getPatient_birthdate());
        patient.setEthnicGroupe(epadPost.getPatient_ethnicGroup());
        patient.setName(epadPost.getPatient_name());
        patient.setSex(epadPost.getPatient_sex());

        //User
        User user = new User();
        user.setLoginName(epadPost.getUser_loginName());
        user = this.findUser(user); //Check if the user exists in DB

        user.setName(epadPost.getUser_name());
        user.setRoleInTrial(epadPost.getUser_roleInTrial());


        //Dicom Image
        ImageReferenceEntity image = new ImageReferenceEntity();
        image.setSopInstanceUid(epadPost.getImage_sopInstanceUid());
        image = this.findImage(image); //Check if the image exists in DB

        image.setImageMatrix(epadPost.getImage_matrix());


        //Annotation
        ImageAnnotation annotation = new ImageAnnotation();
        annotation.setEpadId(epadPost.getAnnotation_epadId());
        annotation = this.findAnnotation(annotation); //Check if the annotation exists in DB

        // Quick fix for not persisting cosmetic annotation name changes
        if(epadPost.getAnnotation_name().contains(" ("))
            annotation.setName(epadPost.getAnnotation_name().substring(0, epadPost.getAnnotation_name().indexOf(" (")));
        else
            annotation.setName(epadPost.getAnnotation_name());
        annotation.setComment(epadPost.getAnnotation_comment());
        annotation.setDatetime(epadPost.getAnnotation_datetime());
        annotation.setLabel(epadPost.getAnnotation_label());
        annotation.setCoordinates(epadPost.getAnnotation_coordinates());


        //Analysis Results
        QuantitativeImagingFeatures analysisResult = new QuantitativeImagingFeatures();
        analysisResult.setProjectId(epadPost.getProject_id());
        analysisResult.setFeatureType(epadPost.getFeatureType());
        analysisResult.setFeatureValues(epadResponse.getQuantitativeImagingFeatures().get(0));
        analysisResult.setModelFileName(epadPost.getModelFileName());
        //analysisResult = this.findAnalysis(analysisResult, annotation.getId()); // Check if the analysis exists

        //Construct associations
        image.setPatient(patient);
        //annotation.getQuantitativeImagingFeatures().add(analysisResult);
        annotation.setImageReference(image);
        analysisResult.setImageAnnotation(annotation);

        this.em.persist(user);
        this.em.persist(image);
    }

    /**
     * Used to extract ROI features, persist data and update model
     *
     * @param project_id
     * @return List of analysis features
     */
    @Override
    public List<QuantitativeImagingFeatures> getAnalysisResults(String project_id) {
        Query query = em.createQuery("Select q FROM QuantitativeImagingFeatures q WHERE q.projectId = :projectId");
        query.setParameter("projectId", project_id);
        @SuppressWarnings("unchecked")
        List<QuantitativeImagingFeatures> list = query.getResultList();
        return list;
    }

    @Override
    public List<QuantitativeImagingFeatures> getAnalysisResultsForModel(String modelFileName) {
        Query query = em.createQuery("Select q FROM QuantitativeImagingFeatures q WHERE q.modelFileName = :modelFileName");
        query.setParameter("modelFileName", modelFileName);
        @SuppressWarnings("unchecked")
        List<QuantitativeImagingFeatures> list = query.getResultList();
        return list;
    }

    @Override
    public QuantitativeImagingFeatures findAnalysis(QuantitativeImagingFeatures feature,
                                                    int annotation) {
        Query query = em.createQuery("select qf FROM QuantitativeImagingFeatures qf WHERE qf.imageAnnotation.id = :id");
        query.setParameter("id", annotation);
        @SuppressWarnings("unchecked")
        List<QuantitativeImagingFeatures> features = query.getResultList();
        QuantitativeImagingFeatures fromDB = features.isEmpty() ? null : features.get(0);
        QuantitativeImagingFeatures result = fromDB != null ? fromDB : feature;
        return result;
    }

    @Override
    public boolean deleteTrainingModelFeatures(String modelFileName) {
        Query query = em.createQuery("DELETE FROM QuantitativeImagingFeatures qf WHERE qf.modelFileName = :name");
        query.setParameter("name", modelFileName);

        try {
            query.executeUpdate();
            return true;
        } catch (Exception e) {
            return false;
        }
    }
	
}
