package ch.icosys.popjava.core.scripts;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class Popjavac {

	private static final String POPJAVA_FOLDER;

	static {
		String popJavaLocation = System.getenv("POPJAVA_LOCATION");

		if (popJavaLocation == null || popJavaLocation.isEmpty()) {
			if (ScriptUtils.isWindows()) {
				POPJAVA_FOLDER = "C:\\Users\\asraniel\\workspace\\PopJava\\release\\";
			} else {
				POPJAVA_FOLDER = "/usr/local/popj/";
			}
		} else {
			POPJAVA_FOLDER = popJavaLocation + File.separatorChar;
		}
	}

	private static final String JAVAC;

	static {
		String executable = "javac";
		try {
			Process p = Runtime.getRuntime().exec(executable);

			if (p.waitFor() != 0) {
				executable = null;
			}
		} catch (IOException | InterruptedException e) {
			e.printStackTrace();
		}

		if (executable == null) {
			if (ScriptUtils.isWindows()) {
				executable = "C:\\Program Files\\Java\\jdk1.7.0_11\\bin\\javac";
			} else {
				executable = "javac";
			}
		}
		JAVAC = executable;
	}

	private static final String JAR_FOLDER = "JarFile";

	public static final String POP_JAVA_JAR_FILE = "popjava.jar";

	private static final String COMPILER_JAR = POPJAVA_FOLDER + JAR_FOLDER + File.separatorChar + "popjparser.jar";

	private static final String POPJAVA_JAR = POPJAVA_FOLDER + JAR_FOLDER + File.separatorChar + POP_JAVA_JAR_FILE;

	private static final String COMPILER_MAIN = "POPJParser";

	private static boolean verbose = false;

	private static boolean noclean = false;

	private static boolean onlyTranslate = false;

	private static boolean onlyClean = false;

	private static final String HELP_MESSAGE = "POP-Java Compiler v1.0\n\n" +

			"This program is used to compile a POP-Java program\n\n" +

			"Usage: popjc <options> <source files>\n\n" +

			"OPTIONS:\n" + "   -h, --help               Show this message\n"
			+ "   -n, --noclean            Do not clean the intermediate Java file generated by the POP-Java parser\n"
			+ "   -l, --clean              Only clean the generated Java files, perform no other task \n"
			+ "   -t, --translate          Only translate the pjava files into java, do not compile them\n"
			+ "   -p, --popcpp <xml_file>  Compile a POP-Java parallel class for POP-C++ usage (Need XML additional informations file)\n"
			+ "   -j, --jar <filename>     Create a JAR archive with the given name (Need the JAR file name)\n"
			+ "   -v, --verbose            Verbose mode\n"
			+ "   -c, --classpath <files>  Include JAR or compiled Java class to the compilation process. Files must be separated by a semicolon \":\"\n\n"
			+

			"OPTIONS FOR POP-C++ INTEROPERABILITY:\n"
			+ "   -x, --xmlpopcpp <files>  Generate a canvas of the POP-C++ XML additional informations file for the given Java files. This option must be used alone\n"
			+ "   -g, --generate <pjava>   Generate the POP-C++ partial implementation to use the given POP-Java parclass in a POP-C++ application (NOT IMPLEMENTED YET)\n";

	private static void printHelp() {
		System.out.println(HELP_MESSAGE);
	}

	public static void main(String[] args) {
		main2(new String[] { "-x", "/home/asraniel/workspace/POPJavaFuture/example/mixed1/Integer.java" });
	}

	public static void main2(String[] args) {

		for (String arg : args) {
			System.out.println(arg);
		}

		if (args.length == 0 || ScriptUtils.containsOption(args, "-h") || ScriptUtils.containsOption(args, "--help")) {
			printHelp();
			return;
		}

		List<String> arguments = ScriptUtils.arrayToList(args);

		final String jar = ScriptUtils.getOption(arguments, "", false, "-j", "--jar");
		final String popcInfo = ScriptUtils.getOption(arguments, "", false, "-p", "--popcpp");

		String classPath = ScriptUtils.getOption(arguments, "", false, "-c", "--classpath");

		String generate = ScriptUtils.getOption(arguments, "", true, "-x", "--xmlpopcpp");

		verbose = ScriptUtils.removeOption(arguments, "-v", "--verbose");
		onlyTranslate = ScriptUtils.removeOption(arguments, "-t", "--translate");
		onlyClean = ScriptUtils.removeOption(arguments, "-l", "--clean");
		noclean = ScriptUtils.removeOption(arguments, "-n", "--noclean") | onlyTranslate;

		System.out.println("Files to compile:");
		if (arguments.size() == 1 && new File(arguments.get(0)).isDirectory()) {
			// TODO: Search for .pjava file in the provided folder
		}

		for (String file : arguments) {
			System.out.println(file);
		}

		if (!generate.isEmpty()) {
			try {
				generateXML("additional-infos.xml", arguments);
			} catch (Exception e) {
				e.printStackTrace();
			}

			return;
		}

		List<String> binaries = extractBinaries(arguments);

		for (String file : binaries) {
			if (classPath.isEmpty()) {
				classPath += File.pathSeparatorChar;
			}
			classPath += file;
		}

		List<String> classFiles = processFiles(arguments, classPath, popcInfo);

		if (!jar.isEmpty()) {
			jarFiles(jar, classFiles);
		}
	}

	private static void generateXML(String xml, List<String> files) throws IOException {
		BufferedWriter writer = new BufferedWriter(new FileWriter(xml));

		writer.write("<popjparser-infos>" + ScriptUtils.getNewline());

		for (String file : files) {
			writer.write("\t<popc-parclass file=\"" + file + "\" name=\"\" classuid=\"\" hasDestructor=\"true\"/>"
					+ ScriptUtils.getNewline());
		}

		writer.write("</popjparser-infos>");

		writer.close();
	}

	private static List<String> extractBinaries(List<String> files) {
		List<String> binaries = new ArrayList<>();

		for (int i = 0; i < files.size(); i++) {
			String file = files.get(i);
			if (file.endsWith(".jar") || file.endsWith(".class")) {
				binaries.add(file);
				files.remove(i);
				i--;
			}
		}

		return binaries;
	}

	private static List<String> expandSources(List<String> sourceFiles) {
		List<String> sources = new ArrayList<>();
		try {

			for (String path : sourceFiles) {
				File temp = new File(path);

				String name = temp.getName();
				Path parent = new File("").toPath();
				if (temp.getParentFile() != null) {
					parent = temp.getParentFile().toPath();
				}

				DirectoryStream<Path> stream = Files.newDirectoryStream(parent, name);
				for (Path source : stream) {
					String file = source.toFile().getPath();

					if (!new File(file).exists()) {
						System.err.println("File " + file + " does not exist, aborting compilation");
					}

					sources.add(file);
				}

				stream.close();
			}

		} catch (IOException e) {
			e.printStackTrace();
		}
		return sources;
	}

	private static List<String> processFiles(List<String> sourceParameter, String classPath, String popcInfo) {

		List<String> sourceFiles = expandSources(sourceParameter);

		String sources = "";
		for (int i = 0; i < sourceFiles.size(); i++) {
			String file = sourceFiles.get(i);
			if (!isMain(file) && file.endsWith(".pjava")) {
				sources += file;
				if (i != sourceFiles.size() - 1) {
					sources += File.pathSeparatorChar;
				}
			}
		}

		if (sources.length() > 0) {
			while (sources.charAt(sources.length() - 1) == File.pathSeparatorChar) {
				sources = sources.substring(0, sources.length() - 1);
			}
		}

		List<String> javaFiles = new ArrayList<>();
		List<String> convertedFiles = new ArrayList<>();

		// Convert files to java if needed
		for (String file : sourceFiles) {
			if (file.endsWith(".java")) {
				javaFiles.add(file);
			} else {
				String javaFile = file.replace(".pjava", ".java");
				convertedFiles.add(javaFile);

				convertFile(file, javaFile, sources, popcInfo);
			}
		}

		List<String> allFiles = new ArrayList<>(javaFiles);
		allFiles.addAll(convertedFiles);

		List<String> classFiles = Collections.emptyList();
		if (!onlyTranslate && !onlyClean) {
			classFiles = compileFiles(allFiles, classPath);
		}

		if (!noclean) {
			for (String file : convertedFiles) {
				new File(file).delete();
			}
		}
		// Compile them
		return classFiles;
	}

	private static boolean grep(final File file, final String search) throws IOException {
		BufferedReader reader = new BufferedReader(new FileReader(file));

		String line;
		while ((line = reader.readLine()) != null) {
			if (line.contains(search)) {
				reader.close();
				return true;
			}
		}

		reader.close();
		return false;
	}

	private static boolean isMain(final String input) {
		try {
			return grep(new File(input), "public static void main");
		} catch (IOException e) {
			e.printStackTrace();
		}

		return false;
	}

	private static void convertFile(final String input, final String output, final String sources,
			final String popcInfos) {
		ArrayList<String> parameters = new ArrayList<>();

		parameters.add("java");
		parameters.add("-cp");
		parameters.add(COMPILER_JAR);
		parameters.add(COMPILER_MAIN);

		if (isMain(input)) {
			parameters.add("-m");
		}

		parameters.add("-file=" + input);
		if (popcInfos != null && !popcInfos.isEmpty()) {
			parameters.add("-popcinfos=" + popcInfos);
		}
		parameters.add("-parclasses=" + sources);
		BufferedWriter writer;
		try {
			writer = new BufferedWriter(new FileWriter(new File(output)));
			if (ScriptUtils.runNativeApplication(ScriptUtils.listToArray(parameters),
					"Could not find java executable in path", writer, verbose) != 0) {
				System.err.println("Error while compiling " + input);
				System.exit(1);
			}
		} catch (IOException e) {
			e.printStackTrace();
		}

	}

	private static List<String> compileFiles(List<String> javaFiles, String classPathElements) {
		ArrayList<String> outputFiles = new ArrayList<>();
		// javac -cp ".:$ADD_CLASSPATH:$POPJAVA_JAR$CLASSPATHADD" $COMPILE_FILES
		// $STDFILES

		System.out.println("Compile java files: " + javaFiles.size() + " files");

		String[] command = new String[3 + javaFiles.size()];
		command[0] = JAVAC;
		command[1] = "-cp";
		command[2] = POPJAVA_JAR + File.pathSeparatorChar + classPathElements;

		int index = 3;

		for (String file : javaFiles) {
			command[index++] = file;

			outputFiles.add(file.replace(".java", ".class"));
		}

		ScriptUtils.runNativeApplication(command,
				"javac is not installed, please install the java jdk (not jre) or add it to the PATH", null, verbose);

		return outputFiles;
	}

	private static void jarFiles(final String jar, List<String> classFiles) {
		String[] command = new String[3 + classFiles.size()];
		command[0] = "jar";
		command[1] = "cf";
		command[2] = jar;

		int index = 3;
		for (String file : classFiles) {
			command[index++] = file;
		}

		ScriptUtils.runNativeApplication(command, "The jar command could not be found", null, verbose);

		for (String file : classFiles) {
			new File(file).delete();
		}
	}
}
