package ch.icosys.popjava.core.scripts;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

import ch.icosys.popjava.core.scripts.shell.CommandHandler;
import ch.icosys.popjava.core.scripts.shell.CommandInfo;
import ch.icosys.popjava.core.scripts.shell.ConsoleHandler;
import ch.icosys.popjava.core.scripts.shell.ICommand;
import ch.icosys.popjava.core.scripts.shell.command.CDebug;
import ch.icosys.popjava.core.scripts.shell.command.CJobManager;
import ch.icosys.popjava.core.scripts.shell.command.CKeystore;

/**
 * This is a simple shell an administrator can run to configure POP-Java and the
 * POPJavaJobManager.
 * 
 * @author Davide Mazzoleni
 */
public class POPJShell {

	public static final ShellConfiguration configuration = new ShellConfiguration();

	private final ConsoleHandler consoleHandler;

	private final CommandHandler commandHandler;

	public POPJShell() {
		consoleHandler = ConsoleHandler.getInstance();
		commandHandler = new CommandHandler();
		initCommands();
	}

	private void start() {
		System.out.println("Welcome to POP-Java simple configuration shell.");
		System.out.println("This shell is not interactive, you must type every command.");
		System.out.println("Use `help` to know the available commands.");
		System.out.println("Every command has a --help (-h) flag which print its options.");
		while (true) {
			try {
				System.out.print("$ ");
				CommandInfo ci = consoleHandler.readCommand();
				if (ci != null) {
					commandHandler.execute(ci);
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	private void initCommands() {
		commandHandler.add(new CKeystore());
		commandHandler.add(new CJobManager());
		commandHandler.add(new Help());
		commandHandler.add(new CDebug());
	}

	public static void optionNotFound(String keyword, String help) {
		System.out.format("%s: command not found\n", keyword);
		System.out.println(help);
	}

	public static void main(String[] args) {
		POPJShell shell = new POPJShell();
		shell.start();
	}

	private class Help implements ICommand {

		public Help() {
		}

		@Override
		public String keyword() {
			return "help";
		}

		@Override
		public int execute(CommandInfo info) {
			System.out.println(commandHandler.help());
			return 0;
		}

		@Override
		public String help() {
			return "";
		}

		@Override
		public String description() {
			return "print this help";
		}
	}

	public static class ShellConfiguration {

		private enum S {
			PRIVATE_NETWORK
		}

		private static final Path LOCATION = Paths.get("pshell.properties").toAbsolutePath();

		private final Properties PROPS = new Properties();

		public ShellConfiguration() {
			try (FileInputStream in = new FileInputStream(LOCATION.toFile())) {
				PROPS.load(in);
			} catch (IOException e) {
				try {
					Set<PosixFilePermission> perms = new HashSet<>();
					perms.add(PosixFilePermission.OWNER_READ);
					perms.add(PosixFilePermission.OWNER_WRITE);

					Files.createFile(LOCATION);
					Files.setPosixFilePermissions(LOCATION, perms);
				} catch (IOException ex) {
					System.err.println("Couldn't save shell configuration, it may not work in the future.");
				}
			}
		}

		public String getPrivateNetwork() {
			return PROPS.getProperty(S.PRIVATE_NETWORK.name());
		}

		public void setPrivateNetwork(String networkUUID) {
			saveValue(S.PRIVATE_NETWORK, networkUUID);
		}

		private void saveValue(S key, String val) {
			PROPS.setProperty(key.name(), val);
			try (FileOutputStream out = new FileOutputStream(LOCATION.toFile())) {
				PROPS.store(out, "Generated by POP Shell");
			} catch (IOException e) {
				System.err.println("Couldn't save POP Shell configuration.");
			}
		}
	}
}
