/*-
 * Plantuml builder
 *
 * Copyright (C) 2017 Focus IT
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package ch.ifocusit.plantuml.classdiagram;

import ch.ifocusit.plantuml.PlantUmlBuilder;
import ch.ifocusit.plantuml.classdiagram.model.Association;
import ch.ifocusit.plantuml.classdiagram.model.ClassMember;
import ch.ifocusit.plantuml.classdiagram.model.Package;
import ch.ifocusit.plantuml.classdiagram.model.attribute.ClassAttribute;
import ch.ifocusit.plantuml.classdiagram.model.attribute.MethodAttribute;
import ch.ifocusit.plantuml.classdiagram.model.clazz.Clazz;
import ch.ifocusit.plantuml.classdiagram.model.clazz.JavaClazz;
import ch.ifocusit.plantuml.classdiagram.model.method.ClassMethod;
import ch.ifocusit.plantuml.utils.ClassUtils;
import ch.ifocusit.plantuml.utils.PlantUmlUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.reflect.ClassPath;

import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Stream;

import static ch.ifocusit.plantuml.classdiagram.model.Association.AssociationType.*;
import static ch.ifocusit.plantuml.classdiagram.model.Cardinality.MANY;
import static ch.ifocusit.plantuml.classdiagram.model.Cardinality.NONE;
import static ch.ifocusit.plantuml.utils.ClassUtils.DOLLAR;
import static org.apache.commons.lang3.ClassUtils.getAllInterfaces;
import static org.apache.commons.lang3.StringUtils.EMPTY;

/**
 * Build class diagram from Class definition.
 *
 * @author Julien Boz
 */
public class ClassDiagramBuilder implements NamesMapper, LinkMaker {

    private final Set<java.lang.Package> packages = new LinkedHashSet<>();
    private final Set<Class> classesRepository = new LinkedHashSet<>();
    private Predicate<ClassAttribute> additionalFieldPredicate = a -> true; // always true by default

    private static final List<String> DEFAULT_METHODS_EXCLUDED = Lists.newArrayList("equals", "hashCode", "toString");

    // by default java Object methods and getter/setter will be ignored
    private Predicate<ClassMethod> additionalMethodPredicate = m -> !DEFAULT_METHODS_EXCLUDED.contains(m.getName()) && ClassUtils.isNotGetterSetter(m.getMethod());

    private final PlantUmlBuilder builder = new PlantUmlBuilder();

    private final Set<JavaClazz> clazzes = new TreeSet<>();
    private final Set<ClassAssociation> detectedAssociations = new HashSet<>();

    private NamesMapper namesMapper = this;

    private LinkMaker linkMaker = this;

    private String header;
    private String footer;

    private final Map<Class, JavaClazz> cache = new HashMap<>();

    /**
     * Add not specified Object.
     */
    private boolean withDependencies = false;

    private boolean hideSelfLink = true;

    public ClassDiagramBuilder() {
    }

    public ClassDiagramBuilder setHeader(String header) {
        this.header = header;
        return this;
    }

    public ClassDiagramBuilder setFooter(String footer) {
        this.footer = footer;
        return this;
    }

    public ClassDiagramBuilder excludes(String... excludes) {
        // keep the corresponding fields
        Predicate<ClassAttribute> notMatchField = field -> Stream.of(excludes).noneMatch(excl -> field.toStringAttribute().matches(excl));
        this.additionalFieldPredicate = this.additionalFieldPredicate.and(notMatchField);

        // keep the corresponding fields
        Predicate<ClassMethod> notMatchMethod = field -> Stream.of(excludes).noneMatch(excl -> field.toStringMethod().matches(excl));
        this.additionalMethodPredicate = this.additionalMethodPredicate.and(notMatchMethod);
        return this;
    }

    public ClassDiagramBuilder addFieldPredicate(Predicate<ClassAttribute> predicate) {
        this.additionalFieldPredicate = this.additionalFieldPredicate.and(predicate);
        return this;
    }

    public ClassDiagramBuilder addMethodPredicate(Predicate<ClassMethod> predicate) {
        this.additionalMethodPredicate = this.additionalMethodPredicate.and(predicate);
        return this;
    }

    public ClassDiagramBuilder addClasse(Iterable<Class> classes) {
        classes.forEach(this.classesRepository::add);
        return this;
    }

    public ClassDiagramBuilder addClasse(Class... classes) {
        this.classesRepository.addAll(Arrays.asList(classes));
        return this;
    }

    public ClassDiagramBuilder addPackage(java.lang.Package... packages) {
        this.packages.addAll(Arrays.asList(packages));
        return this;
    }

    public ClassDiagramBuilder withNamesMapper(NamesMapper namesMapper) {
        this.namesMapper = namesMapper;
        return this;
    }

    public ClassDiagramBuilder withLinkMaker(LinkMaker linkMaker) {
        this.linkMaker = linkMaker;
        return this;
    }

    public String build() {
        // parse classes repository
        // extract java classes definitions
        readClasses();
        // from java classes, detect associations
        detectAssociations();
        // generate diagram from configuration
        builder.start();
        builder.appendPart(header);
        addPackages(); // add package definition
        addTypes(); // add types definition
        addAssociations(); // then add their associations
        builder.appendPart(footer);
        builder.end();
        return builder.build();
    }

    protected void addPackages() {
        packages.stream().forEach(pkg -> {
            try {
                ClassPath classPath = ClassPath.from(Thread.currentThread().getContextClassLoader());
                Clazz[] classes = classPath.getTopLevelClasses(pkg.getName()).stream()
                        .map(ClassPath.ClassInfo::load)
                        .map(this::createJavaClass)
                        .sorted()
                        .toArray(Clazz[]::new);
                builder.addPackage(Package.from(pkg), classes);
            } catch (IOException e) {
                throw new IllegalStateException("Cannot load classesRepository from package " + pkg, e);
            }

        });
    }

    protected boolean canAppearsInDiagram(Class aClass) {
        return !"void".equals(aClass.getName()) && !aClass.getName().startsWith("java.") && (withDependencies || classesRepository.contains(aClass));
    }

    protected void detectAssociations() {
        // browse each defined classesRepository
        clazzes.forEach(javaClazz -> {
            // add inheritance associations
            Stream.concat(Stream.of(javaClazz.getRelatedClass().getSuperclass()), getAllInterfaces(javaClazz.getRelatedClass()).stream())
                    .filter(Objects::nonNull)
                    // exclude class if not in repository
                    .filter(this::canAppearsInDiagram)
                    // create an association between current class and it's parent
                    .forEach(hierarchicalClass -> {
                        ClassAssociation assoc = new ClassAssociation();
                        assoc.classB = javaClazz.getRelatedClass();
                        assoc.classA = hierarchicalClass;
                        assoc.setbName(namesMapper.getClassName(assoc.classB));
                        assoc.setbCardinality(NONE);
                        assoc.setaName(namesMapper.getClassName(assoc.classA));
                        assoc.setaCardinality(NONE);
                        assoc.setLabel(EMPTY);
                        assoc.setType(INHERITANCE);

                        detectedAssociations.add(assoc);
                    });

            // no field association if fields are hidden
            if (!hideFields(javaClazz)) {
                javaClazz.getAttributes().stream()
                        .filter(attribute -> !attribute.getField().isEnumConstant())
                        .forEach(classAttribute -> {
                            classAttribute.getConcernedTypes().stream()
                                    .filter(this::canAppearsInDiagram)
                                    .forEach(classToLinkWith -> {
                                        addOrUpdateAssociation(javaClazz.getRelatedClass(), classToLinkWith, classAttribute);
                                    });
                        });
            }

            // no method association if methods are hidden
            if (!hideMethods(javaClazz)) {
                javaClazz.getMethods().forEach(classMethod -> {
                    classMethod.getParameters().ifPresent(methodAttributes -> {
                        Stream.of(methodAttributes)
                                .forEach(methodAttribute -> {
                                    methodAttribute.getConcernedTypes().stream()
                                            .filter(this::canAppearsInDiagram)
                                            .forEach(classToLinkWith -> {
                                                addOrUpdateAssociation(javaClazz.getRelatedClass(), classToLinkWith, methodAttribute);
                                            });
                                });
                    });
                    classMethod.getConcernedReturnedTypes().stream()
                            .filter(this::canAppearsInDiagram)
                            .forEach(classToLinkWith -> {
                                addOrUpdateAssociation(javaClazz.getRelatedClass(), classToLinkWith, classMethod);
                            });
                });
            }
        });
    }

    private boolean hideFields(JavaClazz javaClazz) {
        return PlantUmlUtils.hideFields(javaClazz, header) || PlantUmlUtils.hideFields(javaClazz, footer);
    }

    private boolean hideMethods(JavaClazz javaClazz) {
        return PlantUmlUtils.hideMethods(javaClazz, header) || PlantUmlUtils.hideMethods(javaClazz, footer);
    }

    private void addOrUpdateAssociation(Class originClass, Class classToLinkWith, ClassMember classMember) {

        // hide inner link
        if (hideSelfLink && originClass.equals(classToLinkWith)) {
            return; // do not add this link
        }

        // look for an existing association
        Optional<ClassAssociation> existing = detectedAssociations.stream()
                .filter(assoc -> assoc.concern(originClass, classToLinkWith))
                .findFirst();

        Class typeWithGeneric = classMember.getType();

        String label = "use";
        if (classMember instanceof MethodAttribute) {
            label += classMember.getName().startsWith("arg") ? EMPTY : " as " + classMember.getName();
        } else if (classMember instanceof ClassAttribute) {
            label = ((ClassAttribute) classMember).getName();
        }

        if (existing.isPresent()) {
            if (existing.get().isNoSameOrigin(originClass))
                // do not add the second attribut to the association collection
                // mark attribute as bidirectional
                existing.get().setBidirectional();
            if (classMember instanceof ClassAttribute) {
                // update cardinality
                existing.get().setaCardinality(ClassUtils.isCollection(typeWithGeneric) ? MANY : NONE);
                // change name
                existing.get().setLabel(existing.get().getLabel() + "/" + label);
            }
        } else {
            // add association with this class
            ClassAssociation assoc = new ClassAssociation();
            assoc.classA = originClass;
            assoc.setaName(namesMapper.getClassName(assoc.classA));
            assoc.setaCardinality(ClassUtils.isCollection(assoc.classA) ? MANY : NONE);
            assoc.classB = classToLinkWith;
            assoc.setbName(namesMapper.getClassName(assoc.classB));
            assoc.setbCardinality(ClassUtils.isCollection(typeWithGeneric) ? MANY : NONE);
            assoc.setLabel(label);
            assoc.setType(DIRECTION);
            detectedAssociations.add(assoc);
        }
    }

    protected void readClasses() {
        // add all classesRepository definition
        // readFields will manage field type definition, exclusions, ...
        classesRepository.forEach(clazz -> clazzes.add(createJavaClass(clazz)));
    }

    protected void addTypes() {
        clazzes.forEach(builder::addType);
    }

    protected JavaClazz createJavaClass(Class aClass) {
        return cache.computeIfAbsent(aClass, clazz -> JavaClazz.from(clazz, readFields(clazz), readMethods(clazz))
                .setOverridedName(namesMapper.getClassName(clazz))
                .setLink(linkMaker.getClassLink(clazz))
        );
    }

    protected Predicate<ClassAttribute> filterFields() {
        return additionalFieldPredicate;
    }

    protected Predicate<ClassMethod> filterMethods() {
        return additionalMethodPredicate;
    }

    protected ClassMethod[] readMethods(Class aClass) {
        return Stream.of(aClass.getDeclaredMethods())
                // only public and non static methods
                .filter(method -> !Modifier.isStatic(method.getModifiers()) && Modifier.isPublic(method.getModifiers()))
                .map(this::createClassMethod)
                // excludes specific fields
                .filter(filterMethods())
                .sorted()
                .toArray(ClassMethod[]::new);
    }

    protected ClassMethod createClassMethod(java.lang.reflect.Method method) {
        ClassMethod classMethod = new ClassMethod(method, namesMapper.getMethodName(method));
        classMethod.setLink(linkMaker.getMethodLink(method));

        return classMethod;
    }

    protected ClassAttribute[] readFields(Class aClass) {
        return Stream.of(aClass.getDeclaredFields())
                // exclude inner class
                .filter(field -> !field.getName().startsWith(DOLLAR))
                // exclude static fields
                .filter(field -> field.getDeclaringClass().isEnum() || !Modifier.isStatic(field.getModifiers()))
                .map(this::createClassAttribute)
                // excludes specific fields
                .filter(filterFields())
                .toArray(ClassAttribute[]::new);
    }

    protected ClassAttribute createClassAttribute(Field field) {
        ClassAttribute attribute = new ClassAttribute(field, namesMapper.getFieldName(field));
        attribute.setLink(linkMaker.getFieldLink(field));
        return attribute;
    }

    protected void addAssociations() {
        detectedAssociations.stream().sorted().forEach(builder::addAssociation);
    }

    public ClassDiagramBuilder withDependencies(boolean flag) {
        withDependencies = flag;
        return this;
    }

    public ClassDiagramBuilder hideSelfLink() {
        hideSelfLink = true;
        return this;
    }

    public ClassDiagramBuilder showSelfLink() {
        hideSelfLink = false;
        return this;
    }

    public ClassDiagramBuilder withDependencies() {
        return withDependencies(true);
    }

    private static class ClassAssociation extends Association implements Comparable<ClassAssociation> {
        private Class classA;
        private Class classB;

        public void setBidirectional() {
            type = BI_DIRECTION;
        }

        public boolean concern(Class otherA, Class otherB) {
            return Sets.intersection(Sets.newHashSet(classA, classB), Sets.newHashSet(otherA, otherB)).size() == 2;
        }

        @Override
        public int compareTo(final ClassAssociation o) {
            return getKey().compareTo(o.getKey());
        }

        private String getKey() {
            return aName + bName;
        }

        public boolean isNoSameOrigin(final Class initialClass) {
            return !classA.equals(initialClass);
        }
    }
}
