package ch.iterial.keycloak.plugins.directus;

import org.keycloak.Config.Scope;

import java.util.Locale;
import java.util.StringJoiner;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiFunction;
import java.util.function.Function;

public record DirectusConnectionConfig(String url,
                                       String token,
                                       String realm,
                                       String role,
                                       String provider,
                                       long ttl,
                                       SyncConfig sync,
                                       boolean valid) {
    @Override
    public String toString() {
        return new StringJoiner(", ", DirectusConnectionConfig.class.getSimpleName() + "[", "]")
                .add("url='" + url + "'")
                .add("token='" + (token == null ? "null" : "********") + "'")
                .add("realm='" + realm + "'")
                .add("role='" + role + "'")
                .add("provider='" + provider + "'")
                .add("ttl=" + ttl)
                .add("sync=" + sync)
                .toString();
    }

    public record SyncConfig(boolean create, boolean update, boolean delete) {
        @Override
        public String toString() {
            return new StringJoiner(", ", SyncConfig.class.getSimpleName() + "[", "]")
                    .add("create=" + create)
                    .add("update=" + update)
                    .add("delete=" + delete)
                    .toString();
        }
    }

    public static DirectusConnectionConfig fromScope(final Scope scope) {
        final AtomicBoolean validity = new AtomicBoolean(true);
        final String url = PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.URL);
        final String token = PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.TOKEN);
        final String realm = PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.REALM);
        final String role = PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.ROLE);
        final String provider = PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.PROVIDER);
        final Long ttl = PropertyReaderFactory.forLong(scope, validity).getValue(PropertyNames.TTL);
        final Boolean syncCreate = PropertyReaderFactory.forBoolean(scope, validity).getValue(PropertyNames.SYNC_CREATE);
        final Boolean syncUpdate = PropertyReaderFactory.forBoolean(scope, validity).getValue(PropertyNames.SYNC_UPDATE);
        final Boolean syncDelete = PropertyReaderFactory.forBoolean(scope, validity).getValue(PropertyNames.SYNC_DELETE);

        return new DirectusConnectionConfig(
                PropertyUtils.trimToNull(url),
                PropertyUtils.trimToNull(token),
                PropertyUtils.trimToNull(realm),
                PropertyUtils.trimToNull(role),
                PropertyUtils.trimToNull(provider),
                PropertyUtils.toFinite(ttl),
                new SyncConfig(
                        PropertyUtils.isTrue(syncCreate),
                        PropertyUtils.isTrue(syncUpdate),
                        PropertyUtils.isTrue(syncDelete)
                ),
                validity.get()
        );
    }

    public enum PropertyNames {
        URL(true),
        TOKEN(true),
        REALM(false),
        ROLE(false),
        PROVIDER(false),
        TTL(false),
        SYNC_CREATE(false),
        SYNC_UPDATE(false),
        SYNC_DELETE(false);

        private final boolean required;

        PropertyNames(final boolean required) {
            this.required = required;
        }

        public boolean isRequired() {
            return required;
        }

        public String getKey() {
            return name().toLowerCase(Locale.ENGLISH).replace("_", "-");
        }

        private String getEnvVariableName() {
            return "KC_DIRECTUS_" + name();
        }
    }

    private record PropertyReader<T>(BiFunction<String, T, T> reader,
                                     Function<String, T> converter,
                                     AtomicBoolean validity) {
        public T getValue(final PropertyNames property) {
            final T value = reader.apply(property.getKey(), converter.apply(getSystemPropertiesValue(property)));
            final boolean invalid = value == null && property.isRequired();

            validity.set(validity.get() && !invalid);

            return value;
        }

        private static String getSystemPropertiesValue(final PropertyNames propertyName) {
            return PropertyUtils.trimToNull(System.getenv(propertyName.getEnvVariableName()));
        }
    }

    private static abstract class PropertyReaderFactory {
        private PropertyReaderFactory() {
            super();
        }

        static PropertyReader<String> forString(final Scope scope, final AtomicBoolean validity) {
            return new PropertyReader<>(scope::get, PropertyUtils::trimToNull, validity);
        }

        static PropertyReader<Long> forLong(final Scope scope, final AtomicBoolean validity) {
            return new PropertyReader<>(scope::getLong, value -> value == null ? null : Long.valueOf(value), validity);
        }

        static PropertyReader<Boolean> forBoolean(final Scope scope, final AtomicBoolean validity) {
            return new PropertyReader<>(scope::getBoolean, value -> Boolean.TRUE.toString().equalsIgnoreCase(PropertyUtils.trimToNull(value)), validity);
        }
    }

    private static abstract class PropertyUtils {
        private PropertyUtils() {
            super();
        }

        static long toFinite(final Long value) {
            return value == null ? -1L : value;
        }

        static boolean isTrue(final Boolean value) {
            return value != null && value;
        }

        static String trimToNull(final String string) {
            if (string == null) {
                return null;
            }
            final String trimmed = string.trim();
            if (trimmed.isEmpty()) {
                return null;
            }
            return trimmed;
        }
    }

}
